////////////////////////////////////////////////////////////////////////////////
/// @brief logger stream
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2007-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "LoggerStream.h"

#include <Basics/Logger.h>

using namespace triagens::basics;

// -----------------------------------------------------------------------------
// helper
// -----------------------------------------------------------------------------

namespace {
  void computeInfo (LoggerData::Info& info) {
    if (info.severity == LoggerData::SEVERITY_UNKNOWN) {
      switch (info.category) {
        case LoggerData::CATEGORY_FATAL: info.severity = LoggerData::SEVERITY_EXCEPTION; break;
        case LoggerData::CATEGORY_ERROR: info.severity = LoggerData::SEVERITY_EXCEPTION; break;
        case LoggerData::CATEGORY_WARNING: info.severity = LoggerData::SEVERITY_EXCEPTION; break;

        case LoggerData::CATEGORY_REQUEST_IN_START: info.severity = LoggerData::SEVERITY_TECHNICAL; break;
        case LoggerData::CATEGORY_REQUEST_IN_END: info.severity = LoggerData::SEVERITY_TECHNICAL; break;
        case LoggerData::CATEGORY_REQUEST_OUT_START: info.severity = LoggerData::SEVERITY_TECHNICAL; break;
        case LoggerData::CATEGORY_REQUEST_OUT_END: info.severity = LoggerData::SEVERITY_TECHNICAL; break;
        case LoggerData::CATEGORY_HEARTBEAT: info.severity = LoggerData::SEVERITY_TECHNICAL; break;

        case LoggerData::CATEGORY_MODULE_IN_START: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_MODULE_IN_END: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_FUNCTION_IN_START: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_FUNCTION_IN_END: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_STEP: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_LOOP: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;
        case LoggerData::CATEGORY_HEARTPULSE: info.severity = LoggerData::SEVERITY_DEVELOPMENT; break;

        default: info.severity = LoggerData::SEVERITY_UNKNOWN;
      }
    }
  }
}

namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // LoggerStream
    // -----------------------------------------------------------------------------

    LoggerStream::LoggerStream ()
      : stream(new stringstream()) {
    }



    LoggerStream::LoggerStream (LoggerData::Info const& info)
      : stream(new stringstream()), info(info) {
    }



    LoggerStream::LoggerStream (LoggerStream const& copy)
      : stream(new stringstream()), info(copy.info) {
      stream->str(copy.stream->str());
    }



    LoggerStream::~LoggerStream () {
      computeInfo(info);

      if (stream != 0) {
        Logger::output(static_cast<stringstream*>(stream)->str(), info);

        delete stream;
      }
    }



    LoggerStream& LoggerStream::operator<< (ostream& (*fptr)(ostream&)) {
      if (stream != 0) {
        *stream << fptr;
      }

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::level_e level) {
      info.level = level;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::category_e category) {
      info.category = category;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::severity_e severity) {
      info.severity = severity;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Functional const& functional) {
      info.functional = functional;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::MessageIdentifier const& messageIdentifier) {
      info.messageIdentifier = messageIdentifier;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Peg const& peg) {
      info.peg = peg;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Task const& task) {
      info.task = task;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Position const& position) {
      info.position = position;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Measure const& measure) {
      info.measure = measure;

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::Extra const& extra) {
      if (extra.position == LoggerData::Extra::npos) {
        info.extras.push_back(extra);

        size_t pos = info.extras.size() - 1;
        info.extras[pos].position = pos;
      }
      else {
        if (info.extras.size() <= extra.position) {
          info.extras.resize(extra.position + 1);
        }

        info.extras[extra.position] = extra;
      }

      return *this;
    }



    LoggerStream& LoggerStream::operator<< (LoggerData::UserIdentifier const& userIdentifier) {
      info.userIdentifier = userIdentifier;

      return *this;
    }
  }
}
