////////////////////////////////////////////////////////////////////////////////
/// @brief logger info
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2007-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_LOGGER_INFO_H
#define TRIAGENS_BASICS_LOGGER_INFO_H 1

#include <Basics/Common.h>

#include <Basics/Thread.h>

namespace triagens {
  namespace basics {
    namespace LoggerData {

      // -----------------------------------------------------------------------------
      // ApplicationName (global)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the application name
      ////////////////////////////////////////////////////////////////////////////////

      struct ApplicationName {

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the application name
        ////////////////////////////////////////////////////////////////////////////////

        string name;
      };

      // -----------------------------------------------------------------------------
      // Facility (global)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the facility name
      ////////////////////////////////////////////////////////////////////////////////

      struct Facility {

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the facility name
        ////////////////////////////////////////////////////////////////////////////////

        string name;
      };

      // -----------------------------------------------------------------------------
      // HostName (global)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the hostname
      ////////////////////////////////////////////////////////////////////////////////

      struct HostName {

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the hostname name
        ////////////////////////////////////////////////////////////////////////////////

        string name;
      };

      // -----------------------------------------------------------------------------
      // MessageIdentifier (automatic)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the message identifier
      ////////////////////////////////////////////////////////////////////////////////

      struct MessageIdentifier {

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief the mesesage identifier
        ////////////////////////////////////////////////////////////////////////////////

        string name;
      };

      // -----------------------------------------------------------------------------
      // ProcessIdentifier (automatic)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the proc and thread identifiers
      ////////////////////////////////////////////////////////////////////////////////

      struct ProcessIdentifier {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a proc identifier
          ////////////////////////////////////////////////////////////////////////////////

          ProcessIdentifier ();

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the process identifier
          ////////////////////////////////////////////////////////////////////////////////

          Thread::pid_t process;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the pthread identifier
          ////////////////////////////////////////////////////////////////////////////////

          Thread::tpid_t threadProcess;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the thread identifier
          ////////////////////////////////////////////////////////////////////////////////

          Thread::tid_t thread;
      };

      // -----------------------------------------------------------------------------
      // Category (log level)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the level of the log message
      ////////////////////////////////////////////////////////////////////////////////

      enum level_e {
        LEVEL_FATAL = 0,       // EXCEPTION
        LEVEL_ERROR = 1,       // EXCEPTION
        LEVEL_WARNING = 2,     // EXCEPTION
        LEVEL_INFO = 3,        // DEVELOPMENT, TECHNICAL, FUNCTIONAL
        LEVEL_DEBUG = 4,       // DEVELOPMENT, TECHNICAL
        LEVEL_TRACE = 5        // DEVELOPMENT, TECHNICAL
      };

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the category of the log message
      ////////////////////////////////////////////////////////////////////////////////

      enum category_e {

        // exceptions
        CATEGORY_FATAL,
        CATEGORY_ERROR,
        CATEGORY_WARNING,

        // technical
        CATEGORY_HEARTBEAT,
        CATEGORY_REQUEST_IN_END,
        CATEGORY_REQUEST_IN_START,
        CATEGORY_REQUEST_OUT_END,
        CATEGORY_REQUEST_OUT_START,

        // development
        CATEGORY_FUNCTION_IN_END,
        CATEGORY_FUNCTION_IN_START,
        CATEGORY_HEARTPULSE,
        CATEGORY_LOOP,
        CATEGORY_MODULE_IN_END,
        CATEGORY_MODULE_IN_START,
        CATEGORY_STEP
      };

      // -----------------------------------------------------------------------------
      // Severity (log level)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the type of the log message
      ////////////////////////////////////////////////////////////////////////////////

      enum severity_e {
        SEVERITY_EXCEPTION,
        SEVERITY_TECHNICAL,
        SEVERITY_FUNCTIONAL,
        SEVERITY_DEVELOPMENT,
        SEVERITY_HUMAN,
        SEVERITY_UNKNOWN
      };

      // -----------------------------------------------------------------------------
      // Functional (log level)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the fucntional name
      ////////////////////////////////////////////////////////////////////////////////

      struct Functional {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a fucntional name
          ///
          /// @param[in]
          ///    name         the function name
          ////////////////////////////////////////////////////////////////////////////////

          Functional (string const& name = "");

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the fucntional name
          ////////////////////////////////////////////////////////////////////////////////

          string name;
      };

      // -----------------------------------------------------------------------------
      // Peg (hierarchy)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the peg name
      ////////////////////////////////////////////////////////////////////////////////

      struct Peg {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a peg name
          ///
          /// @param[in]
          ///    name         the peg name
          ////////////////////////////////////////////////////////////////////////////////

          Peg (string const& name = "");

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the peg name
          ////////////////////////////////////////////////////////////////////////////////

          string name;
      };

      // -----------------------------------------------------------------------------
      // Task (hierarchy)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the task name
      ////////////////////////////////////////////////////////////////////////////////

      struct Task {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a task name
          ///
          /// @param[in]
          ///    name         the task name
          ////////////////////////////////////////////////////////////////////////////////

          Task (string const& name = "");

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the task name
          ////////////////////////////////////////////////////////////////////////////////

          string name;
      };

      // -----------------------------------------------------------------------------
      // Position (hierarchy, automatic)
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief position information
      ////////////////////////////////////////////////////////////////////////////////

      struct Position {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a new position
          ///
          /// @param[in]
          ///    function             the function name
          ///
          /// @param[in]
          ///    file                 the file name
          ///
          /// @param[in]
          ///    line                 the file position
          ////////////////////////////////////////////////////////////////////////////////

          Position (string const& function = "", string const& file = "", int line = 0);

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief function name
          ////////////////////////////////////////////////////////////////////////////////

          string function;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief file name
          ////////////////////////////////////////////////////////////////////////////////

          string file;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief line number
          ////////////////////////////////////////////////////////////////////////////////

          int line;
      };

      // -----------------------------------------------------------------------------
      // Measure
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the measure unit
      ////////////////////////////////////////////////////////////////////////////////

      enum unit_e {
        UNIT_SECONDS,
        UNIT_MILLI_SECONDS,
        UNIT_MICRO_SECONDS,
        UNIT_NANO_SECONDS,

        UNIT_BYTE,
        UNIT_KILO_BYTE,
        UNIT_MEGA_BYTE,
        UNIT_GIGA_BYTE,

        UNIT_LESS
      };

      ////////////////////////////////////////////////////////////////////////////////
      /// @brief the measure
      ////////////////////////////////////////////////////////////////////////////////

      struct Measure {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a measure value
          ///
          /// @param[in]
          ///    measure              the measure value
          ///
          /// @param[in]
          ///    unit                 the measure unit
          ////////////////////////////////////////////////////////////////////////////////

          Measure (double measure = 0.0, unit_e unit = UNIT_LESS);

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the measure value
          ////////////////////////////////////////////////////////////////////////////////

          double value;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the measure unit
          ////////////////////////////////////////////////////////////////////////////////

          unit_e unit;
      };

      // -----------------------------------------------------------------------------
      // Extra
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the extra attribute
      ////////////////////////////////////////////////////////////////////////////////

      struct Extra {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a extra attribute
          ///
          /// @param[in]
          ///    pos                  the position of the extra attribute
          ///
          /// @param[in]
          ///    name                 the extra attribute
          ////////////////////////////////////////////////////////////////////////////////

          Extra (size_t pos, string const& name);

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a extra attribute
          ///
          /// @param[in]
          ///    name                 the extra attribute
          ////////////////////////////////////////////////////////////////////////////////

          Extra (string const& name = "");

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief empty position
          ////////////////////////////////////////////////////////////////////////////////

          static size_t const npos;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the number of the extra attribute
          ////////////////////////////////////////////////////////////////////////////////

          size_t position;

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the extra attribute
          ////////////////////////////////////////////////////////////////////////////////

          string name;
      };

      // -----------------------------------------------------------------------------
      // UserIdentifier
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the user identifier
      ////////////////////////////////////////////////////////////////////////////////

      struct UserIdentifier {
        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief constructs a user identifier
          ///
          /// @param[in]
          ///    name                 the user identifier
          ////////////////////////////////////////////////////////////////////////////////

          UserIdentifier (string const& name = "");

        public:

          ////////////////////////////////////////////////////////////////////////////////
          /// @brief the user identifier
          ////////////////////////////////////////////////////////////////////////////////

          string user;
      };

      // -----------------------------------------------------------------------------
      // Info block
      // -----------------------------------------------------------------------------

      ////////////////////////////////////////////////////////////////////////////////
      /// @ingroup Logging
      /// @brief the info block
      ////////////////////////////////////////////////////////////////////////////////

      struct Info {
        Info();

        static ApplicationName applicationName;
        static Facility facility;
        static HostName hostName;

        MessageIdentifier messageIdentifier;

        level_e level;
        category_e category;
        severity_e severity;

        Functional functional;

        Peg peg;
        Task task;
        Position position;

        Measure measure;
        vector<Extra> extras;

        UserIdentifier userIdentifier;
        ProcessIdentifier processIdentifier;

        string prefix;
      };
    }

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Logging
    /// @brief logger info
    ////////////////////////////////////////////////////////////////////////////////

    class LoggerInfo {
      friend class Logger;

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches a prefix
        ///
        /// @param[in]
        ///    value               the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (string const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches a peg
        ///
        /// @param[in]
        ///    value                the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (LoggerData::Peg const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches a task
        ///
        /// @param[in]
        ///    value                the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (LoggerData::Task const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches an extra
        ///
        /// @param[in]
        ///    value                the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (LoggerData::Extra const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches an user identifier
        ///
        /// @param[in]
        ///    value                the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (LoggerData::UserIdentifier const& value);

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief catches a position
        ///
        /// @param[in]
        ///    value                the output value
        ///
        /// @return
        ///    the current LoggerInfo
        ////////////////////////////////////////////////////////////////////////////////

        LoggerInfo& operator<< (LoggerData::Position const& value);

      protected:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief context
        ////////////////////////////////////////////////////////////////////////////////

        LoggerData::Info info;
    };
  }
}


#endif

