////////////////////////////////////////////////////////////////////////////////
/// @brief logger
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Achim Brandt
/// @author Copyright 2007-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "Logger.h"

#include CONFIG_H

#include <sys/types.h>

#ifdef ENABLE_SYSLOG
#define SYSLOG_NAMES
#include <syslog.h>
#endif

#include <fstream>

#include <Basics/Crc32Hash.h>
#include <Basics/Mutex.h>
#include <Basics/MutexLocker.h>
#include <Basics/StringBuffer.h>
#include <Basics/StringUtils.h>
#include <Basics/Thread.h>

using namespace triagens::basics;
using namespace std;

// -----------------------------------------------------------------------------
// helper functions and constants
// -----------------------------------------------------------------------------

namespace {

  // -----------------------------------------------------------------------------
  // static configuration variables
  // -----------------------------------------------------------------------------

#ifdef ENABLE_SYSLOG

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief use syslog to send messages to remote daemon
  ////////////////////////////////////////////////////////////////////////////////

  bool UseSyslog = false;

#endif

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief use output stream to output messages
  ////////////////////////////////////////////////////////////////////////////////

  ostream* OutputStream = &cerr;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief true, if we own the output stream
  ////////////////////////////////////////////////////////////////////////////////

  bool OwnOutputStream = false;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief true, if show the thread identifier in human logging
  ////////////////////////////////////////////////////////////////////////////////

  bool showThreadIdentifier = false;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief prefix to use in human log messages
  ////////////////////////////////////////////////////////////////////////////////

  string OutputPrefix;

  // -----------------------------------------------------------------------------
  // assign log file or syslog
  // -----------------------------------------------------------------------------

  void AssignLogFile (string const& file) {

    // remove old stream
    if (OutputStream != 0 && OwnOutputStream) {
      delete OutputStream;
    }

    OutputStream = 0;

    // no logging at all
    if (file == "") {
      OutputStream = 0;
      OwnOutputStream = true;
    }

    // standard out
    else if (file == "-") {
      OutputStream = &cout;
      OwnOutputStream = false;
    }

    // standard error
    else if (file == "+") {
      OutputStream = &cerr;
      OwnOutputStream = false;
    }

    // file
    else {
      OutputStream = new ofstream(file.c_str(), ios::app);

      if (! *OutputStream) {
        OutputStream = &cerr;
        OwnOutputStream = false;

        LOGGER_ERROR << "cannot open log file '" << file << "'";
      }
      else {
        OwnOutputStream = true;
      }
    }
  }



#ifdef ENABLE_SYSLOG

  void AssignSyslogLogger (string const& name, string const& facility) {
    if (UseSyslog) {
      ::closelog();
    }

    if (facility.empty()) {
      UseSyslog = false;
    }
    else {
      int value = LOG_LOCAL0;

      if ('0' <= facility[0] && facility[0] <= '9') {
        value = ::atoi(facility.c_str());
      }
      else {
        CODE * ptr = facilitynames;

        while (ptr->c_name != 0) {
          if (strcmp(ptr->c_name, facility.c_str()) == 0) {
            value = ptr->c_val;
            break;
          }

          ++ptr;
        }
      }

      ::openlog(name.c_str(), LOG_CONS | LOG_PID, value);
      UseSyslog = true;
    }
  }

#endif

  // -----------------------------------------------------------------------------
  // send message to file or syslog
  // -----------------------------------------------------------------------------

  Mutex OutputLock;

  void OutputLine (string const& msg, LoggerData::severity_e severity) {
    MUTEX_LOCKER(guard, OutputLock);

    if (OutputStream != 0) {
      *OutputStream << msg << "\n" << flush;
    }

#ifdef ENABLE_SYSLOG

    // compute the priority
    int priority;

    switch (severity) {
      case LoggerData::SEVERITY_EXCEPTION: priority = LOG_CRIT;  break;
      case LoggerData::SEVERITY_FUNCTIONAL: priority = LOG_NOTICE;  break;
      case LoggerData::SEVERITY_TECHNICAL: priority = LOG_INFO;  break;
      case LoggerData::SEVERITY_DEVELOPMENT: priority = LOG_DEBUG;  break;
      default: priority = LOG_DEBUG;  break;
    }

    if (UseSyslog) {
      syslog(priority, "%s", msg.c_str());
    }

#endif
  }

  // -----------------------------------------------------------------------------
  // output buffer of log lines
  // -----------------------------------------------------------------------------

  Mutex BufferLock;

  size_t const OUTPUT_MAX_LENGTH = 256;
  size_t const OUTPUT_BUFFER_SIZE = 1024;
  size_t const OUTPUT_LOG_LEVELS = 6;
  size_t BufferCurrent[OUTPUT_LOG_LEVELS] = { 0, 0, 0, 0, 0, 0 };

  uint64_t LID = 1;

  Logger::buffer_t OutputBuffer[OUTPUT_LOG_LEVELS][OUTPUT_BUFFER_SIZE];

  void StoreOutput (LoggerData::level_e level, time_t timestamp, string text) {
    MUTEX_LOCKER(guard, BufferLock);

    size_t pos = size_t(level);

    if (pos >= OUTPUT_LOG_LEVELS) {
      return;
    }

    size_t cur = BufferCurrent[pos] = (BufferCurrent[pos] + 1) % OUTPUT_BUFFER_SIZE;

    OutputBuffer[pos][cur].lid = LID++;
    OutputBuffer[pos][cur].level = level;
    OutputBuffer[pos][cur].timestamp = timestamp;

    if (text.size() > OUTPUT_MAX_LENGTH) {
      OutputBuffer[pos][cur].text = text.substr(0, OUTPUT_MAX_LENGTH - 4) + " ...";
    }
    else {
      OutputBuffer[pos][cur].text = text;
    }
  }



  bool SidCompare (Logger::buffer_t const& left, Logger::buffer_t const& right) {
    return left.lid < right.lid;
  }

  // -----------------------------------------------------------------------------
  // output human readable log entries
  // -----------------------------------------------------------------------------

  void OutputHuman (string const& text, LoggerData::Info const& info) {
    string line;

    // generate time prefix
    time_t tt = time(0);

#ifdef HAVE_GMTIME_R
    struct tm tb;
    struct tm* t = gmtime_r(&tt, &tb);
#else
    struct tm* t = gmtime(&tt);
#endif

    char s[32];
    strftime(s,  sizeof(s) - 1, "%Y-%m-%dT%H:%M:%SZ", t);

    // use the thread identifier
    string process = StringUtils::itoa((uint64_t) info.processIdentifier.process);

    if (showThreadIdentifier) {
      process += "-" + StringUtils::itoa((uint64_t) info.processIdentifier.thread);
    }

    line += s;
    line += " ";

    if (! OutputPrefix.empty()) {
      line += OutputPrefix + " ";
    }

    line += "[" + process + "] ";

    string pos = "(" + info.position.file + ":" + StringUtils::itoa(info.position.line) + ") ";

    switch (info.level) {
      case LoggerData::LEVEL_FATAL: line += "FATAL ";  break;
      case LoggerData::LEVEL_ERROR: line += "ERROR ";  break;
      case LoggerData::LEVEL_WARNING: line += "WARNING ";  break;
      case LoggerData::LEVEL_INFO: line += "INFO ";  break;
      case LoggerData::LEVEL_DEBUG: line += "DEBUG " + pos;  break;
      case LoggerData::LEVEL_TRACE: line += "TRACE " + pos;  break;
      default: line += "UNKNOWN ";  break;
    }

    // store message
    StoreOutput(info.level, tt, text);

    // split at newlinews
    if (text.find_first_of("\r\n") == string::npos) {
      OutputLine(line + StringUtils::escapeUnicode(text, false), info.severity);
    }
    else {
      vector<string> lines = StringUtils::split(text, '\n');

      for (vector<string>::iterator i = lines.begin();  i != lines.end();  ++i) {
        string const& part = *i;

        OutputLine(line + StringUtils::escapeUnicode(part, false), info.severity);
      }
    }
  }

  // -----------------------------------------------------------------------------
  // output machine readable log entries
  // -----------------------------------------------------------------------------

  string LoggerFormat = "%Z;1;%S;%C;%H;%p-%t;%F;%A;%f;%m;%K;%f:%l;%x;%P;%u;%V;%U;%E";
  string SpecialCharacters = ";%\r\t\n";



  void OutputMachine (string const& text, LoggerData::Info const& info) {
    char const* format = LoggerFormat.c_str();
    char const* end = format + LoggerFormat.size();
    time_t tt = time(0);

    StringBuffer line;
    line.initialise();

    for (;  format < end;  ++format) {
      if (*format == '%') {
        ++format;

        switch (*format) {

          // -----------------------------------------------------------------------------
          // end-of-line
          // -----------------------------------------------------------------------------

          case '\0':
            --format;
            break;

          // -----------------------------------------------------------------------------
          // application name
          // -----------------------------------------------------------------------------

          case 'A': {
            line.appendText(info.applicationName.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // category
          // -----------------------------------------------------------------------------

          case 'C': {
            if (info.severity == LoggerData::SEVERITY_FUNCTIONAL && ! info.functional.name.empty()) {
              line.appendText(info.functional.name);
            }
            else {
              switch (info.category) {
                case LoggerData::CATEGORY_FATAL: line.appendText("FATAL"); break;
                case LoggerData::CATEGORY_ERROR: line.appendText("ERROR"); break;
                case LoggerData::CATEGORY_WARNING: line.appendText("WARNING"); break;

                case LoggerData::CATEGORY_REQUEST_IN_START: line.appendText("REQUEST-IN-START"); break;
                case LoggerData::CATEGORY_REQUEST_IN_END: line.appendText("REQUEST-IN-END"); break;
                case LoggerData::CATEGORY_REQUEST_OUT_START: line.appendText("REQUEST-OUT-START"); break;
                case LoggerData::CATEGORY_REQUEST_OUT_END: line.appendText("REQUEST-OUT-END"); break;
                case LoggerData::CATEGORY_HEARTBEAT: line.appendText("HEARTBEAT"); break;

                case LoggerData::CATEGORY_MODULE_IN_START: line.appendText("REQUEST-MODULE-IN-START"); break;
                case LoggerData::CATEGORY_MODULE_IN_END: line.appendText("REQUEST-MODULE-IN-END"); break;
                case LoggerData::CATEGORY_FUNCTION_IN_START: line.appendText("FUNCTION-IN-START"); break;
                case LoggerData::CATEGORY_FUNCTION_IN_END: line.appendText("FUNCTION-IN-END"); break;
                case LoggerData::CATEGORY_STEP: line.appendText("STEP"); break;
                case LoggerData::CATEGORY_LOOP: line.appendText("LOOP"); break;
                case LoggerData::CATEGORY_HEARTPULSE: line.appendText("HEARTPULSE"); break;
              }
            }

            break;
          }

          // -----------------------------------------------------------------------------
          // extras
          // -----------------------------------------------------------------------------

          case 'E': {
            for (vector<LoggerData::Extra>::const_iterator i = info.extras.begin();
                 i != info.extras.end();
                 ++i) {
              if (i != info.extras.begin()) {
                line.appendChar(';');
              }

              LoggerData::Extra const& extra = *i;

              line.appendText(StringUtils::escapeHex(extra.name, SpecialCharacters));
            }

            break;
          }

          // -----------------------------------------------------------------------------
          // facility
          // -----------------------------------------------------------------------------

          case 'F': {
            line.appendText(info.facility.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // module name
          // -----------------------------------------------------------------------------

          case 'f': {
            line.appendText(info.position.file);
            break;
          }

          // -----------------------------------------------------------------------------
          // host name
          // -----------------------------------------------------------------------------

          case 'H': {
            line.appendText(info.hostName.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // task
          // -----------------------------------------------------------------------------

          case 'K': {
            line.appendText(info.task.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // method name
          // -----------------------------------------------------------------------------

          case 'l': {
            line.appendInteger(info.position.line);
            break;
          }

          // -----------------------------------------------------------------------------
          // message identifier
          // -----------------------------------------------------------------------------

          case 'M': {
            line.appendText(info.messageIdentifier.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // method name
          // -----------------------------------------------------------------------------

          case 'm': {
            line.appendText(info.position.function);
            break;
          }

          // -----------------------------------------------------------------------------
          // process identifier
          // -----------------------------------------------------------------------------

          case 'p': {
            line.appendInteger((uint64_t) info.processIdentifier.process);
            break;
          }

          // -----------------------------------------------------------------------------
          // peg
          // -----------------------------------------------------------------------------

          case 'P': {
            line.appendText(info.peg.name);
            break;
          }

          // -----------------------------------------------------------------------------
          // severity
          // -----------------------------------------------------------------------------

          case 'S': {
            switch (info.severity) {
              case LoggerData::SEVERITY_EXCEPTION: line.appendInteger(2);  break;
              case LoggerData::SEVERITY_FUNCTIONAL: line.appendInteger(5);  break;
              case LoggerData::SEVERITY_TECHNICAL: line.appendInteger(6);  break;
              case LoggerData::SEVERITY_DEVELOPMENT: line.appendInteger(7);  break;
              default: line.appendInteger(7);  break;
            }

            break;
          }

          // -----------------------------------------------------------------------------
          // pthread identifier
          // -----------------------------------------------------------------------------

          case 's': {
            line.appendInteger((uint64_t) info.processIdentifier.threadProcess);
            break;
          }

          // -----------------------------------------------------------------------------
          // timestamp
          // -----------------------------------------------------------------------------

          case 'T': {
#ifdef HAVE_GMTIME_R
            struct tm tb;
            struct tm* t = localtime_r(&tt, &tb);
#else
            struct tm* t = localtime(&tt);
#endif

            char s[32];
            strftime(s,  sizeof(s) - 1, "%Y-%m-%dT%H:%M:%S", t);

            line.appendText(s);

            break;
          }

          // -----------------------------------------------------------------------------
          // thread identifier
          // -----------------------------------------------------------------------------

          case 't': {
            line.appendInteger((uint64_t) info.processIdentifier.thread);
            break;
          }

          // -----------------------------------------------------------------------------
          // measure unit
          // -----------------------------------------------------------------------------

          case 'U': {
            switch (info.measure.unit) {
              case LoggerData::UNIT_SECONDS:  line.appendText("s");  break;
              case LoggerData::UNIT_MILLI_SECONDS:  line.appendText("ms");  break;
              case LoggerData::UNIT_MICRO_SECONDS:  line.appendText("us");  break;
              case LoggerData::UNIT_NANO_SECONDS:  line.appendText("ns");  break;

              case LoggerData::UNIT_BYTE:  line.appendText("b");  break;
              case LoggerData::UNIT_KILO_BYTE:  line.appendText("kb");  break;
              case LoggerData::UNIT_MEGA_BYTE:  line.appendText("mb");  break;
              case LoggerData::UNIT_GIGA_BYTE:  line.appendText("gb");  break;

              case LoggerData::UNIT_LESS:  break;
            }

            break;
          }

          // -----------------------------------------------------------------------------
          // user identifier
          // -----------------------------------------------------------------------------

          case 'u': {
            line.appendText(info.userIdentifier.user);
            break;
          }

          // -----------------------------------------------------------------------------
          // measure value
          // -----------------------------------------------------------------------------

          case 'V': {
            line.appendDecimal(info.measure.value);
            break;
          }

          // -----------------------------------------------------------------------------
          // text
          // -----------------------------------------------------------------------------

          case 'x': {
            if (! info.prefix.empty()) {
              line.appendText(StringUtils::escapeHex(info.prefix, SpecialCharacters));
            }

            line.appendText(StringUtils::escapeHex(text, SpecialCharacters));

            break;
          }

          // -----------------------------------------------------------------------------
          // timestamp in zulu
          // -----------------------------------------------------------------------------

          case 'Z': {
#ifdef HAVE_GMTIME_R
            struct tm tb;
            struct tm* t = gmtime_r(&tt, &tb);
#else
            struct tm* t = gmtime(&tt);
#endif

            char s[32];
            strftime(s,  sizeof(s) - 1, "%Y-%m-%dT%H:%M:%SZ", t);

            line.appendText(s);

            break;
          }
        }
      }
      else {
        line.appendChar(*format);
      }
    }

    line.appendChar('\0');

    OutputLine(line.c_str(), info.severity);
    StoreOutput(LoggerData::LEVEL_DEBUG, tt, text);

    line.free();
  }
}


namespace triagens {
  namespace basics {

    // -----------------------------------------------------------------------------
    // buffer_t
    // -----------------------------------------------------------------------------

    vector<Logger::buffer_t> Logger::buffer (LoggerData::level_e level, uint64_t start) {
      vector<buffer_t> result;

      size_t pos = size_t(level);

      if (pos >= OUTPUT_LOG_LEVELS) {
        pos = OUTPUT_LOG_LEVELS - 1;
      }

      {
        MUTEX_LOCKER(guard, BufferLock);

        // merge the various log levels in one vector and sort according to LID
        for (size_t i = 0;  i <= pos;  ++i) {
          for (size_t j = 0;  j < OUTPUT_BUFFER_SIZE;  ++j) {
            size_t cur = (BufferCurrent[i] + j) % OUTPUT_BUFFER_SIZE;

            if (OutputBuffer[i][cur].lid >= start) {
              result.push_back(OutputBuffer[i][cur]);
            }
          }
        }
      }

      sort(result.begin(), result.end(), SidCompare);

      return result;
    }

    // -----------------------------------------------------------------------------
    // static variables
    // -----------------------------------------------------------------------------

    Logger Logger::singleton;

    bool Logger::exception = true;
    bool Logger::technical = false;
    bool Logger::functional = false;
    bool Logger::development = false;
    bool Logger::human = true;

    bool Logger::fatal = true;
    bool Logger::error = true;
    bool Logger::warning = false;
    bool Logger::info = false;
    bool Logger::debug = false;
    bool Logger::trace = false;

    // -----------------------------------------------------------------------------
    // constructors and destructors
    // -----------------------------------------------------------------------------

    Logger::Logger () {
    }



    Logger::~Logger () {
      if (OutputStream != 0 && OwnOutputStream) {
        delete OutputStream;
      }

      OutputStream = 0;

#ifdef ENABLE_SYSLOG

      if (UseSyslog) {
        ::closelog();
      }

#endif
    }

    // -----------------------------------------------------------------------------
    // static public methods
    // -----------------------------------------------------------------------------

    void Logger::initialise () {
    }



    void Logger::setApplicationName (string const& name) {
      LoggerData::Info::applicationName.name = name;
    }



    void Logger::setFacility (string const& name) {
      LoggerData::Info::facility.name = name;
    }



    void Logger::setHostName (string const& name) {
      LoggerData::Info::hostName.name = name;
    }



    void Logger::setLogFormat (string const& format) {
      LoggerFormat = format;
    }



    void Logger::setLogLevel (string const& l) {
      string level = StringUtils::tolower(l);

      fatal = true;
      error = false;
      warning = false;
      info = false;
      trace = false;
      debug = false;

      if (level == "fatal") {
      }
      else if (level == "error") {
        error = true;
      }
      else if (level == "warning") {
        error = true;
        warning = true;
      }
      else if (level == "info") {
        error = true;
        warning = true;
        info = true;
      }
      else if (level == "debug") {
        error = true;
        warning = true;
        info = true;
        debug = true;
      }
      else if (level == "trace") {
        error = true;
        warning = true;
        info = true;
        debug = true;
        trace = true;
      }
      else {
        error = true;
        warning = true;

        LOGGER_ERROR << "strange log level '" << level << "', going to 'warning'";
      }
    }



    string Logger::logLevel () {
      if (trace) {
        return "trace";
      }

      if (debug) {
        return "debug";
      }

      if (info) {
        return "info";
      }

      if (warning) {
        return "warning";
      }

      if (error) {
        return "error";
      }

      return "fatal";
    }



    void Logger::setLogSeverity (string const& severities) {
      vector<string> split = StringUtils::split(severities);

      exception = false;
      technical = false;
      functional = false;
      development = false;
      human = false;

      for (vector<string>::const_iterator i = split.begin();  i != split.end();  ++i) {
        string const& type = StringUtils::tolower(*i);

        if (type == "exception") {
          exception = true;
        }
        else if (type == "technical") {
          technical = true;
        }
        else if (type == "functional") {
          functional = true;
        }
        else if (type == "development") {
          development = true;
        }
        else if (type == "human") {
          human = true;
        }
        else if (type == "all") {
          exception = true;
          technical = true;
          functional = true;
          development = true;
          human = true;
        }
        else if (type == "non-human") {
          exception = true;
          technical = true;
          functional = true;
          development = true;
        }
      }
    }



    bool Logger::isStandardStream () {
      return OwnOutputStream;
    }



    void Logger::definePrefix (string const& prefix) {
      OutputPrefix = prefix;
    }



    void Logger::useThreadIdentifier (bool show) {
      showThreadIdentifier = show;
    }



    void Logger::assignLogFile (string const& file) {
      AssignLogFile(file);
    }



#ifdef ENABLE_SYSLOG

    void Logger::assignSyslog (string const& name, string const& facility) {
      AssignSyslogLogger(name, facility);
    }

#endif


    void Logger::output (string const& text, LoggerData::Info const& info) {
      if (info.severity == LoggerData::SEVERITY_HUMAN) {
        if (! Logger::isHuman()) {
          return;
        }
      }

      // human readable
      if (info.severity == LoggerData::SEVERITY_HUMAN) {
        OutputHuman(text, info);
      }

      // machine readable logging
      else {
        OutputMachine(text, info);
      }
    }
  }
}
