////////////////////////////////////////////////////////////////////////////////
/// @brief implementation of crc32
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Oreste Costa-Panaia
/// @author Wikipedia
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#include "Crc32Hash.h"

#include <Basics/Mutex.h>
#include <Basics/MutexLocker.h>

using namespace triagens::basics;

// -----------------------------------------------------------------------------
// global crc polynomial
// -----------------------------------------------------------------------------

namespace {
  Mutex Crc32Lock;

  uint32_t Crc32Polynomial[256];

  volatile sig_atomic_t Crc32PolynomialInitialised = 0;
}

// -----------------------------------------------------------------------------
// helper functions
// -----------------------------------------------------------------------------

namespace {
  uint32_t reflect (uint32_t value, const int size) {
    uint32_t reflected = 0;

    // swap bit 0 for bit 7, bit 1 For bit 6, etc....
    for (int i = 1;  i < (size + 1);  ++i)  {
      if (value & 1)  {
        reflected = reflected | (1 << (size - i));
      }

      value = value >> 1;
    }

    return reflected;
  }



  void generateCrc32Polynomial () {
    if (Crc32PolynomialInitialised != 0) {
      return;
    }

    // grab the lock and initialise
    MUTEX_LOCKER(guard, Crc32Lock);

    if (Crc32PolynomialInitialised != 0) {
      return;
    }

    // assign the CRC-32 bit polynomial
    uint32_t polynomial = 0x04C11DB7;

    // fill the array of characters 256 values representing ASCII character codes.
    for(int i = 0; i < 256; i++)  {
      Crc32Polynomial[i] = reflect(i, 8) << 24;

      for(int j = 0; j < 8; j++) {
        Crc32Polynomial[i] = (Crc32Polynomial[i] << 1)  ^ ((Crc32Polynomial[i] & (1 << 31)) ? polynomial : 0);
      }

      Crc32Polynomial[i] = reflect(Crc32Polynomial[i], 32);
    }

    Crc32PolynomialInitialised = 1;
  }
}

// -----------------------------------------------------------------------------
// implementation of crc32
// -----------------------------------------------------------------------------

namespace triagens {
  namespace basics {
    namespace Crc32Hash {

      // -----------------------------------------------------------------------------
      // public methods
      // -----------------------------------------------------------------------------

      uint32_t initialiseValue () {
        generateCrc32Polynomial();

        return (0xffffffff);
      }



      uint32_t finaliseValue (uint32_t value) {
        return (value ^ 0xffffffff);
      }



      uint32_t crcValue (uint32_t value, string const& data) {
        return crcValue(value, (const unsigned char*) data.c_str(), data.size());
      }



      uint32_t crcValue (uint32_t value, unsigned char const * data, uint64_t length) {
        if (data == 0) {
          return value;
        }
        while (length--)  {
          value = (value >> 8) ^ (Crc32Polynomial[(value & 0xFF) ^ (*data)]);
          ++data;
        }

        return value;
      }



      uint32_t crcValue (uint32_t value, unsigned char const * data) {
        if (data == 0) {
          return value;
        }
        while (*data != 0)  {
          value = (value >> 8) ^ (Crc32Polynomial[(value & 0xFF) ^ (*data)]);
          ++data;
        }

        return value;
      }
    }
  }
}
