////////////////////////////////////////////////////////////////////////////////
/// @brief High-Performance Database Framework made by triagens
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Copyright 2009-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_COMMON_H
#define TRIAGENS_BASICS_COMMON_H 1

// -----------------------------------------------------------------------------
// include the C header files that are always present on all systems
// -----------------------------------------------------------------------------

#define __STDC_LIMIT_MACROS
#include <stdint.h>

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// -----------------------------------------------------------------------------
// include the C++ header files that are always present on all systems
// -----------------------------------------------------------------------------

#include <algorithm>
#include <deque>
#include <iostream>
#include <list>
#include <map>
#include <queue>
#include <set>
#include <sstream>
#include <stack>
#include <string>
#include <vector>

// -----------------------------------------------------------------------------
// include the boost header files
// -----------------------------------------------------------------------------

#ifdef NO_BOOST

namespace boost {
  class noncopyable {
   protected:
      noncopyable () {
      }

      ~noncopyable () {
      }

   private:
      noncopyable (const noncopyable&);
      const noncopyable& operator= (const noncopyable&);
  };
}

#else

#include <boost/version.hpp>
#include <boost/utility.hpp>

#endif

// -----------------------------------------------------------------------------
// include the configuration
// -----------------------------------------------------------------------------

#include <Basics/Configuration.h>

// -----------------------------------------------------------------------------
// include the system dependent C header files
// -----------------------------------------------------------------------------

#ifdef INCLUDE_SIGNAL_H
#include <signal.h>
#endif

#ifdef INCLUDE_IO_H
#include <io.h>
#endif

#ifdef INCLUDE_SYS_TIME_H
#include <sys/time.h>
#endif

////////////////////////////////////////////////////////////////////////////////
/// @brief the general namespace of the library
////////////////////////////////////////////////////////////////////////////////

namespace triagens {
  using namespace std;
}

// -----------------------------------------------------------------------------
// Helper Functions
// -----------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
/// @brief warn if return is unused
////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_GCC_ATTRIBUTE
#define WARN_UNUSED __attribute__ ((warn_unused_result))
#else
#define WARN_UNUSED /* unused */
#endif

////////////////////////////////////////////////////////////////////////////////
/// @brief expect false
///
/// @param
///    a        the value
////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_GCC_BUILTIN
#define EF(a) __builtin_expect(a, false)
#else
#define EF(a) a
#endif

////////////////////////////////////////////////////////////////////////////////
/// @brief expect true
///
/// @param
///    a        the value
////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_GCC_BUILTIN
#define ET(a) __builtin_expect(a, true)
#else
#define ET(a) a
#endif

////////////////////////////////////////////////////////////////////////////////
/// @brief prefetch read
///
/// @param
///    a        the value to prefetch
////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_GCC_BUILTIN
#define PR(a) __builtin_prefetch(a, 0, 0)
#else
#define PR(a) a
#endif

////////////////////////////////////////////////////////////////////////////////
/// @brief prefetch write
///
/// @param
///    a        the value to prefetch
////////////////////////////////////////////////////////////////////////////////

#ifdef HAVE_GCC_BUILTIN
#define PW(a) __builtin_prefetch(a, 1, 0)
#else
#define PW(a) a
#endif

// -----------------------------------------------------------------------------
// File Descriptors
// -----------------------------------------------------------------------------

namespace triagens {

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief type of a socket
  ////////////////////////////////////////////////////////////////////////////////

  typedef int socket_t;
}

// -----------------------------------------------------------------------------
// Blobs
// -----------------------------------------------------------------------------

namespace triagens {

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief global blob type
  ///
  /// A block contains a length and data. The size of the blob is limited to
  /// 4 GByte.
  ////////////////////////////////////////////////////////////////////////////////

  struct blob_t {
    uint32_t length;
    char* data;
  };
}

// -----------------------------------------------------------------------------
// Date and Time
// -----------------------------------------------------------------------------

namespace triagens {

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief type of date-time entries (seconds since 1970-01-01)
  ////////////////////////////////////////////////////////////////////////////////

  typedef double datetime_t;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief type of date entries (days since 1970-01-01)
  ////////////////////////////////////////////////////////////////////////////////

  typedef int32_t date_t;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief type of duration entries in seconds or seconds since 00:00:00
  ////////////////////////////////////////////////////////////////////////////////

  typedef int32_t seconds_t;

  ////////////////////////////////////////////////////////////////////////////////
  /// @brief type of duration in milli seconds
  ////////////////////////////////////////////////////////////////////////////////

  typedef int64_t msec_t;
}

// -----------------------------------------------------------------------------
// Special Configuration Options
// -----------------------------------------------------------------------------

#define ENABLE_FIGURES 1
#define ENABLE_LOGGER 1
#define ENABLE_LOGGER_TIMING 1
#define ENABLE_TIMING 1

#endif
