////////////////////////////////////////////////////////////////////////////////
/// @brief associative array for POD data
///
/// @file
/// Implementation of associative arrays for POD data. The description
/// about hashing and equality test must be passed as description
/// structure. This structure must define the following methods:
///
///   clearElement(ELEMENT&)
///   deleteElement (only required if clearAndDelete is used)
///   hashElement(ELEMENT const&)
///   hashKey(KEY const&)
///   isEmptyElement(ELEMENT const&)
///   isEqualElementElement(ELEMENT const&, ELEMENT const&)
///   isEqualKeyElement(KEY const&, ELEMENT const&)
///
/// DISCLAIMER
///
/// Copyright 2010-2011 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Dr. Frank Celler
/// @author Martin Schoenert
/// @author Copyright 2006-2010, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

#ifndef TRIAGENS_BASICS_ASSOCIATIVE_ARRAY_H
#define TRIAGENS_BASICS_ASSOCIATIVE_ARRAY_H 1

#include <Basics/Common.h>

namespace triagens {
  namespace basics {

    ////////////////////////////////////////////////////////////////////////////////
    /// @ingroup Utilities
    /// @brief associative array for POD data
    ///
    /// An associative array for POD data. You must use map or hash_map if you
    /// want to store real objects. The associative array stores elements of a given
    /// type. An element must contain its key. There is no seperate buffer for
    /// keys. The description describes how to generate the hash value for keys
    /// and elements, how to compare keys and elements, and how to check for empty
    /// elements.
    ////////////////////////////////////////////////////////////////////////////////

    template <typename KEY, typename ELEMENT, typename DESC>
    class AssociativeArray : boost::noncopyable {
      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new associative array for POD data
        ////////////////////////////////////////////////////////////////////////////////

        explicit
        AssociativeArray (uint64_t size)
          : desc() {
          initialise(size);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief constructs a new associative array for POD data
        ////////////////////////////////////////////////////////////////////////////////

        AssociativeArray (uint64_t size, const DESC& desc)
          : desc(desc) {
          initialise(size);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief deletes a associative array for POD data
        ////////////////////////////////////////////////////////////////////////////////

        ~AssociativeArray () {
          delete[] table;
        }

      public:

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief swaps two dictonaries
        ////////////////////////////////////////////////////////////////////////////////

        void swap (AssociativeArray* other) {
          DESC tmpDesc = desc;
          desc = other->desc;
          other->desc = tmpDesc;

          uint64_t tmpInt = nrAlloc;
          nrAlloc = other->nrAlloc;
          other->nrAlloc = tmpInt;

          tmpInt = nrUsed;
          nrUsed = other->nrUsed;
          other->nrUsed = tmpInt;

          tmpInt = nrFinds;
          nrFinds = other->nrFinds;
          other->nrFinds = tmpInt;

          tmpInt = nrAdds;
          nrAdds = other->nrAdds;
          other->nrAdds = tmpInt;

          tmpInt = nrRems;
          nrRems = other->nrRems;
          other->nrRems = tmpInt;

          tmpInt = nrRems;
          nrRems = other->nrRems;
          other->nrRems = tmpInt;

          tmpInt = nrResizes;
          nrResizes = other->nrResizes;
          other->nrResizes = tmpInt;

          tmpInt = nrProbesF;
          nrProbesF = other->nrProbesF;
          other->nrProbesF = tmpInt;

          tmpInt = nrProbesA;
          nrProbesA = other->nrProbesA;
          other->nrProbesA = tmpInt;

          tmpInt = nrProbesD;
          nrProbesD = other->nrProbesD;
          other->nrProbesD = tmpInt;

          tmpInt = nrProbesR;
          nrProbesR = other->nrProbesR;
          other->nrProbesR = tmpInt;

          ELEMENT* tmpTable = table;
          table = other->table;
          other->table = tmpTable;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns number of elements
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t size () const {
          return nrUsed;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns the capacity
        ////////////////////////////////////////////////////////////////////////////////

        uint64_t capacity () const {
          return nrAlloc;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief returns element table
        ////////////////////////////////////////////////////////////////////////////////

        ELEMENT const * tableAndSize (size_t& size) const {
          size = nrAlloc;

          return table;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief clears the array
        ////////////////////////////////////////////////////////////////////////////////

        void clear () {
          delete[] table;
          initialise(nrAlloc);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief clears the array and deletes the elements
        ////////////////////////////////////////////////////////////////////////////////

        void clearAndDelete () {
          for (uint64_t i = 0;  i < nrAlloc;  i++) {
            desc.deleteElement(table[i]);
          }

          delete[] table;
          initialise(nrAlloc);
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finds an element with a given key
        ////////////////////////////////////////////////////////////////////////////////

        ELEMENT const& findKey (KEY const& key) const {

          // update statistics
          nrFinds++;

          // compute the hash
          uint32_t hash = desc.hashKey(key);

          // search the table
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualKeyElement(key, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesF++;
          }

          // return whatever we found
          return table[i];
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief finds a given element
        ////////////////////////////////////////////////////////////////////////////////

        ELEMENT const& findElement (ELEMENT const& element) const {

          // update statistics
          nrFinds++;

          // compute the hash
          uint32_t hash = desc.hashElement(element);

          // search the table
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualElementElement(element, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesF++;
          }

          // return whatever we found
          return table[i];
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new element
        ////////////////////////////////////////////////////////////////////////////////

        bool addElement (ELEMENT const& element, bool overwrite = true) {

          // update statistics
          nrAdds++;

          // search the table
          uint64_t hash = desc.hashElement(element);
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualElementElement(element, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesA++;
          }

          // if we found an element, return
          if (! desc.isEmptyElement(table[i])) {
            if (overwrite) {
              memcpy(&table[i], &element, sizeof(ELEMENT));
            }

            return false;
          }

          // add a new element to the associative array
          memcpy(&table[i], &element, sizeof(ELEMENT));
          nrUsed++;

          // if we were adding and the table is more than half full, extend it
          if (nrAlloc < 2 * nrUsed) {
            ELEMENT * oldTable = table;
            uint64_t  oldAlloc = nrAlloc;

            nrAlloc = 2 * nrAlloc + 1;
            nrUsed  = 0;
            nrResizes++;

            table = new ELEMENT[nrAlloc];

            for (uint64_t j = 0;  j < nrAlloc;  j++) {
              desc.clearElement(table[j]);
            }

            for (uint64_t j = 0;  j < oldAlloc;  j++) {
              if (! desc.isEmptyElement(oldTable[j])) {
                addNewElement(oldTable[j]);
              }
            }

            delete[] oldTable;
          }

          return true;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief adds a new element with key
        ////////////////////////////////////////////////////////////////////////////////

        bool addElement (KEY const& key, ELEMENT const& element, bool overwrite = true) {

          // update statistics
          nrAdds++;

          // search the table
          uint64_t hash = desc.hashKey(key);
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualKeyElement(key, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesA++;
          }

          // if we found an element, return
          if (! desc.isEmptyElement(table[i])) {
            if (overwrite) {
              memcpy(&table[i], &element, sizeof(ELEMENT));
            }

            return false;
          }

          // if we were adding and the table is more than half full, extend it
          if (nrAlloc < 2 * (nrUsed + 1)) {
            ELEMENT * oldTable = table;
            uint64_t oldAlloc = nrAlloc;

            nrAlloc = 2 * nrAlloc + 1;
            nrUsed  = 0;
            nrResizes++;

            table = new ELEMENT[nrAlloc];

            for (uint64_t i = 0;  i < nrAlloc;  i++) {
              desc.clearElement(table[i]);
            }

            for (uint64_t i = 0;  i < oldAlloc;  i++) {
              if (! desc.isEmptyElement(oldTable[i])) {
                addNewElement(oldTable[i]);
              }
            }

            delete[] oldTable;
          }

          // add a new element to the associative array
          addNewElement(key, element);

          return true;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief removes a key
        ////////////////////////////////////////////////////////////////////////////////

        ELEMENT removeKey (KEY const& key) {

          // update statistics
          nrRems++;

          // search the table
          uint64_t hash = desc.hashKey(key);
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualKeyElement(key, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesD++;
          }

          // if we did not find such an item
          if (desc.isEmptyElement(table[i])) {
            return table[i];
          }

          // return found element
          ELEMENT element = table[i];

          // remove item
          desc.clearElement(table[i]);
          nrUsed--;

          // and now check the following places for items to move here
          uint64_t k = (i + 1) % nrAlloc;

          while (! desc.isEmptyElement(table[k])) {
            uint64_t j = desc.hashElement(table[k]) % nrAlloc;

            if ((i < k && ! (i < j && j <= k))
                || (k < i && ! (i < j || j <= k))) {
              table[i] = table[k];
              desc.clearElement(table[k]);
              i = k;
            }

            k = (k + 1) % nrAlloc;
          }

          // return success
          return element;
        }

        ////////////////////////////////////////////////////////////////////////////////
        /// @brief removes an element
        ////////////////////////////////////////////////////////////////////////////////

        bool removeElement (ELEMENT const& element) {

          // update statistics
          nrRems++;

          // search the table
          uint64_t hash = desc.hashElement(element);
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i]) && ! desc.isEqualElementElement(element, table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesD++;
          }

          // if we did not find such an item return false
          if (desc.isEmptyElement(table[i])) {
            return false;
          }

          // remove item
          desc.clearElement(table[i]);
          nrUsed--;

          // and now check the following places for items to move here
          uint64_t k = (i + 1) % nrAlloc;

          while (! desc.isEmptyElement(table[k])) {
            uint64_t j = desc.hashElement(table[k]) % nrAlloc;

            if ((i < k && ! (i < j && j <= k))
                || (k < i && ! (i < j || j <= k))) {
              table[i] = table[k];
              desc.clearElement(table[k]);
              i = k;
            }

            k = (k + 1) % nrAlloc;
          }

          // return success
          return true;
        }

      // -----------------------------------------------------------------------------
      // private methods
      // -----------------------------------------------------------------------------

      private:
        void initialise (uint64_t size) {
          table = new ELEMENT [size];

          for (uint64_t i = 0;  i < size;  i++) {
            desc.clearElement(table[i]);
          }

          nrAlloc   = size;
          nrUsed    = 0;
          nrFinds   = 0;
          nrAdds    = 0;
          nrRems    = 0;
          nrResizes = 0;
          nrProbesF = 0;
          nrProbesA = 0;
          nrProbesD = 0;
          nrProbesR = 0;
        }



        void addNewElement (ELEMENT const& element) {

          // compute the hash
          uint64_t hash = desc.hashElement(element);

          // search the table
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesR++;
          }

          // add a new element to the associative array
          memcpy(&table[i], &element, sizeof(ELEMENT));
          nrUsed++;
        }



        void addNewElement (KEY const& key, ELEMENT const& element) {

          // compute the hash
          uint64_t hash = desc.hashKey(key);

          // search the table
          uint64_t i = hash % nrAlloc;

          while (! desc.isEmptyElement(table[i])) {
            i = (i + 1) % nrAlloc;
            nrProbesR++;
          }

          // add a new element to the associative array
          memcpy(&table[i], &element, sizeof(ELEMENT));
          nrUsed++;
        }

      // -----------------------------------------------------------------------------
      // private variables
      // -----------------------------------------------------------------------------

      private:
        DESC desc;

        uint64_t                 nrAlloc;    // the size of the table
        uint64_t                 nrUsed;     // the number of used entries
        ELEMENT *                table;      // the table itself

        mutable uint64_t         nrFinds;    // statistics
        mutable uint64_t         nrAdds;     // statistics
        mutable uint64_t         nrRems;     // statistics
        mutable uint64_t         nrResizes;  // statistics

        mutable uint64_t         nrProbesF;  // statistics
        mutable uint64_t         nrProbesA;  // statistics
        mutable uint64_t         nrProbesD;  // statistics
        mutable uint64_t         nrProbesR;  // statistics
    };
  }
}

#endif
