/**
 *   SFCGAL
 *
 *   Copyright (C) 2012-2013 Oslandia <infos@oslandia.com>
 *   Copyright (C) 2012-2013 IGN (http://www.ign.fr)
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *   
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.

 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, see <http://www.gnu.org/licenses/>.
 */
#include <fstream>

#include <SFCGAL/Point.h>
#include <SFCGAL/LineString.h>
#include <SFCGAL/Polygon.h>
#include <SFCGAL/Triangle.h>
#include <SFCGAL/PolyhedralSurface.h>
#include <SFCGAL/TriangulatedSurface.h>
#include <SFCGAL/Solid.h>
#include <SFCGAL/GeometryCollection.h>
#include <SFCGAL/MultiPoint.h>
#include <SFCGAL/MultiLineString.h>
#include <SFCGAL/MultiPolygon.h>
#include <SFCGAL/MultiSolid.h>
#include <SFCGAL/io/asc.h>

#include "../../../test_config.h"

#include <boost/test/unit_test.hpp>
using namespace boost::unit_test ;

using namespace SFCGAL ;

BOOST_AUTO_TEST_SUITE( SFCGAL_ASCTest )

BOOST_AUTO_TEST_CASE( testReadExample )
{
    std::string filename( SFCGAL_TEST_DIRECTORY );
    filename += "/data/example.asc" ;

    std::ifstream ifs( filename.c_str() );
    BOOST_REQUIRE( ifs.good() ) ;
    std::auto_ptr< Grid > grid( io::readASC( ifs ) ) ;
    BOOST_CHECK_EQUAL( grid->ncols(), 4U );
    BOOST_CHECK_EQUAL( grid->nrows(), 6U );

    BOOST_CHECK_EQUAL( grid->dx(), 50.0 );
    BOOST_CHECK_EQUAL( grid->dy(), 50.0 );

    Envelope limits = grid->limits();
    BOOST_CHECK_EQUAL( limits.xMin(), 0.0 );
    BOOST_CHECK_EQUAL( limits.xMax(), 200.0 );
    BOOST_CHECK_EQUAL( limits.yMin(), 0.0 );
    BOOST_CHECK_EQUAL( limits.yMax(), 300.0 );

    // first line
    BOOST_CHECK( isNaN( grid->z( 0,0 ) ) );
    BOOST_CHECK( isNaN( grid->z( 0,1 ) ) );
    BOOST_CHECK_EQUAL( grid->z( 0,2 ), 5.0 );
    BOOST_CHECK_EQUAL( grid->z( 0,3 ), 2.0 );

    // last line
    BOOST_CHECK_EQUAL( grid->z( 5,0 ), 13.0 );
    BOOST_CHECK_EQUAL( grid->z( 5,1 ), 5.0 );
    BOOST_CHECK_EQUAL( grid->z( 5,2 ), 1.0 );
    BOOST_CHECK( isNaN( grid->z( 5,3 ) ) );

    // pixel is area convention
    BOOST_CHECK_EQUAL( grid->pixelConvention(), PIXEL_IS_AREA ) ;
    Point a00 = grid->point( 0,0 );
    BOOST_CHECK_EQUAL( a00.x(), 25.0 );
    BOOST_CHECK_EQUAL( a00.y(), 275.0 );

    {
        grid->setPixelConvention( PIXEL_IS_AREA );
        std::auto_ptr< TriangulatedSurface > triangulatedSurface( grid->toTrianguledSurface() ) ;
        std::string expected( "TIN M(((75.00000 225.00000 20.00000,125.00000 225.00000 100.00000,125.00000 275.00000 5.00000,75.00000 225.00000 20.00000)),((125.00000 225.00000 100.00000,175.00000 225.00000 36.00000,175.00000 275.00000 2.00000,125.00000 225.00000 100.00000)),((125.00000 225.00000 100.00000,175.00000 275.00000 2.00000,125.00000 275.00000 5.00000,125.00000 225.00000 100.00000)),((25.00000 175.00000 3.00000,75.00000 175.00000 8.00000,75.00000 225.00000 20.00000,25.00000 175.00000 3.00000)),((75.00000 175.00000 8.00000,125.00000 175.00000 35.00000,125.00000 225.00000 100.00000,75.00000 175.00000 8.00000)),((75.00000 175.00000 8.00000,125.00000 225.00000 100.00000,75.00000 225.00000 20.00000,75.00000 175.00000 8.00000)),((125.00000 175.00000 35.00000,175.00000 175.00000 10.00000,175.00000 225.00000 36.00000,125.00000 175.00000 35.00000)),((125.00000 175.00000 35.00000,175.00000 225.00000 36.00000,125.00000 225.00000 100.00000,125.00000 175.00000 35.00000)),((25.00000 125.00000 32.00000,75.00000 125.00000 42.00000,75.00000 175.00000 8.00000,25.00000 125.00000 32.00000)),((25.00000 125.00000 32.00000,75.00000 175.00000 8.00000,25.00000 175.00000 3.00000,25.00000 125.00000 32.00000)),((75.00000 125.00000 42.00000,125.00000 125.00000 50.00000,125.00000 175.00000 35.00000,75.00000 125.00000 42.00000)),((75.00000 125.00000 42.00000,125.00000 175.00000 35.00000,75.00000 175.00000 8.00000,75.00000 125.00000 42.00000)),((125.00000 125.00000 50.00000,175.00000 125.00000 6.00000,175.00000 175.00000 10.00000,125.00000 125.00000 50.00000)),((125.00000 125.00000 50.00000,175.00000 175.00000 10.00000,125.00000 175.00000 35.00000,125.00000 125.00000 50.00000)),((25.00000 75.00000 88.00000,75.00000 75.00000 75.00000,75.00000 125.00000 42.00000,25.00000 75.00000 88.00000)),((25.00000 75.00000 88.00000,75.00000 125.00000 42.00000,25.00000 125.00000 32.00000,25.00000 75.00000 88.00000)),((75.00000 75.00000 75.00000,125.00000 75.00000 27.00000,125.00000 125.00000 50.00000,75.00000 75.00000 75.00000)),((75.00000 75.00000 75.00000,125.00000 125.00000 50.00000,75.00000 125.00000 42.00000,75.00000 75.00000 75.00000)),((125.00000 75.00000 27.00000,175.00000 75.00000 9.00000,175.00000 125.00000 6.00000,125.00000 75.00000 27.00000)),((125.00000 75.00000 27.00000,175.00000 125.00000 6.00000,125.00000 125.00000 50.00000,125.00000 75.00000 27.00000)),((25.00000 25.00000 13.00000,75.00000 25.00000 5.00000,75.00000 75.00000 75.00000,25.00000 25.00000 13.00000)),((25.00000 25.00000 13.00000,75.00000 75.00000 75.00000,25.00000 75.00000 88.00000,25.00000 25.00000 13.00000)),((75.00000 25.00000 5.00000,125.00000 25.00000 1.00000,125.00000 75.00000 27.00000,75.00000 25.00000 5.00000)),((75.00000 25.00000 5.00000,125.00000 75.00000 27.00000,75.00000 75.00000 75.00000,75.00000 25.00000 5.00000)),((125.00000 25.00000 1.00000,175.00000 75.00000 9.00000,125.00000 75.00000 27.00000,125.00000 25.00000 1.00000)))" );
        BOOST_CHECK_EQUAL( triangulatedSurface->asText( 5 ), expected );
    }
    {
        grid->setPixelConvention( PIXEL_IS_POINT );
        std::auto_ptr< TriangulatedSurface > triangulatedSurface( grid->toTrianguledSurface() ) ;
        std::string expected( "TIN M(((66.66667 240.00000 20.00000,133.33333 240.00000 100.00000,133.33333 300.00000 5.00000,66.66667 240.00000 20.00000)),((133.33333 240.00000 100.00000,200.00000 240.00000 36.00000,200.00000 300.00000 2.00000,133.33333 240.00000 100.00000)),((133.33333 240.00000 100.00000,200.00000 300.00000 2.00000,133.33333 300.00000 5.00000,133.33333 240.00000 100.00000)),((0.00000 180.00000 3.00000,66.66667 180.00000 8.00000,66.66667 240.00000 20.00000,0.00000 180.00000 3.00000)),((66.66667 180.00000 8.00000,133.33333 180.00000 35.00000,133.33333 240.00000 100.00000,66.66667 180.00000 8.00000)),((66.66667 180.00000 8.00000,133.33333 240.00000 100.00000,66.66667 240.00000 20.00000,66.66667 180.00000 8.00000)),((133.33333 180.00000 35.00000,200.00000 180.00000 10.00000,200.00000 240.00000 36.00000,133.33333 180.00000 35.00000)),((133.33333 180.00000 35.00000,200.00000 240.00000 36.00000,133.33333 240.00000 100.00000,133.33333 180.00000 35.00000)),((0.00000 120.00000 32.00000,66.66667 120.00000 42.00000,66.66667 180.00000 8.00000,0.00000 120.00000 32.00000)),((0.00000 120.00000 32.00000,66.66667 180.00000 8.00000,0.00000 180.00000 3.00000,0.00000 120.00000 32.00000)),((66.66667 120.00000 42.00000,133.33333 120.00000 50.00000,133.33333 180.00000 35.00000,66.66667 120.00000 42.00000)),((66.66667 120.00000 42.00000,133.33333 180.00000 35.00000,66.66667 180.00000 8.00000,66.66667 120.00000 42.00000)),((133.33333 120.00000 50.00000,200.00000 120.00000 6.00000,200.00000 180.00000 10.00000,133.33333 120.00000 50.00000)),((133.33333 120.00000 50.00000,200.00000 180.00000 10.00000,133.33333 180.00000 35.00000,133.33333 120.00000 50.00000)),((0.00000 60.00000 88.00000,66.66667 60.00000 75.00000,66.66667 120.00000 42.00000,0.00000 60.00000 88.00000)),((0.00000 60.00000 88.00000,66.66667 120.00000 42.00000,0.00000 120.00000 32.00000,0.00000 60.00000 88.00000)),((66.66667 60.00000 75.00000,133.33333 60.00000 27.00000,133.33333 120.00000 50.00000,66.66667 60.00000 75.00000)),((66.66667 60.00000 75.00000,133.33333 120.00000 50.00000,66.66667 120.00000 42.00000,66.66667 60.00000 75.00000)),((133.33333 60.00000 27.00000,200.00000 60.00000 9.00000,200.00000 120.00000 6.00000,133.33333 60.00000 27.00000)),((133.33333 60.00000 27.00000,200.00000 120.00000 6.00000,133.33333 120.00000 50.00000,133.33333 60.00000 27.00000)),((0.00000 0.00000 13.00000,66.66667 0.00000 5.00000,66.66667 60.00000 75.00000,0.00000 0.00000 13.00000)),((0.00000 0.00000 13.00000,66.66667 60.00000 75.00000,0.00000 60.00000 88.00000,0.00000 0.00000 13.00000)),((66.66667 0.00000 5.00000,133.33333 0.00000 1.00000,133.33333 60.00000 27.00000,66.66667 0.00000 5.00000)),((66.66667 0.00000 5.00000,133.33333 60.00000 27.00000,66.66667 60.00000 75.00000,66.66667 0.00000 5.00000)),((133.33333 0.00000 1.00000,200.00000 60.00000 9.00000,133.33333 60.00000 27.00000,133.33333 0.00000 1.00000)))" );
        BOOST_CHECK_EQUAL( triangulatedSurface->asText( 5 ), expected );
    }
}


BOOST_AUTO_TEST_SUITE_END()




