# TnmDisman.tcl --
#
#	This file implements a set of procedures to interact with
#	a remote scripting engine via the IETF Script MIB.
#
# Copyright (c) 1998-1999 Technical University of Braunschweig.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.

package require Tnm 3.0
package require TnmSnmp 3.0
package provide TnmDisman 3.0.0

namespace eval TnmDisman {
    namespace export GetLanguages GetExtensions

    namespace export InstallScript DeleteScript LoadScript UnloadScript
    namespace export SetScriptStorage SetScriptDescription SetScriptSource
    namespace export SetScriptLanguage 

    namespace export GetScriptOwners GetScripts GetScriptInfo
}

proc TnmDisman::GetLanguages {s} {

    set result {}
    lappend vbl [list smLangLanguage]
    $s walk vbl $vbl {
	set lang [Tnm::mib unpack [Tnm::snmp oid $vbl 0]]
	set langID [Tnm::snmp value $vbl 0]
	lappend result [list $lang $langID]
    }

    return $result
}

proc TnmDisman::GetExtensions {s language} {

    set result {}
    lappend vbl [list smExtsnExtension.$language]
    $s walk vbl $vbl {
	set extsn [lindex [Tnm::mib unpack [Tnm::snmp oid $vbl 0]] 1]
	set extsnID [Tnm::snmp value $vbl 0]
	lappend result [list $extsn $extsnID]
    }

    return $result
}

proc TnmDisman::InstallScript {s owner name language source} {

    lappend vbl [list [Tnm::mib pack smScriptLanguage $owner $name] $language]
    lappend vbl [list [Tnm::mib pack smScriptSource $owner $name] $source]
    lappend vbl [list [Tnm::mib pack smScriptRowStatus $owner $name] createAndGo]

    if {[catch {$s set $vbl} vbl]} {
	error "unable to create row: $vbl"
    }

    return
}

proc TnmDisman::DeleteScript {s owner name} {
    
    TnmDisman::UnloadScript $s $owner $name

    lappend vbl [list [Tnm::mib pack smScriptRowStatus $owner $name] destroy]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to destroy row: $vbl"
    }

    return
}

proc TnmDisman::LoadScript {s owner name {timeout 60}} {

    lappend vbl [list [Tnm::mib pack smScriptAdminStatus $owner $name] enabled]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to enable script: $vbl"
    }

    set transient {disabled retrieving compiling}
    set status "disabled"
    for {set n $timeout} {$n >= 0} {incr n -1} {
        if {[catch {$s get [Tnm::mib pack smScriptOperStatus $owner $name]} vbl]} {
            error "unable to obtain script status: $vbl"
        }
        set status [Tnm::snmp value $vbl 0]
        puts $status
	if {[lsearch $transient $status] < 0} break
        after 1000
    }
    if {$status != "enabled"} {
        error "unable to enable script: $vbl"
    }

    return ""
}

proc TnmDisman::UnloadScript {s owner name {timeout 60}} {

    lappend vbl [list [Tnm::mib pack smScriptAdminStatus $owner $name] disabled]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to enable script: $vbl"
    }

    set status "enabled"
    for {set n $timeout} {$n >= 0} {incr n -1} {
        if {[catch {$s get [Tnm::mib pack smScriptOperStatus $owner $name]} vbl]} {
            error "unable to obtain script status: $vbl"
        }
        set status [Tnm::snmp value $vbl 0]
        puts $status
	if {$status == "disabled"} break
        after 1000
    }
    if {$status != "disabled"} {
        error "unable to disable script: $vbl"
    }

    return
}

proc TnmDisman::SetScriptStorage {s owner name storage} {

    lappend vbl [list [Tnm::mib pack smScriptStorageType $owner $name] $storage]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to set storage: $vbl"
    }

    return
}

proc TnmDisman::SetScriptDescription {s owner name description} {

    lappend vbl [list [Tnm::mib pack smScriptDescr $owner $name] $description]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to set description: $vbl"
    }

    return
}

proc TnmDisman::SetScriptSource {s owner name source} {
    TnmDisman::UnloadScript $s $owner $name

    lappend vbl [list [Tnm::mib pack smScriptSource $owner $name] $source]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to set script source: $vbl"
    }

    return
}

proc TnmDisman::SetScriptLanguage {s owner name language} {
    TnmDisman::UnloadScript $s $owner $name

    lappend vbl [list [Tnm::mib pack smScriptLanguage $owner $name] $language]
    if {[catch {$s set $vbl} vbl]} {
        error "unable to set script source: $vbl"
    }

    return
}

proc TnmDisman::GetScriptOwners {s} {
    $s walk vbl smScriptOperStatus {
	set oid [Tnm::snmp oid $vbl 0]
	set owner [lindex [Tnm::mib unpack $oid] 0]
	set name  [lindex [Tnm::mib unpack $oid] 1]
	lappend result($owner) $name
    }
    if {[info exists result]} {
	return [array names result]
    }
    return
}

proc TnmDisman::GetScripts {s owner} {

    set oid [join [lreplace [split [Tnm::mib pack smScriptOperStatus $owner {}] .] end end] .]

    set result {}
    $s walk vbl $oid {
	lappend result [lindex [Tnm::mib unpack [Tnm::snmp oid $vbl 0]] 1]
    }

    return $result
}

