/*
 * tnmMibTcl.c --
 *
 *	The Tcl interface to the MIB parser.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 * Copyright (c) 1996-1997 University of Twente.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * Forward declarations for procedures defined later in this file:
 */

static TnmMibType*
GetMibType	_ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *objPtr));

static TnmMibNode*
GetMibNode	_ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *objPtr, 
			     TnmOid **oidPtrPtr, TnmOid *nodeOidPtr));
static int
GetMibNodeOrType _ANSI_ARGS_((Tcl_Interp *interp, Tcl_Obj *objPtr,
			      TnmMibType **typePtrPtr,
			      TnmMibNode **nodePtrPtr));
static int
WalkTree	_ANSI_ARGS_((Tcl_Interp *interp, 
			     Tcl_Obj *varName, Tcl_Obj *body,
			     TnmMibNode* nodePtr, TnmOid *oidPtr));

/*
 *----------------------------------------------------------------------
 *
 * GetMibType --
 *
 *	This procedure tries to convert the argument in objPtr into
 *	a MIB type pointer. 
 *
 * Results:
 *	This procedure returns a pointer to the MIB type or a NULL
 *	pointer if the lookup failed. An error message is left in the
 *	interpreter in case of a failed lookup.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static TnmMibType*
GetMibType(interp, objPtr)
    Tcl_Interp *interp;
    Tcl_Obj *objPtr;
{
    TnmMibType *typePtr = TnmMibFindType(Tcl_GetStringFromObj(objPtr,NULL));
    if (! typePtr) {
	Tcl_ResetResult(interp);
	Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
			       "unknown type \"", 
			       Tcl_GetStringFromObj(objPtr, NULL),
			       "\"", (char *) NULL);
	return NULL;
    }
    return typePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * GetMibNode --
 *
 *	This procedure tries to convert the argument in objPtr into
 *	a MIB node pointer. 
 *
 * Results:
 *	This procedure returns a pointer to the MIB node or a NULL
 *	pointer if the lookup failed. The object identifier of the
 *	matching node is written to nodeOidPtr, if this pointer is not
 *	NULL. An error message is left in the interpreter in case of
 *	an failed lookup.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static TnmMibNode*
GetMibNode(interp, objPtr, oidPtrPtr, nodeOidPtr)
    Tcl_Interp *interp;
    Tcl_Obj *objPtr;
    TnmOid **oidPtrPtr;
    TnmOid *nodeOidPtr;
{
    TnmMibNode *nodePtr = NULL;
    TnmOid *oidPtr;

    if (oidPtrPtr) {
	*oidPtrPtr = NULL;
    }
    oidPtr = TnmGetOidFromObj(interp, objPtr);
    if (oidPtr) { 
	nodePtr = TnmMibNodeFromOid(oidPtr, nodeOidPtr);
    }
    if (! nodePtr) {
	Tcl_ResetResult(interp);
	Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
			       "unknown node \"", 
			       Tcl_GetStringFromObj(objPtr, NULL),
			       "\"", (char *) NULL);
	return NULL;
    }
    if (oidPtrPtr) {
	*oidPtrPtr = oidPtr;
    }
    return nodePtr;
}

/*
 *----------------------------------------------------------------------
 *
 * GetMibNodeOrType --
 *
 *	This procedure tries to convert the argument in objPtr into
 *	a MIB type or a MIB node pointer.
 *
 * Results:
 *	This procedure returns TCL_OK on success. The type pointer is
 *	left in typePtrPtr and the node pointer is left in nodePtrPtr.
 *	Only one pointer will be set. The other pointer will be set
 *	to NULL. TCL_ERROR is returned if the conversion fails and an
 *	error message is left in the interpreter.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
GetMibNodeOrType(interp, objPtr, typePtrPtr, nodePtrPtr)
    Tcl_Interp *interp;
    Tcl_Obj *objPtr;
    TnmMibType **typePtrPtr;
    TnmMibNode **nodePtrPtr;
{
    *nodePtrPtr = (TnmMibNode *) NULL;
    *typePtrPtr = GetMibType(interp, objPtr);
    if (*typePtrPtr) {
	return TCL_OK;
    }
    Tcl_ResetResult(interp);
    *nodePtrPtr = GetMibNode(interp, objPtr, NULL, NULL);
    if (*nodePtrPtr) {
	return TCL_OK;
    }
    Tcl_ResetResult(interp);
    Tcl_AppendStringsToObj(Tcl_GetObjResult(interp), 
			   "unknown node or type \"", 
			   Tcl_GetStringFromObj(objPtr, NULL),
			   "\"", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmMibLoadFile --
 *
 *	This procedure reads MIB definitions from a file and adds the
 *	objects to the internal MIB tree. This function expands ~
 *	filenames and searches $tnm(library)/site and $tnm(library)/mibs
 *	for the given filename. The frozen image of MIB definitions is 
 *	written into a machine specific directory.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	New frozen MIB files may be created.
 *
 *----------------------------------------------------------------------
 */

int
TnmMibLoadFile(interp, file)
    Tcl_Interp *interp;
    char *file;
{
    Tcl_DString fileBuffer, frozenFileBuffer;
    char *library, *arch, *fileName, *frozenFileName = NULL;
    char **fileArgv = NULL, **libArgv = NULL;
    int fileArgc, libArgc;
    int code = TCL_OK;

    Tcl_DStringInit(&fileBuffer);
    Tcl_DStringInit(&frozenFileBuffer);

    /*
     * Split the file argument into a path and the file name. Also,
     * get the Tnm library path and split it up into pieces.
     */

    Tcl_SplitPath(file, &fileArgc, &fileArgv);
    library = Tcl_GetVar2(interp, "tnm", "library", TCL_GLOBAL_ONLY);
    if (library) {
	Tcl_SplitPath(library, &libArgc, &libArgv);
    }

    /* 
     * Check if we can write a frozen file. Construct the path to the
     * directory where we keep frozen files. Create a machine specific
     * sub-directory if needed. The result of the following block is
     * the native frozen file name or NULL if we can't write a frozen
     * file.
     */

    arch = Tcl_GetVar2(interp, "tnm", "arch", TCL_GLOBAL_ONLY);
    if (library != NULL && arch != NULL) {

	int tmpArgc;
	char **tmpArgv = (char **) ckalloc((libArgc + 2) * sizeof(char *));

	for (tmpArgc = 0; tmpArgc < libArgc; tmpArgc++) {
	    tmpArgv[tmpArgc] = libArgv[tmpArgc];
	}
	tmpArgv[tmpArgc++] = arch;

	frozenFileName = Tcl_JoinPath(tmpArgc, tmpArgv, &frozenFileBuffer);
	if (access(frozenFileName, F_OK) != 0) {
	    if (mkdir(frozenFileName, 0777) != 0) {
		frozenFileName = NULL;
	    }
	}

	if (frozenFileName) {
	    char *name = ckalloc(strlen(fileArgv[fileArgc-1]) + 20);
	    strcpy(name, fileArgv[fileArgc-1]);
	    strcat(name, ".idy");
	    tmpArgv[tmpArgc++] = name;
	    Tcl_DStringFree(&frozenFileBuffer);
	    frozenFileName = Tcl_JoinPath(tmpArgc, tmpArgv, &frozenFileBuffer);
	    ckfree(name);
	}
	ckfree((char *) tmpArgv);
    }

    /* 
     * Search for the MIB file we are trying to load. First try the
     * file argument translated into the platform specific format. If
     * not found, check $tnm(library)/site and $tnm(library)/mibs.
     */

    {
	Tcl_DString tmpBuffer;
	Tcl_DStringInit(&tmpBuffer);
	fileName = Tcl_JoinPath(fileArgc, fileArgv, &tmpBuffer);
	fileName = Tcl_TranslateFileName(interp, fileName, &fileBuffer);
	Tcl_DStringFree(&tmpBuffer);
    }

    if (! fileName) {
	code = TCL_ERROR;
	goto exit;
    }

    if (access(fileName, R_OK) != 0) {

	int tmpArgc;
	char **tmpArgv = (char **) ckalloc((libArgc + 2) * sizeof(char *));

	if (library) {
	    for (tmpArgc = 0; tmpArgc < libArgc; tmpArgc++) {
		tmpArgv[tmpArgc] = libArgv[tmpArgc];
	    }
	    tmpArgv[tmpArgc++] = "site";
	    tmpArgv[tmpArgc++] = fileArgv[fileArgc-1];
	    Tcl_DStringFree(&fileBuffer);
	    fileName = Tcl_JoinPath(tmpArgc, tmpArgv, &fileBuffer);
	}

	if (library && (access(fileName, R_OK) != 0)) {
	    tmpArgv[libArgc] = "mibs";
	    Tcl_DStringFree(&fileBuffer);
	    fileName = Tcl_JoinPath(tmpArgc, tmpArgv, &fileBuffer);
	}
	    
	if (library && (access(fileName, R_OK) != 0)) {
	    fileName = NULL;
	}

	ckfree((char *) tmpArgv);
    }

    /*
     * If we have the file name now, call the parser to do its job.
     */

    if (fileName) {
	tnmMibTree = TnmMibParse(fileName, frozenFileName, tnmMibTree);
	if (tnmMibTree == NULL) {
	    Tcl_AppendResult(interp, "parsing MIB file ", fileName," failed", 
			     (char *) NULL);
	    code = TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "couldn't open MIB file \"", file,
			 "\": ", Tcl_PosixError(interp), (char *) NULL);
	code = TCL_ERROR;
    }

 exit:
    /* 
     * Free up all the memory that we have allocated.
     */

    Tcl_DStringFree(&fileBuffer);
    Tcl_DStringFree(&frozenFileBuffer);
    if (fileArgv) {
	ckfree((char *) fileArgv);
    }
    if (libArgv) {
	ckfree((char *) libArgv);
    }
    return code;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmMibLoadCore --
 *
 *	This procedure reads core MIB definitions and adds the objects
 *	to the internal MIB tree. The set of core MIB definitions is
 *	taken from the global Tcl variable tnm(mibs:core).
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmMibLoadCore(interp)
    Tcl_Interp *interp;
{
    Tcl_Obj *listPtr, *part1Ptr, *part2Ptr, **objv;
    int i, objc;
    static int alreadyDone = 0;

    if (alreadyDone) {
	return TCL_OK;
    }

    part1Ptr = Tcl_NewStringObj("tnm", -1);
    part2Ptr = Tcl_NewStringObj("mibs:core", -1);
    listPtr = Tcl_ObjGetVar2(interp, part1Ptr, part2Ptr, TCL_GLOBAL_ONLY);
    if (Tcl_ListObjGetElements(interp, listPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < objc; i++) {
	char *string = Tcl_GetStringFromObj(objv[i], NULL);
	if (TnmMibLoadFile(interp, string) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    alreadyDone = 1;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TnmMibLoad --
 *
 *	This procedure reads the set of default MIB definitions and
 *	adds the objects to the internal MIB tree. The set of default
 *	MIB definitions is taken from the global Tcl variables
 *	tnm(mibs:core) and tnm(mibs).
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
TnmMibLoad(interp)
    Tcl_Interp *interp;
{
    Tcl_Obj *listPtr, *part1Ptr, *part2Ptr, **objv;
    int i, objc;
    static int alreadyDone = 0;

    if (alreadyDone) {
	return TCL_OK;
    }

    if (TnmMibLoadCore(interp) != TCL_OK) {
	return TCL_ERROR;
    }

    part1Ptr = Tcl_NewStringObj("tnm", -1);
    part2Ptr = Tcl_NewStringObj("mibs", -1);
    listPtr = Tcl_ObjGetVar2(interp, part1Ptr, part2Ptr, TCL_GLOBAL_ONLY);
    if (Tcl_ListObjGetElements(interp, listPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < objc; i++) {
	char *string = Tcl_GetStringFromObj(objv[i], NULL);
	if (TnmMibLoadFile(interp, string) != TCL_OK) {
	    return TCL_ERROR;
	}
    }

    alreadyDone = 1;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * WalkTree --
 *
 *	This procedure implements a recursive MIB walk. The varName 
 *	argument defines the Tcl variable used to identify the current
 *	MIB node and label identifies the root node of the sub-tree.
 *	The current position in the MIB tree is given by nodePtr.
 *
 *	The oidPtr argument is optional and only used when the label
 *	is an object identifier and not a name. In this case, we
 *	assemble the current path in the tree in the buffer pointed to
 *	by oidPtr.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
WalkTree(interp, varName, body, nodePtr, oidPtr)
    Tcl_Interp *interp;
    Tcl_Obj *varName;
    Tcl_Obj *body;
    TnmMibNode* nodePtr;
    TnmOid *oidPtr;
{
    int result = TCL_OK;
    int length = TnmOidGetLength(oidPtr);

    for (; nodePtr; nodePtr = nodePtr->nextPtr) {
	TnmOidSet(oidPtr, length-1, nodePtr->subid);
	if (!Tcl_ObjSetVar2(interp, varName, NULL, TnmNewOidObj(oidPtr), 
			    TCL_LEAVE_ERR_MSG | TCL_PARSE_PART1)) {
	    result = TCL_ERROR;
	    goto loopDone;
	}

	result = Tcl_EvalObj(interp, body);
        if ((result == TCL_OK || result == TCL_CONTINUE) 
	    && nodePtr->childPtr) {
	    TnmOidSetLength(oidPtr, length+1);
	    result = WalkTree(interp, varName, body, 
			      nodePtr->childPtr, oidPtr);
	    TnmOidSetLength(oidPtr, length);
	}

	if (result != TCL_OK) {
	    if (result == TCL_CONTINUE) {
		result = TCL_OK;
	    } else if (result == TCL_BREAK) {
		goto loopDone;
	    } else if (result == TCL_ERROR) {
		char msg[100];
		sprintf(msg, "\n    (\"mib walk\" body line %d)",
			interp->errorLine);
		Tcl_AddErrorInfo(interp, msg);
		goto loopDone;
	    } else {
		goto loopDone;
	    }
	}
    }

  loopDone:
    if (result == TCL_OK) {
	Tcl_ResetResult(interp);
    }
    return result;
}
#if 0

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibCmd --
 *
 *	This procedure is invoked to process the "mib" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_MibCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int	argc;
    char **argv;
{
    TnmMibNode *nodePtr;
    TnmMibType *typePtr;
    char *expanded, *result = NULL;
    static int initialized = 0;
    int cmd, offset;

    enum commands {
	cmdAccess, cmdCompare, cmdChilds, cmdDefval, cmdDescr, cmdDisplay,
	cmdEnums, cmdExists, cmdFile, cmdFormat, cmdIndex, cmdInstance,
	cmdLabel, cmdLength, cmdLoad, cmdMacro, cmdModule, cmdName, 
	cmdOid, cmdParent, cmdScan, cmdSyntax, cmdType, cmdWalk
    };

    static TnmTable cmdTable[] = {
	{ cmdAccess,	"access" },	{ cmdChilds,	"children" },
	{ cmdCompare,	"compare" },	{ cmdDefval,	"defval" },
	{ cmdDescr,	"description" },{ cmdDisplay,	"displayhint" },
	{ cmdEnums,	"enums" },	{ cmdExists,	"exists" },
	{ cmdFile,	"file" },	{ cmdFormat,	"format" },
	{ cmdIndex,	"index" },	{ cmdInstance,	"instance" },
	{ cmdLabel,	"label" }, 	{ cmdLength,	"length" },
	{ cmdLoad,	"load" },	{ cmdMacro,	"macro" },
	{ cmdModule,	"module" },	{ cmdName,	"name" },
	{ cmdOid,	"oid" },	{ cmdParent,	"parent" },
	{ cmdScan,	"scan" },	{ cmdSyntax,	"syntax" },
	{ cmdType,	"type" },	{ cmdWalk,	"walk" },
	{ 0, NULL }
    };

    if (argc < 2) {
	TnmWrongNumArgs(interp, 1, argv, "option ?arg arg ...?");
	return TCL_ERROR;
    }

    /*
     * Backward compatibility: Map the "successor" command to the
     * "children" command.
     */

    if (strcmp(argv[1], "successor") == 0) {
	argv[1] = "children";
    }

    cmd = TnmGetTableKey(cmdTable, argv[1]);
    if (cmd == -1) {
	TnmBadOption(interp, argv[1], TnmGetTableValues(cmdTable));
	return TCL_ERROR;
    }

    /*
     * Auto-load the default set of MIB definitions if not initialized
     * yet. This makes use of the global Tcl variable tnm(mibs).
     */

    if (! initialized) {
	char *mibFileList;

	initialized = 1;
	mibFileList = Tcl_GetVar2(interp, "tnm", "mibs:core", TCL_GLOBAL_ONLY);
	if (LoadFileList(interp, mibFileList) != TCL_OK) {
	    return TCL_ERROR;
	}

	if (strcmp(argv[1], "load") != 0) {
	    mibFileList = Tcl_GetVar2(interp, "tnm", "mibs", TCL_GLOBAL_ONLY);
	    if (mibFileList == NULL) {
		mibFileList = Tcl_GetVar(interp, "scotty_mibs", 
					 TCL_GLOBAL_ONLY);
	    }
	    if (LoadFileList(interp, mibFileList) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
    }

    switch ((enum commands) cmd) {
    case cmdWalk: {
	TnmOid oid;
	int code;

	if (argc != 5) {
	    TnmWrongNumArgs(interp, 2, argv, "varName node command");
	    return TCL_ERROR;
	}
	TnmOidInit(&oid);
	code = TnmOidFromString(&oid, argv[3]);
	code = WalkTree(interp, argv[2], argv[3], argv[4], NULL, 
			(code == TCL_OK) ? &oid : NULL);
	TnmOidFree(&oid);
	if (code != TCL_OK && code != TCL_BREAK) {
	    return TCL_ERROR;
	}
	break;
    }

    case cmdCompare: {
	TnmOid oid1, oid2;

	if (argc != 4) {
	    TnmWrongNumArgs(interp, 2, argv, "oid1 oid2");
	    return TCL_ERROR;
	}
	TnmOidInit(&oid1);
	TnmOidInit(&oid2);
	result = (TnmIsOid(argv[2])) ? argv[2] : TnmMibGetOid(argv[2]);
	if (! result) {
	    NotFound(interp, noOid, argv[2]);
	    return TCL_ERROR;
	}
	(void) TnmOidFromString(&oid1, result);
	result = (TnmIsOid(argv[3])) ? argv[3] : TnmMibGetOid(argv[3]);
	if (! result) {
	    TnmOidFree(&oid1);
	    NotFound(interp, noOid, argv[3]);
            return TCL_ERROR;
	}
	(void) TnmOidFromString(&oid2, result);
	sprintf(interp->result, "%d", TnmOidCompare(&oid1, &oid2));
	TnmOidFree(&oid1);
	TnmOidFree(&oid2);
	break;
    }

    case cmdScan:
	if (argc != 4) {
	    TnmWrongNumArgs(interp, 2, argv, "nodeOrType value");
	    return TCL_ERROR;
	}
	typePtr = TnmMibFindType(argv[2]);
        if (typePtr) {
            result = TnmMibScanValue(typePtr, typePtr->syntax, argv[3]);
	} else {
	    result = TnmMibScan(argv[2], 0, argv[3]);
	}
	if (! result) {
	    NotFound(interp, noNodeOrType, argv[2]);
	    return TCL_ERROR;
	}
	Tcl_SetResult(interp, result, TCL_VOLATILE);
	break;

    case cmdFormat:
	if (argc != 4) {
	    TnmWrongNumArgs(interp, 2, argv, "nodeOrType value");
            return TCL_ERROR;
	}
	typePtr = TnmMibFindType(argv[2]);
	if (typePtr) {
	    result = TnmMibFormatValue(typePtr, typePtr->syntax, argv[3]);
	} else {
	    result = TnmMibFormat(argv[2], 0, argv[3]);
	}
	if (! result) {
	    NotFound(interp, noNodeOrType, argv[2]);
	    return TCL_ERROR;
	}
        Tcl_SetResult(interp, result, TCL_VOLATILE);
	break;

    case cmdAccess:
	if (argc != 3) {
	    TnmWrongNumArgs(interp, 2, argv, "node");
	    return TCL_ERROR;
	}
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
            NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
        }
	result = TnmGetTableValue(tnmMibAccessTable, nodePtr->access);
	Tcl_SetResult(interp, result ? result : "unknown", TCL_STATIC);
	break;

    case cmdChilds:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	expanded = TnmHexToOid(argv[2]);
	if (! expanded) {
	    expanded = argv[2];
	}
	nodePtr = TnmMibFindNode(expanded, NULL, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
	}
	if (nodePtr->childPtr) {
	    TnmMibNode *nPtr;
	    Tcl_DString dst;
	    char buf[20];
	    int retoid = TnmIsOid(expanded);
	    Tcl_DStringInit(&dst);
	    for (nPtr = nodePtr->childPtr; nPtr; nPtr = nPtr->nextPtr) {
		if (retoid) {
                    Tcl_DStringAppend(&dst, expanded, -1);
                    buf[0] = '.';
		    sprintf(buf, ".%u", nPtr->subid);
                    Tcl_DStringAppend(&dst, buf, -1);
                } else {
		    if (nPtr->moduleName) {
			Tcl_DStringAppend(&dst, nPtr->moduleName, -1);
			Tcl_DStringAppend(&dst, "!", 1);
		    }
		    Tcl_DStringAppend(&dst, nPtr->label, -1);
                }
		Tcl_AppendElement(interp, Tcl_DStringValue(&dst));
		Tcl_DStringFree(&dst);
	    }
	}
        break;

    case cmdDefval:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
            NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
        }
	if (nodePtr->index 
	    && nodePtr->syntax != ASN1_SEQUENCE_OF 
	    && nodePtr->syntax != ASN1_SEQUENCE) {
	    Tcl_SetResult(interp, nodePtr->index, TCL_STATIC);
	}
        break;

    case cmdDescr:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
            NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
        }
	result = TnmMibGetString(nodePtr->fileName, nodePtr->fileOffset);
	if (result) {
	    Tcl_SetResult(interp, result, TCL_STATIC);
	}
        break;

    case cmdDisplay:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "type");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
	if (! typePtr) {
            NotFound(interp, noType, argv[2]);
            return TCL_ERROR;
        }
	if (typePtr->displayHint) {
	    Tcl_SetResult(interp, typePtr->displayHint, TCL_STATIC);
	}
	break;

    case cmdEnums:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "type");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
	if (! typePtr) {
	    NotFound(interp, noType, argv[2]);
	    return TCL_ERROR;
	}
	if (typePtr->enumList) {
            TnmMibEnum *ePtr;
	    char buf[20];
            for (ePtr = typePtr->enumList; ePtr; ePtr = ePtr->nextPtr) {
		sprintf(buf, "%d", ePtr->value);
		Tcl_AppendElement(interp, ePtr->label);
		Tcl_AppendElement(interp, buf);
	    }
	}
	break;

    case cmdExists:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "nodeOrType");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
	if (typePtr) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    nodePtr = TnmMibFindNode(argv[2], NULL, 1);
	    Tcl_SetResult(interp, nodePtr ? "1" : "0", TCL_STATIC);
	}
	break;

    case cmdFile:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "nodeOrType");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
	if (typePtr) {
	    result = typePtr->fileName;
	} else {
	    nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	    if (nodePtr) {
		result = nodePtr->fileName;
	    } else {
		NotFound(interp, noNodeOrType, argv[2]);
		return TCL_ERROR;
	    }
	}
	Tcl_SetResult(interp, result ? result : "", TCL_STATIC);
        break;

    case cmdIndex:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
	}
	if (nodePtr->syntax == ASN1_SEQUENCE_OF && nodePtr->childPtr) {
	    nodePtr = nodePtr->childPtr;
	}
	if (nodePtr->syntax == ASN1_SEQUENCE && nodePtr->index) {

	    /*
	     * This is a hack because the MIB parser is broken. We
	     * check if index value points to a node of type
	     * ASN1_SEQUENCE or ASN1_SEQUENCE_OF. This is an
	     * indication that the parser used the index field to
	     * store the augmented table entry. In this case, return
	     * the index value of the node pointed to.
	     */

	    TnmMibNode *iPtr;
	    iPtr = TnmMibFindNode(nodePtr->index, NULL, 1);
	    result = ((iPtr && iPtr->index) ? iPtr->index : nodePtr->index);
	}
        Tcl_SetResult(interp, result ? result : "", TCL_VOLATILE);
        break;

    case cmdInstance:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "oid");
            return TCL_ERROR;
        }
	expanded = TnmHexToOid(argv[2]);
	if (! expanded) {
	    expanded = argv[2];
	}
	nodePtr = TnmMibFindNode(expanded, &offset, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
	}
	if (nodePtr->macro == TNM_MIB_OBJECTTYPE && offset > 0) {
	    result = expanded + offset;
	    if (result[0] == '.') result++;
	    Tcl_SetResult(interp, result, TCL_VOLATILE);
	}
	break;

    case cmdLabel:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
	    NotFound(interp, noOid, argv[2]);
            return TCL_ERROR;
	}
	Tcl_SetResult(interp, nodePtr->label, TCL_STATIC);
        break;

    case cmdLoad:
	if (argc != 3) {
	    TnmWrongNumArgs(interp, 2, argv, "file");
	    return TCL_ERROR;
	}
	return TnmMibLoadFile(interp, argv[2]);

    case cmdMacro:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "nodeOrType");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
        if (typePtr) {
            result = TnmGetTableValue(tnmMibMacroTable, typePtr->macro);
	} else {
	    nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	    if (nodePtr) {
		result = TnmGetTableValue(tnmMibMacroTable, nodePtr->macro);
	    } else {
		NotFound(interp, noNodeOrType, argv[2]);
		return TCL_ERROR;
	    }
	}
	Tcl_SetResult(interp, result ? result : "", TCL_STATIC);
        break;

    case cmdModule:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "nodeOrType");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
        if (typePtr) {
            result = typePtr->moduleName;
	} else {
	    nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	    if (nodePtr) {
		result = nodePtr->moduleName;
	    } else {
		NotFound(interp, noNodeOrType, argv[2]);
                return TCL_ERROR;
	    }
	}
	Tcl_SetResult(interp, result ? result : "", TCL_STATIC);
        break;

    case cmdName:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	expanded = TnmHexToOid(argv[2]);
	nodePtr = TnmMibFindNode(expanded ? expanded : argv[2], &offset, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
	}
	if (nodePtr->moduleName) {
	    Tcl_AppendResult(interp, nodePtr->moduleName, "!", (char *) NULL);
	}
	Tcl_AppendResult(interp, nodePtr->label, (char *) NULL);
	if (offset > 0) {
	    Tcl_AppendResult(interp, 
			     expanded ? expanded + offset : argv[2] + offset,
			     (char *) NULL);
	}
        break;

    case cmdOid:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
        result = TnmMibGetOid(argv[2]);
        if (! result) {
            NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
        }
        Tcl_SetResult(interp, result, TCL_VOLATILE);
        break;

    case cmdParent:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	expanded = TnmHexToOid(argv[2]);
	if (! expanded) {
	    expanded = argv[2];
	}
	nodePtr = TnmMibFindNode(expanded, NULL, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
	    return TCL_ERROR;
	}
	if (nodePtr->parentPtr && nodePtr->parentPtr->label) {
	    if (TnmIsOid(argv[2])) {
		TnmOid oid;
		TnmOidInit(&oid);
		TnmMibNodeToOid(nodePtr->parentPtr, &oid);
		Tcl_SetResult(interp, TnmOidToString(&oid), TCL_STATIC);
		TnmOidFree(&oid);
	    } else {
		if (nodePtr->parentPtr->moduleName) {
		    Tcl_AppendResult(interp, nodePtr->parentPtr->moduleName,
				     "!", (char *) NULL);
		}
		Tcl_AppendResult(interp, nodePtr->parentPtr->label, 
				 (char *) NULL);
	    }
	}
        break;

    case cmdSyntax:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "nodeOrType");
            return TCL_ERROR;
        }
	typePtr = TnmMibFindType(argv[2]);
        if (typePtr) {
            result = TnmGetTableValue(tnmSnmpTypeTable, typePtr->syntax);
	} else {
	    nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	    if (! nodePtr) {
		NotFound(interp, noNodeOrType, argv[2]);
		return TCL_ERROR;
	    }
	    if (nodePtr->macro == TNM_MIB_OBJECTTYPE) {
		if (nodePtr->typePtr && nodePtr->typePtr->name) {
		    if (nodePtr->typePtr->macro == TNM_MIB_OBJECTTYPE) {
			result = TnmGetTableValue(tnmSnmpTypeTable, 
						  nodePtr->typePtr->syntax);
		    } else {
			result = nodePtr->typePtr->name;
		    }
		} else {
		    result = TnmGetTableValue(tnmSnmpTypeTable,
					      nodePtr->syntax);
		}
	    }
	}
	Tcl_SetResult(interp, result ? result : "", TCL_STATIC);
        break;

    case cmdType:
	if (argc != 3) {
            TnmWrongNumArgs(interp, 2, argv, "node");
            return TCL_ERROR;
        }
	nodePtr = TnmMibFindNode(argv[2], NULL, 0);
	if (! nodePtr) {
	    NotFound(interp, noNode, argv[2]);
            return TCL_ERROR;
	}
	if (nodePtr->typePtr) {
	    if (nodePtr->typePtr->moduleName) {
		Tcl_AppendResult(interp, nodePtr->moduleName, "!", 
				 (char *) NULL);
	    }
	    Tcl_AppendResult(interp, nodePtr->typePtr->name, (char *) NULL);
	} else {
	    result = TnmGetTableValue(tnmSnmpTypeTable, nodePtr->syntax);
	    Tcl_SetResult(interp, result, TCL_STATIC);
	}
	break;

    case cmdLength: {
	TnmOid oid;
	if (argc != 3) {
	    TnmWrongNumArgs(interp, 2, argv, "oid");
	    return TCL_ERROR;
	}
	TnmOidInit(&oid);
	result = (TnmIsOid(argv[2])) ? argv[2] : TnmMibGetOid(argv[2]);
	if (! result) {
	    NotFound(interp, noOid, argv[2]);
	    return TCL_ERROR;
	}
	(void) TnmOidFromString(&oid, result);
	sprintf(interp->result, "%d", TnmOidGetLength(&oid));
	TnmOidFree(&oid);
	break;
    }
    }

    return TCL_OK;
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * Tnm_MibObjCmd --
 *
 *	This procedure is invoked to process the "mib" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_MibObjCmd(clientData, interp, objc, objv)
    ClientData clientData;
    Tcl_Interp *interp;
    int	objc;
    Tcl_Obj *CONST objv[];
{
    TnmMibNode *nodePtr;
    TnmMibType *typePtr;
    TnmOid *oidPtr;
    Tcl_Obj *objPtr;
    char *result = NULL;
    static int initialized = 0;
    int code, isShared;

    enum commands {
	cmdAccess, cmdChild, cmdCompare, cmdDefval, cmdDescr, 
	cmdDisplay, cmdEnums, cmdExists, cmdFile, cmdFormat, cmdIndex,
	cmdInstance, cmdLabel, cmdLength, cmdLoad, cmdMacro, cmdModule,
	cmdName, cmdOid, cmdParent, cmdScan, cmdSyntax, cmdType, cmdWalk
    } cmd;

    static char *cmdTable[] = {
	"access", "children", "compare", "defval", "description", 
	"displayhint", "enums", "exists", "file", "format", "index",
	"instance", "label", "length", "load", "macro", "module",
	"name", "oid", "parent", "scan", "syntax", "type", "walk",
	(char *) NULL
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "option ?arg arg ...?");
	return TCL_ERROR;
    }

    /*
     * Backward compatibility: Map the "successor" command to the
     * "children" command.
     */

    if (strcmp(Tcl_GetStringFromObj(objv[1], NULL), "successor") == 0) {
	Tcl_SetStringObj(objv[1], "successor", -1);
    }

    code = Tcl_GetIndexFromObj(interp, objv[1], cmdTable, 
			       "option", TCL_EXACT, (int *) &cmd);
    if (code != TCL_OK) {
	return code;
    }

    /*
     * Auto-load the default set of MIB definitions, if not initialized
     * yet. This makes use of the global Tcl variable tnm(mibs).
     */

    if (! initialized) {
	if (TnmMibLoadCore(interp) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (strcmp(Tcl_GetStringFromObj(objv[1], NULL), "load") != 0) {
	    if (TnmMibLoad(interp) != TCL_OK) {
		return TCL_ERROR;
	    }
	}
	initialized = 1;
    }

    switch (cmd) {
    case cmdWalk: {
	int code;
	TnmOid nodeOid;
	if (objc != 5) {
	    Tcl_WrongNumArgs(interp, 2, objv, "varName node command");
	    return TCL_ERROR;
	}
	TnmOidInit(&nodeOid);
	nodePtr = GetMibNode(interp, objv[3], NULL, &nodeOid);
	if (! nodePtr) {
	    TnmOidFree(&nodeOid);
            return TCL_ERROR;
        }
	if (nodePtr->childPtr) {
	    nodePtr = nodePtr->childPtr;
	    TnmOidAppend(&nodeOid, nodePtr->subid);
	    code = WalkTree(interp, objv[2], objv[4], nodePtr, &nodeOid);
	}
	TnmOidFree(&nodeOid);
	if (code != TCL_OK && code != TCL_BREAK) {
	    return TCL_ERROR;
	}
	break;
    }

    case cmdCompare: {
	TnmOid *oidPtr1, *oidPtr2;
	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "oid1 oid2");
	    return TCL_ERROR;
	}
	oidPtr1 = TnmGetOidFromObj(interp, objv[2]);
	if (! oidPtr1) { 
	    return TCL_ERROR;
	}
	oidPtr2 = TnmGetOidFromObj(interp, objv[3]);
	if (! oidPtr2) { 
	    return TCL_ERROR;
	}
	Tcl_SetIntObj(Tcl_GetObjResult(interp), 
		      TnmOidCompare(oidPtr1, oidPtr2));
	break;
    }

    case cmdScan:
	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType value");
	    return TCL_ERROR;
	}
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
        if (typePtr) {
            result = TnmMibScanValue(typePtr, typePtr->syntax, 
				     Tcl_GetStringFromObj(objv[3], NULL));
	} else {
	    result = TnmMibScan(Tcl_GetStringFromObj(objv[2], NULL), 0,
				Tcl_GetStringFromObj(objv[3], NULL));
	}
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
	break;

    case cmdFormat:
	if (objc != 4) {
	    Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType value");
            return TCL_ERROR;
	}
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (typePtr) {
	    result = TnmMibFormatValue(typePtr, typePtr->syntax, 
				       Tcl_GetStringFromObj(objv[3], NULL));
	} else {
	    result = TnmMibFormat(Tcl_GetStringFromObj(objv[2], NULL), 0,
				  Tcl_GetStringFromObj(objv[3], NULL));
	}
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
	break;

    case cmdAccess:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "node");
	    return TCL_ERROR;
	}
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
        }
	result = TnmGetTableValue(tnmMibAccessTable, nodePtr->access);
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
	break;

    case cmdChild: {
	TnmOid nodeOid;
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	TnmOidInit(&nodeOid);
	nodePtr = GetMibNode(interp, objv[2], NULL, &nodeOid);
	if (! nodePtr) {
            return TCL_ERROR;
	}
	if (nodePtr->childPtr) {
	    Tcl_Obj *listPtr, *elemObjPtr;
	    int index = TnmOidGetLength(&nodeOid);
	    TnmOidAppend(&nodeOid, 0);
	    listPtr = Tcl_GetObjResult(interp);
	    for (nodePtr = nodePtr->childPtr;
		 nodePtr;
		 nodePtr = nodePtr->nextPtr) {
		TnmOidSet(&nodeOid, index, nodePtr->subid);
		elemObjPtr = TnmNewOidObj(&nodeOid);
		TnmOidObjSetRep(elemObjPtr, TnmOidObjGetRep(objv[2]));
		Tcl_InvalidateStringRep(elemObjPtr);
                Tcl_ListObjAppendElement(interp, listPtr, elemObjPtr);
	    }
	}
        break;
    }

    case cmdDefval:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
        }
	if (nodePtr->index 
	    && nodePtr->syntax != ASN1_SEQUENCE_OF 
	    && nodePtr->syntax != ASN1_SEQUENCE) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), nodePtr->index, -1);
	}
        break;

    case cmdDescr:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
        }
	result = TnmMibGetString(nodePtr->fileName, nodePtr->fileOffset);
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdDisplay:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "type");
            return TCL_ERROR;
        }
	typePtr = GetMibType(interp, objv[2]);
	if (! typePtr) {
            return TCL_ERROR;
        }
	if (typePtr->displayHint) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), 
			     typePtr->displayHint, -1);
	}
	break;

    case cmdEnums:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "type");
            return TCL_ERROR;
        }
	typePtr = GetMibType(interp, objv[2]);
	if (! typePtr) {
	    return TCL_ERROR;
	}
	if (typePtr->enumList) {
	    Tcl_Obj *listPtr, *elemObjPtr;
            TnmMibEnum *ePtr;
	    listPtr = Tcl_GetObjResult(interp);
            for (ePtr = typePtr->enumList; ePtr; ePtr = ePtr->nextPtr) {
		elemObjPtr = Tcl_NewStringObj(ePtr->label, -1);
		Tcl_ListObjAppendElement(interp, listPtr, elemObjPtr);
		elemObjPtr = Tcl_NewIntObj(ePtr->value);
		Tcl_ListObjAppendElement(interp, listPtr, elemObjPtr);
	    }
	}
	break;

    case cmdExists:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType");
            return TCL_ERROR;
        }
	code = GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr);
	Tcl_SetBooleanObj(Tcl_GetObjResult(interp), code == TCL_OK);
	break;

    case cmdFile:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType");
            return TCL_ERROR;
        }
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	result = typePtr ? typePtr->fileName : nodePtr->fileName;
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdIndex:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
	}
	if (nodePtr->syntax == ASN1_SEQUENCE_OF && nodePtr->childPtr) {
	    nodePtr = nodePtr->childPtr;
	}
	if (nodePtr->syntax == ASN1_SEQUENCE && nodePtr->index) {

	    /*
	     * This is a hack because the MIB parser is broken. We
	     * check if index value points to a node of type
	     * ASN1_SEQUENCE or ASN1_SEQUENCE_OF. This is an
	     * indication that the parser used the index field to
	     * store the augmented table entry. In this case, return
	     * the index value of the node pointed to.
	     */

	    TnmMibNode *iPtr;
	    iPtr = TnmMibFindNode(nodePtr->index, NULL, 1);
	    result = ((iPtr && iPtr->index) ? iPtr->index : nodePtr->index);
	}
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdInstance: {
	TnmOid nodeOid;
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "oid");
            return TCL_ERROR;
        }
	TnmOidInit(&nodeOid);
	nodePtr = GetMibNode(interp, objv[2], &oidPtr, &nodeOid);
	if (! nodePtr) {
	    TnmOidFree(&nodeOid);
            return TCL_ERROR;
	}
	if (nodePtr->macro == TNM_MIB_OBJECTTYPE) {
	    int i;
	    TnmOid *instPtr = (TnmOid *) Tcl_Alloc(sizeof(TnmOid));
	    TnmOidInit(instPtr);
	    for (i = TnmOidGetLength(&nodeOid); 
		 i < TnmOidGetLength(oidPtr); i++) {
		TnmOidAppend(instPtr, TnmOidGet(oidPtr, i));
	    }
	    TnmSetOidObj(Tcl_GetObjResult(interp), instPtr);
	}
	TnmOidFree(&nodeOid);
	break;
    }

    case cmdLabel:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
	}
	Tcl_SetStringObj(Tcl_GetObjResult(interp), nodePtr->label, -1);
        break;

    case cmdLoad:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "file");
	    return TCL_ERROR;
	}
	return TnmMibLoadFile(interp, Tcl_GetStringFromObj(objv[2], NULL));

    case cmdMacro:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType");
            return TCL_ERROR;
        }
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	result = TnmGetTableValue(tnmMibMacroTable,
				  typePtr ? typePtr->macro : nodePtr->macro);
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdModule:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType");
            return TCL_ERROR;
        }
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	result = typePtr ? typePtr->moduleName : nodePtr->moduleName;
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdName: {
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
	    return TCL_ERROR;
	}
	TnmOidObjSetRep(objv[2], TNM_OID_AS_NAME);
	Tcl_InvalidateStringRep(objv[2]);
	Tcl_SetObjResult(interp, objv[2]);
        break;
    }

    case cmdOid:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "string");
            return TCL_ERROR;
        }
	oidPtr = TnmGetOidFromObj(interp, objv[2]);
	if (! oidPtr) { 
	    return TCL_ERROR;
	}
	objPtr = objv[2];
	isShared = Tcl_IsShared(objPtr);
	if (isShared) {
	    objPtr = Tcl_DuplicateObj(objPtr);
	}
	/*
	 * Invalidate the string representation in all cases to
	 * make sure that hexadecimal sub-identifier are converted
	 * into decimal sub-identifier.
	 */
	TnmOidObjSetRep(objPtr, TNM_OID_AS_OID);
	Tcl_InvalidateStringRep(objPtr);
        Tcl_SetObjResult(interp, objPtr);
        break;

    case cmdParent: {
	TnmOid nodeOid;
	int i;
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	TnmOidInit(&nodeOid);
	nodePtr = GetMibNode(interp, objv[2], NULL, &nodeOid);
	if (! nodePtr) {
	    TnmOidFree(&nodeOid);
	    return TCL_ERROR;
	}
	if (nodePtr->parentPtr && nodePtr->parentPtr->label) {
	    objPtr = Tcl_GetObjResult(interp);
	    oidPtr = TnmGetOidFromObj(interp, objPtr);
	    for (i = 0; i < TnmOidGetLength(&nodeOid) - 1; i++) {
		TnmOidAppend(oidPtr, TnmOidGet(&nodeOid, i));
	    }
	    TnmOidObjSetRep(objPtr, TnmOidObjGetRep(objv[2]));
	    Tcl_InvalidateStringRep(objPtr);
	}
	TnmOidFree(&nodeOid);
        break;
    }

    case cmdSyntax:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "nodeOrType");
            return TCL_ERROR;
        }
	if (GetMibNodeOrType(interp, objv[2], &typePtr, &nodePtr) != TCL_OK) {
	    return TCL_ERROR;
	}
        if (typePtr) {
	    result = TnmGetTableValue(tnmSnmpTypeTable, typePtr->syntax);
	} else {
	    if (nodePtr->macro == TNM_MIB_OBJECTTYPE) {
		if (nodePtr->typePtr && nodePtr->typePtr->name) {
		    if (nodePtr->typePtr->macro == TNM_MIB_OBJECTTYPE) {
			result = TnmGetTableValue(tnmSnmpTypeTable, 
						  nodePtr->typePtr->syntax);
		    } else {
			result = nodePtr->typePtr->name;
		    }
		} else {
		    result = TnmGetTableValue(tnmSnmpTypeTable,
					      nodePtr->syntax);
		}
	    }
	}
	if (result) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
        break;

    case cmdType:
	if (objc != 3) {
            Tcl_WrongNumArgs(interp, 2, objv, "node");
            return TCL_ERROR;
        }
	nodePtr = GetMibNode(interp, objv[2], NULL, NULL);
	if (! nodePtr) {
            return TCL_ERROR;
	}
	if (nodePtr->typePtr) {
	    if (nodePtr->typePtr->moduleName) {
		Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
			       nodePtr->moduleName, "!", (char *) NULL);
	    }
	    Tcl_AppendStringsToObj(Tcl_GetObjResult(interp),
				   nodePtr->typePtr->name, (char *) NULL);
	} else {
	    result = TnmGetTableValue(tnmSnmpTypeTable, nodePtr->syntax);
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), result, -1);
	}
	break;

    case cmdLength:
	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "oid");
	    return TCL_ERROR;
	}
	oidPtr = TnmGetOidFromObj(interp, objv[2]);
	if (! oidPtr) { 
	    return TCL_ERROR;
	}
	Tcl_SetIntObj(Tcl_GetObjResult(interp), TnmOidGetLength(oidPtr));
	break;
    }

    return TCL_OK;
}
