/* 
   sand - helps you keep your diary

   cli.c - command line interface module

   Copyright (C) 2001 Per Jonsson

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include "system.h"
#include "ddecode.h"
#include "output.h"
#include "sand.h"

/* cli_output - output of a text in cli mode*/
void cli_output()
{
  dnote note;
  dinfo diary;

  /* Set it up for output */
  input_diary(&diary);
  if (ddecode_init(&diary)!= DIARY_IS)
    {
      message_err("Diary-file isn't a diary");
      exit (EXIT_FAILURE);
    }
  note = ddecode_note(&diary, note_number);
  switch (note.length)
    {
    case ERR_INV_FORMAT:
      message_err("Your Diary is in a corrupt format.\nExiting...\n");
      exit(EXIT_FAILURE);
      break;
    case ERR_NO_SUCH_NOTE:
      message_err("The Note you request from the diary isn't available\n");
      message_err("Please try another\n");
      exit (EXIT_FAILURE);
      break;
    default:
      break;
    }
  out_text_fputnote(note, stdout);
  
  free(diary.text);
}

/* cli_dump - handles the dumping in case of cli mode */
void cli_dump()
{
  int j;
  dnote note;
  dinfo diary;

  /*  Set it up for dumping */
  input_diary(&diary);
  if (ddecode_init(&diary)!= DIARY_IS)
    {
      message_err("Diary-file isn't a diary");
      exit (EXIT_FAILURE);
    }
  
  j = 0;
  do {
    note = ddecode_note_next(&diary);
    switch (note.length)
      {
      case ERR_INV_FORMAT:
	message_err("Your Diary is in a corrupt format.\nExiting...\n");
	exit(EXIT_FAILURE);
	break;
      case ERR_NO_SUCH_NOTE:
	j = ERR_NO_SUCH_NOTE;
	break;
      default:
	out_text_fputnote(note, stdout);
	break;
      }
  } while (j != ERR_NO_SUCH_NOTE);
  
  free(diary.text);
  
}

/* cli_input - handles the input case of cli mode*/
void
cli_input(int status)
{
  char * temp_name;
  FILE * temp_file = (FILE *)NULL;
  char * note = (char *)NULL;
  size_t note_size = (size_t) 0;
  int cont_read = 1;
  int char_count = 0;
  ssize_t char_read;
  dnote diary_note;
  FILE * diary_file = (FILE*) NULL;
  int i;

  temp_name = tempnam(program_rc_dir, TEMP_PREFIX);

  /* Output info about what we are using as a temporary file */
  message_ver("Using ",1);
  message_ver(temp_name,1);
  message_ver("\n",1);

  temp_file = fopen(temp_name, "w+x"); /* Open exclusivly */

  /* Do some error checking on the FILE * */ 
  if (temp_file == NULL)
    {

      if (errno == EINVAL){
	/* The system isn't handling my mode, exiting */
	perror("sand_command");
	message_err("System doesn't like my open mode. Bye!\n");
	message_err("Please report this error to the developer.\n");
	exit (EXIT_FAILURE);
      }
      
      /* Somebody is doing something very bad to my temp files */
      perror("sand_command");
      message_err("System doesn't look secure, exiting....\n");
      exit(EXIT_FAILURE);
    }
  
  /* Print a greeting */
  printf("Welcome to your diary. Enter your note.\n");

  /* Loop to read the note */
  do
    {

      /* Read the line */
      if ( cli_getline( &note, &note_size, stdin) == -1)
	cont_read = 0;

      /* Check to see if the note is ended */
      if ( note[0] == '.' && note[1] == '\n')
	cont_read = 0; /* It is, let's end the loop */
      else
	fputs ( note, temp_file); /* It's not, output the line and continue */

      if (cont_read == 1)
	char_count += char_read; /* Increase the count of characters read */

    } while (cont_read == 1); /* End of read the note loop */

  /* Loop until a allowed answer is recieved, but at least once */
  do {
    /* Create a question if we're going to insert the note into the diary */
    printf("\nInsert into diary (yes/no/review note) [y/n/r]: ");

    /* Read the line */
    if ( cli_getline( &note, &note_size, stdin) == -1)
      cont_read = 0;
    
    rewind(temp_file);

    switch (toupper(note[0])) /* Check what the user has chosen */
      {
      case 'Y': /* Yes */
	{
	  cont_read = 2;
	  break;
	}
      case 'N': /* No */
	{
	  printf("Skiping addition of note to diary.\n");
	  printf("Data entered is available as\n%s\n",temp_name);
	  exit (EXIT_OK);
	  break;
	}
      case 'R': /* Review */
	{
	  printf("Here is your note:\n");
	  dump_file(temp_file,stdout);
	  putc('\n',stdout);
	  cont_read = 1;
	  break;
	}
      default:
	{
	cont_read = 1;
	printf("Invalid choice: %c\n", note[0]);
	break;
	}
      }

  } while ( cont_read == 1 );

  if (cont_read == 2)
    {
      /* Title of note */
      printf("If you like, you can enter a title for you note now.\n"
	     "Otherwise, just enter a . followed by a newline.\n");
      
      /* Read the line */
      char_read = cli_getline( &note, &note_size, stdin);
      if (char_read == -1)
	cont_read = 0;
      
      /* Check to see if the note is ended */
      if ( note[0] == '.' && note[1] == '\n')
	note[0] = '\0';
      
      for (i = 0; i < char_read; i++)
	{
	  if (note[i] == '\n')
	    {
	      note[i] = '\0';
	      i = char_read;
	    }
	}
      
      /* Output */
      diary_file = open_diary(O_RDWR);
      diary_note = dencode_note_now("", note);
      out_diary_special_fputnote(diary_note, diary_file, temp_file);
      if ( unlink (temp_name) != 0)
	{
	  perror("sand_command");
	  message_err("Couldn't remove temporary file\n");
	  exit (EXIT_FAILURE);
	}
	  
    }
  /** WORK HERE **/

}


ssize_t cli_getline(char **LINEPTR, size_t *N, FILE *STREAM)
{
  ssize_t char_read;

  /* Read the line */
  char_read = getline( LINEPTR, N, STREAM);
    
  /* If char_read == -1 then something possibly went wrong */
  if (char_read == -1)
    {
      if (!feof(stdin))        /* Something is seriously wrong */
	{
	  message_err("You're giving me evil input\nExiting....\n");
	  exit (EXIT_FAILURE); /* Escape from the program */
	} 
      else                     /* Where just reading a EOF at an */
	{                      /* unusual place */
	  message_ver("Read EOF. Exiting....\n",1);
	  exit (EXIT_FAILURE); /* Get out of the program */
	}
    }

  return char_read;
}

