#! /usr/bin/env python

'''
Python Interface to SAGA Tools Library
----------
- Category : Import/Export
- Name     : ODBC/OTL
- ID       : db_odbc

Description
----------
Database access via Open Data Base Connection (ODBC) interface. Based on the OTL (Oracle, Odbc and DB2-CLI Template Library), Version 4.0: <a target="_blank" href="http://otl.sourceforge.net/">http://otl.sourceforge.net/</a>
'''

from PySAGA.helper import Tool_Wrapper

def Run_Connect_to_ODBC_Source(SERVER=None, USERNAME=None, PASSWORD=None, Verbose=2):
    '''
    Connect to ODBC Source
    ----------
    [db_odbc.0]\n
    Connects to an ODBC source.\n
    Arguments
    ----------
    - SERVER [`choice`] : Server. Available Choices: Default: 0
    - USERNAME [`text`] : User
    - PASSWORD [`text`] : Password

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '0', 'Connect to ODBC Source')
    if Tool.is_Okay():
        Tool.Set_Option('SERVER', SERVER)
        Tool.Set_Option('USERNAME', USERNAME)
        Tool.Set_Option('PASSWORD', PASSWORD)
        return Tool.Execute(Verbose)
    return False

def Run_Disconnect_from_ODBC_Source(ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, TRANSACT=None, Verbose=2):
    '''
    Disconnect from ODBC Source
    ----------
    [db_odbc.1]\n
    Disconnects an ODBC source connection.\n
    Arguments
    ----------
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '1', 'Disconnect from ODBC Source')
    if Tool.is_Okay():
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Run_CommitRollback_Transaction(ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, SERVERS=None, TRANSACT=None, Verbose=2):
    '''
    Commit/Rollback Transaction
    ----------
    [db_odbc.2]\n
    Execute a commit or rollback on open transactions with ODBC source.\n
    Arguments
    ----------
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - SERVERS [`choice`] : Server. Available Choices: Default: 0
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '2', 'Commit/Rollback Transaction')
    if Tool.is_Okay():
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('SERVERS', SERVERS)
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

def Run_Execute_SQL(ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, SQL=None, COMMIT=None, STOP=None, Verbose=2):
    '''
    Execute SQL
    ----------
    [db_odbc.3]\n
    Execute SQL commands on a connected ODBC source. Separate different commands with a semicolon (';').\n
    Arguments
    ----------
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - SQL [`long text`] : SQL Statment. Default: CREATE TABLE myTable1 (Col1 VARCHAR(255) PRIMARY KEY, Col2 INTEGER);
INSERT INTO myTable1 (Col1, Col2) VALUES('First Value', 1);
DROP TABLE myTable1;

    - COMMIT [`boolean`] : Commit. Default: 1
    - STOP [`boolean`] : Stop on Error. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '3', 'Execute SQL')
    if Tool.is_Okay():
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('SQL', SQL)
        Tool.Set_Option('COMMIT', COMMIT)
        Tool.Set_Option('STOP', STOP)
        return Tool.Execute(Verbose)
    return False

def Run_List_Table_Fields(TABLE=None, ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, TABLES=None, Verbose=2):
    '''
    List Table Fields
    ----------
    [db_odbc.4]\n
    Loads table information from ODBC data source.\n
    Arguments
    ----------
    - TABLE [`output table`] : Field Description
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - TABLES [`choice`] : Tables. Available Choices: Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '4', 'List Table Fields')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('TABLES', TABLES)
        return Tool.Execute(Verbose)
    return False

def Run_Import_Table(TABLE=None, ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, TABLES=None, Verbose=2):
    '''
    Import Table
    ----------
    [db_odbc.5]\n
    Imports a table from a database via ODBC.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - TABLES [`choice`] : Tables. Available Choices: Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '5', 'Import Table')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('TABLES', TABLES)
        return Tool.Execute(Verbose)
    return False

def Run_Export_Table(TABLE=None, ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, NAME=None, EXISTS=None, Verbose=2):
    '''
    Export Table
    ----------
    [db_odbc.6]\n
    Exports a table to a database via ODBC.\n
    Arguments
    ----------
    - TABLE [`input table`] : Table
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - NAME [`text`] : Table Name
    - EXISTS [`choice`] : If table exists.... Available Choices: [0] abort export [1] replace existing table [2] append records, if table structure allows Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '6', 'Export Table')
    if Tool.is_Okay():
        Tool.Set_Input ('TABLE', TABLE)
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('NAME', NAME)
        Tool.Set_Option('EXISTS', EXISTS)
        return Tool.Execute(Verbose)
    return False

def Run_Drop_Table(ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, TABLES=None, Verbose=2):
    '''
    Drop Table
    ----------
    [db_odbc.7]\n
    Deletes a table from a database via ODBC.\n
    Arguments
    ----------
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - TABLES [`choice`] : Tables. Available Choices: Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '7', 'Drop Table')
    if Tool.is_Okay():
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('TABLES', TABLES)
        return Tool.Execute(Verbose)
    return False

def Run_Import_Table_from_SQL_Query(TABLE=None, ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, TABLES=None, FIELDS=None, WHERE=None, GROUP=None, HAVING=None, ORDER=None, DISTINCT=None, Verbose=2):
    '''
    Import Table from SQL Query
    ----------
    [db_odbc.8]\n
    Import a SQL table from a database via ODBC.\n
    Arguments
    ----------
    - TABLE [`output table`] : Table from SQL Query
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password
    - TABLES [`text`] : Tables
    - FIELDS [`text`] : Fields. Default: *
    - WHERE [`text`] : Where
    - GROUP [`text`] : Group by
    - HAVING [`text`] : Having
    - ORDER [`text`] : Order by
    - DISTINCT [`boolean`] : Distinct. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '8', 'Import Table from SQL Query')
    if Tool.is_Okay():
        Tool.Set_Output('TABLE', TABLE)
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        Tool.Set_Option('TABLES', TABLES)
        Tool.Set_Option('FIELDS', FIELDS)
        Tool.Set_Option('WHERE', WHERE)
        Tool.Set_Option('GROUP', GROUP)
        Tool.Set_Option('HAVING', HAVING)
        Tool.Set_Option('ORDER', ORDER)
        Tool.Set_Option('DISTINCT', DISTINCT)
        return Tool.Execute(Verbose)
    return False

def Run_List_ODBC_Servers(SERVERS=None, CONNECTED=None, Verbose=2):
    '''
    List ODBC Servers
    ----------
    [db_odbc.9]\n
    Lists all ODBC sources.\n
    Arguments
    ----------
    - SERVERS [`output table`] : Server
    - CONNECTED [`boolean`] : Only List Connected Servers. Default: 0

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '9', 'List ODBC Servers')
    if Tool.is_Okay():
        Tool.Set_Output('SERVERS', SERVERS)
        Tool.Set_Option('CONNECTED', CONNECTED)
        return Tool.Execute(Verbose)
    return False

def Run_List_Tables(TABLES=None, ODBC_DSN=None, ODBC_USR=None, ODBC_PWD=None, Verbose=2):
    '''
    List Tables
    ----------
    [db_odbc.10]\n
    Lists all tables of an ODBC data source.\n
    Arguments
    ----------
    - TABLES [`output table`] : Tables
    - ODBC_DSN [`text`] : DSN. Data Source Name
    - ODBC_USR [`text`] : User. User Name
    - ODBC_PWD [`text`] : Password. Password

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '10', 'List Tables')
    if Tool.is_Okay():
        Tool.Set_Output('TABLES', TABLES)
        Tool.Set_Option('ODBC_DSN', ODBC_DSN)
        Tool.Set_Option('ODBC_USR', ODBC_USR)
        Tool.Set_Option('ODBC_PWD', ODBC_PWD)
        return Tool.Execute(Verbose)
    return False

def Run_Disconnect_All(TRANSACT=None, Verbose=2):
    '''
    Disconnect All
    ----------
    [db_odbc.11]\n
    Disconnects all connected ODBC sources.\n
    Arguments
    ----------
    - TRANSACT [`choice`] : Transactions. Available Choices: [0] rollback [1] commit Default: 1

    - Verbose [`integer number`] : Verbosity level, 0=silent, 1=tool name and success notification, 2=complete tool output.\n
    Returns
    ----------
    `boolean` : `True` on success, `False` on failure.
    '''
    Tool = Tool_Wrapper('db_odbc', '11', 'Disconnect All')
    if Tool.is_Okay():
        Tool.Set_Option('TRANSACT', TRANSACT)
        return Tool.Execute(Verbose)
    return False

