#include "config.h"

#include <iostream>
// #include <sstream>
#include <iomanip>
#include <string>
#include <cstring>
#include <cctype>

#ifdef HAVE_TIME_H
#include <time.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#include "asserts.h"
#include "error.h"
#include "estring.h"
#include "tstamp.h"

/** C'tor */
timestamp::timestamp()
{
	clear();
	set();
}

/** C'tor */
timestamp::timestamp(const timestamp& a_t)
{
	clear();
	assign(a_t);
}

/** C'tor */
timestamp::timestamp(
	const int a_year,
	const int a_month,
	const int a_day,
	const int a_hour,
	const int a_minute,
	const int a_second
	)
{
	clear();
	assign(a_year,a_month,a_day,a_hour,a_minute,a_second);
}

/** C'tor */
timestamp::timestamp(const std::string& a_s)
{
	clear();
	assign(a_s);
}

/** Set the timestamp to the current time and date */
void timestamp::set(void)
{
	time_t t = 0;
	struct tm* tm_ptr = 0;

	t = time(0);
	if (t == -1)
		throw(ERROR(errno,"Could not retrieve current time"));
	tm_ptr = localtime(&t);
	if (tm_ptr == 0)
		throw(ERROR(errno,"Could not convert current time to local time"));
	m_year = tm_ptr->tm_year + 1900;
	m_month = tm_ptr->tm_mon + 1;
	m_day = tm_ptr->tm_mday;
	m_hour = tm_ptr->tm_hour;
	m_minute = tm_ptr->tm_min;
	m_second = tm_ptr->tm_sec;
	m_resolution = resolution_day;
}

/** Set the timestamp to the value of another timestamp */
void timestamp::assign(const timestamp& a_t)
{
	m_year = a_t.year();
	m_month = a_t.month();
	m_day = a_t.day();
	m_hour = a_t.hour();
	m_minute = a_t.minute();
	m_second = a_t.second();
	m_resolution = a_t.resolution();
}

/** Set the timestamp */
void timestamp::assign(
	const int a_year,
	const int a_month,
	const int a_day,
	const int a_hour,
	const int a_minute,
	const int a_second
	)
{
	std::string es;

	if ((a_year < 0) || (a_year > 9999)) {
		TRY_nomem(es = "Invalid year: \"");
		TRY_nomem(es += estring(a_year));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_year = a_year;

	if ((a_month > 12) || (a_month < 1)) {
		TRY_nomem(es = "Invalid month: \"");
		TRY_nomem(es += estring(a_month));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_month = a_month;

	if ((a_day > 31) || (a_day < 1)) {
		TRY_nomem(es = "Invalid day: \"");
		TRY_nomem(es += estring(a_day));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_day = a_day;

	if ((a_hour < 0) || (a_hour > 23)) {
		TRY_nomem(es = "Invalid hour: \"");
		TRY_nomem(es += estring(a_hour));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_hour = a_hour;

	if ((a_minute < 0) || (a_minute > 59)) {
		TRY_nomem(es = "Invalid minute: \"");
		TRY_nomem(es += estring(a_minute));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_minute = a_minute;

	if ((a_second < 0) || (a_second > 59)) {
		TRY_nomem(es = "Invalid second: \"");
		TRY_nomem(es += estring(a_second));
		TRY_nomem(es += "\"");

		throw(ERROR(0,es));
	}
	m_second = a_second;
}

/** Set the timestamp from a string */
void timestamp::assign(const std::string& a_s)
{
	std::string bes;
	std::string es;
	std::string ies;

	int l_year = -1;
	int l_month = -1;
	int l_day = -1;
	int l_hour = -1;
	int l_minute = -1;
	int l_second = -1;
	resolution_type r = resolution_day;

	TRY_nomem(es = "Invalid timestamp string: \"");
	TRY_nomem(es += a_s);
	TRY_nomem(es += "\"");

	if (!is_timestamp(a_s))
		throw(ERROR(0,es));

	TRY_nomem(bes = "Parse error converting string to timestamp: \"");
	TRY_nomem(bes += a_s);
	TRY_nomem(bes += "\" ");

	TRY_nomem(es = bes + "Invalid year");
	if (a_s.size() >= 4)
	{
		estring str;

		r = resolution_year;
		TRY_nomem(str = a_s.substr(0,4));
		TRY(l_year = str,ies);
	}
	else
		throw(ERROR(0,es));

	if (a_s.size() == 4) {
		assign(l_year,1,1,0,0,0);
		resolution(r);
		return;
	}

	TRY_nomem(es = bes + "Invalid month");
	if (a_s.size() >= 7)
	{
		estring str;

		r = resolution_month;
		TRY_nomem(str = a_s.substr(5,2));
		TRY(l_month = str,ies);
	}
	else
		throw(ERROR(0,es));

	if (a_s.size() == 7) {
		assign(l_year,l_month,1,0,0,0);
		resolution(r);
		return;
	}
	
	TRY_nomem(es = bes + "Invalid day");
	if (a_s.size() >= 10)
	{
		estring str;

		r = resolution_day;
		TRY_nomem(str = a_s.substr(8,2));
		TRY(l_day = str,ies);
	}
	else
		throw(ERROR(0,es));
	
	if (a_s.size() == 10) {
		assign(l_year,l_month,l_day,0,0,0);
		resolution(r);
		return;
	}

	TRY_nomem(es = bes + "Invalid hour");
	if (a_s.size() >= 13)
	{
		estring str;

		r = resolution_hour;
		TRY_nomem(str = a_s.substr(11,2));
		TRY(l_hour = str,ies);
	}
	else
		throw(ERROR(0,es));
	
	if (a_s.size() == 13) {
		assign(l_year,l_month,l_day,l_hour,0,0);
		resolution(r);
		return;
	}
	
	TRY_nomem(es = bes + "Invalid minute");
	if (a_s.size() >= 15)
	{
		estring str;

		r = resolution_minute;
		TRY_nomem(str = a_s.substr(13,2));
		TRY(l_minute = str,ies);
	}
	else
		throw(ERROR(0,es));
	
	if (a_s.size() == 15) {
		assign(l_year,l_month,l_day,l_hour,l_minute,0);
		resolution(r);
		return;
	}

	TRY_nomem(es = bes + "Invalid second");
	if (a_s.size() == 17)
	{
		estring str;

		r = resolution_second;
		TRY_nomem(str = a_s.substr(15,2));
		TRY(l_second = str,ies);
	}
	else
		throw(ERROR(0,es));
	
	assign(l_year,l_month,l_day,l_hour,l_minute,l_second);
	resolution(r);
}

/** Clear the timestamp */
void timestamp::clear(void)
{
	m_year = 0;
	m_month = 0;
	m_day = 0;
	m_hour = 0;
	m_minute = 0;
	m_second = 0;
	m_resolution = resolution_day;
}

/** Set the timestamp resolution */
void timestamp::resolution(timestamp::resolution_type a_r)
{
	m_resolution = a_r;
}

/** Generate a string */
const std::string timestamp::make_str_(const int a_resolution) const
{
	std::string bes;
	std::string es;
	std::string str;
	estring tmp_str;

	TRY_nomem(es = "Could not create timestamp string");

	if (a_resolution >= resolution_year) {
		tmp_str.width(4);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting year");
		TRY(tmp_str = year(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}
	
	if (a_resolution >= resolution_month) {
		TRY_nomem(es = bes);
		TRY(str += "-",es);
		tmp_str.width(2);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting month");
		TRY(tmp_str = month(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}
	
	if (a_resolution >= resolution_day) {
		TRY_nomem(es = bes);
		TRY(str += "-",es);
		tmp_str.width(2);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting day");
		TRY(tmp_str = day(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}
	
	if (a_resolution >= resolution_hour) {
		TRY_nomem(es = bes);
		TRY(str += ".",es);
		tmp_str.width(2);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting hour");
		TRY(tmp_str = hour(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}
	
	if (a_resolution >= resolution_minute) {
		tmp_str.width(2);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting minute");
		TRY(tmp_str = minute(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}
	
	if (a_resolution >= resolution_second) {
		tmp_str.width(2);
		tmp_str.align(estring::right);
		tmp_str.left_fillchar('0');
		TRY_nomem(es = bes + ", error converting second");
		TRY(tmp_str = second(),es);
		TRY_nomem(str += tmp_str.fmt_str());
		tmp_str.reset();
	}

	return(str);
}

/** Generate a string */
const std::string timestamp::str(void) const
{
	std::string str;

	TRY_nomem(str = make_str_(m_resolution));

	return(str);
}

/** Generate a string */
const std::string timestamp::str(
	const timestamp::resolution_type a_resolution) const
{
	std::string str;

	TRY_nomem(str = make_str_(a_resolution));

	return(str);
}

/** Return the timestamp second */
int timestamp::second(void) const
{
	return(m_second);
}

/** Return the timestamp minute */
int timestamp::minute(void) const
{
	return(m_minute);
}

/** Return the timestamp hour */
int timestamp::hour(void) const
{
	return(m_hour);
}

/** Return the timestamp day */
int timestamp::day(void) const
{
	return(m_day);
}

/** Return the timestamp month */
int timestamp::month(void) const
{
	return(m_month);
}

/** Return the timestamp year */
int timestamp::year(void) const
{
	return(m_year);
}

/** Return the timestamp resolution */
timestamp::resolution_type timestamp::resolution(void) const
{
	return(m_resolution);
}

/** Assignment */
timestamp& timestamp::operator = (const timestamp& a_t)
{
	assign(a_t);

	return(*this);
}

/** Assignment */
timestamp& timestamp::operator = (const std::string& a_s)
{
	assign(a_s);

	return(*this);
}

/** Comparison */
bool timestamp::operator < (const timestamp& a_t) const
{
	resolution_type r;
	bool value;

	r = std::min(m_resolution, a_t.resolution());
	value = (str(r) < a_t.str(r));

	return(value);
}

/** Comparison */
bool timestamp::operator > (const timestamp& a_t) const
{
	resolution_type r;
	bool value;

	r = std::min(m_resolution, a_t.resolution());
	value = (str(r) > a_t.str(r));

	return(value);
}

/** Comparison */
bool timestamp::operator == (const timestamp& a_t) const
{
	resolution_type r;
	bool value;

	r = std::min(m_resolution, a_t.resolution());
	value = (str(r) == a_t.str(r));

	return(value);
}

/** Return true if the string is a valid timestamp */
bool is_timestamp(const std::string& a_s)
{
	estring str;
	int i;

	if (a_s.size() < 4) {
		return(false);
	}
	if (!isdigit(a_s[0])) {
		return(false);
	}
	if (!isdigit(a_s[1])) {
		return(false);
	}
	if (!isdigit(a_s[2])) {
		return(false);
	}
	if (!isdigit(a_s[3])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(0,4));
	try { 
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 0) || (i > 9999)) {
		return(false);
	}
	if (a_s.size() == 4) {
		return(true);
	}

	if (a_s.size() < 7) {
		return(false);
	}
	if (a_s[4] != '-') {
		return(false);
	}
	if (!isdigit(a_s[5])) {
		return(false);
	}
	if (!isdigit(a_s[6])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(5,2));
	try {
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 1) || (i > 12)) {
		return(false);
	}
	if (a_s.size() == 7) {
		return(true);
	}

	if (a_s.size() < 10) {
		return(false);
	}
	if (a_s[7] != '-') {
		return(false);
	}
	if (!isdigit(a_s[8])) {
		return(false);
	}
	if (!isdigit(a_s[9])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(8,2));
	try {
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 1) || (i > 31)) {
		return(false);
	}
	if (a_s.size() == 10) {
		return(true);
	}

	if (a_s.size() < 13) {
		return(false);
	}
	if (a_s[10] != '.') {
		return(false);
	}
	if (!isdigit(a_s[11])) {
		return(false);
	}
	if (!isdigit(a_s[12])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(11,2));
	try {
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 0) || (i > 23)) {
		return(false);
	}
	if (a_s.size() == 13) {
		return(true);
	}

	if (a_s.size() < 15) {
		return(false);
	}
	if (!isdigit(a_s[13])) {
		return(false);
	}
	if (!isdigit(a_s[14])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(13,2));
	try {
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 0) || (i > 59)) {
		return(false);
	}
	if (a_s.size() == 15) {
		return(true);
	}

	if (a_s.size() < 17) {
		return(false);
	}
	if (!isdigit(a_s[15])) {
		return(false);
	}
	if (!isdigit(a_s[16])) {
		return(false);
	}
	TRY_nomem(str = a_s.substr(15,2));
	try {
		i = str;
	}
	catch(...) {
		return(false);
	}
	if ((i < 0) || (i > 59)) {
		return(false);
	}
	if (a_s.size() == 17) {
		return(true);
	}

	return(false);
}

