#ifndef __rconfig_h__
#define __rconfig_h__

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <map>

#include "asserts.h"
#include "types.h"
#include "tstamp.h"

//----------------------------------------------------------------------------

class parser;
class parse_job;

/** A configuration manager support class:
	Used to specify a single subdirectory in a path of subdirectory names for a
	unique path under the archive directory for each job
	*/
class archive_path_element
{
public:
	enum element_type {
		jobname,
		groupname,
		hostname,
		pathname,
		permutation,
		literal
	};

	void clear(void);

	archive_path_element();
	archive_path_element(const archive_path_element& a_class);
	archive_path_element(const element_type& a_enum);
	archive_path_element(const std::string& a_str);

	void assign(const archive_path_element& a_class);
	void assign(const element_type& a_enum);
	void assign(const std::string& a_str);

	const element_type& type(void) const;
	const std::string& value(void) const;
	const std::string str(void) const;

	archive_path_element& operator=(const archive_path_element& a_class);
	archive_path_element& operator=(const element_type a_enum);
	archive_path_element& operator=(const std::string& a_str);

private:
	element_type m_type;
	std::string m_literal;
};

/** A configuration manager support class:
	Used to specify a unique subdirectory in the archive for each job (made up
	of archive_path_element's)
 */
class archive_path : public std::vector<archive_path_element>
{
public:
	typedef std::vector<archive_path_element> type;

	void reset(void);

	archive_path();
	archive_path(const archive_path& a_class);
	archive_path(const archive_path_element& a_class);
	archive_path(const std::string& a_class);

	void push_back(const archive_path& a_class);
	void push_back(const archive_path_element& a_class);
	void push_back(const std::string& a_str);

	void assign(const archive_path& a_class);
	void assign(const archive_path_element& a_class);
	void assign(const std::string& a_str);

	const std::string str(void) const;

	archive_path& operator=(const archive_path& a_class);
	archive_path& operator=(const archive_path_element& a_class);
	archive_path& operator=(const std::string& a_str);

private:
};

/** A configuration manager support class:
	Used to map rsync exit codes to actions for rvm to take
 */
class rsync_behavior
{
public:
	enum behavior_type {
		quit,
		fail,
		retry,
		retry_without_hardlinks,
		ok
	};
	typedef behavior_type value_type;
	typedef std::map<uint16, value_type> map_type;
	static const uint16 default_behavior = ((uint16)-1);

	void clear(void);
	void reset(void);

	rsync_behavior();
	rsync_behavior(const rsync_behavior& a_class);

	void assign(const uint16 a_code, const value_type a_action);
	void assign(const rsync_behavior& a_class);
	void assign(const std::string& a_str);

	const value_type operator[](const uint16 a_code) const;
	value_type& operator[](const uint16 a_code);
	rsync_behavior& operator=(const rsync_behavior& a_class);
	rsync_behavior& operator=(const value_type a_enum);
	rsync_behavior& operator=(const std::string& a_str);

	const value_type default_value(void) const;
	const map_type& map_value(void) const;

private:
	map_type m_map;
	value_type m_default;
};

class configuration_manager;

/** Hold configuration data for a single job */
class job
{
public:
	enum rsync_connection_type {
		connection_remote,
		connection_local,
		connection_server,
		connection_ssh_local
	};
	enum rsync_behavior_type {
		behavior_ok,
		behavior_fail,
		behavior_retry,
		behavior_retry_without_hardlinks
	};
	typedef std::vector<std::string> excludes_type;
	typedef std::vector<std::string> includes_type;
	typedef std::vector<std::string> paths_type;

	job();
	job(const job& a_job);

	void clear(void);
	void assign(const job& a_job);

	job& operator=(const job& a_job);

	std::string default_config_path;
	uint16 default_config_line;
	std::string config_path;
	uint16 config_line;
	class archive_path archive_path;
	excludes_type excludes;
	includes_type includes;
	std::string groupname;
	std::string hostname;
	std::string jobname;
	paths_type paths;
	class rsync_behavior rsync_behavior;
	rsync_connection_type rsync_connection;
	bool rsync_hardlink;
	bool rsync_multi_hardlink;
	uint16 rsync_multi_hardlink_max;
	std::string rsync_options;
	std::string ssh_options;
	std::string rsync_remote_user;
	std::string rsync_remote_path;
	std::string rsync_remote_module;
	uint16 rsync_remote_port;
	uint16 rsync_retry_count;
	uint16 rsync_retry_delay;
	uint16 rsync_timeout;

	const std::string generate_archive_path(const std::string& a_path) const;
	const std::string generate_source_path(const std::string& a_path) const;
	const std::string generate_job_id(void) const;
	const std::vector<std::string> generate_rsync_options_vector(void) const;
	const std::vector<std::string> generate_ssh_options_vector(void) const;
	const std::string common_pathname(void) const;
	void check(void);

private:

	friend class parser;
};

/** Keep up with configuration settings for RVM and it's jobs */
class configuration_manager
{
public:
	enum action_type {
		action_help,
		action_version,
		action_archive,
		action_relink,
		action_check_config
	};
	enum cfgfile_type {
		config_file,
		job_file,
	};
	enum overflow_type {
		overflow_quit,
		overflow_delete_oldest,
		overflow_delete_until_free
	};
	enum selection_type {
		selection_round_robin,
		selection_max_free
	};
	enum logging_type {
		logging_manager,
		logging_child,
		logging_rsync
	};
	typedef std::pair<cfgfile_type, std::string> cfgfile_element;
	typedef class std::vector<cfgfile_element> cfgfiles_type;
	typedef std::vector<job> jobs_type;
	typedef std::vector<std::string> vaults_type;

	void clear(void);

	configuration_manager();

	void init(int argc, char const *argv[]);
	void check(void) const;
	const bool initialized(void) const;

	const action_type action(void) const;
	const bool use_default(void) const;
	void default_file(const std::string& a_path);
	const std::string& default_file(void) const;
	void default_logdir(const std::string& a_path);
	const class timestamp& timestamp(void) const;
	const std::string& link_catalog_dir(void) const;
	const std::string& log_dir(void) const;
	const bool delete_old_log_files(void) const;
	const bool delete_old_report_files(void) const;
	const std::string& rsync_local_path(void) const;
	const std::string& delete_command_path(void) const;
	const std::string& ssh_local_path(void) const;
	const uint16& rsync_parallel(void) const;
	const uint16& io_poll_interval(void) const;
	const timestamp::resolution_type timestamp_resolution(void) const;
	const vaults_type& vaults(void) const;
	const overflow_type& vault_overflow_behavior(void) const;
	const uint16& vault_overflow_blocks(void) const;
	const uint16& vault_overflow_inodes(void) const;
	const selection_type& vault_selection_behavior(void) const;
	const bool vault_locking(void) const;
	const job& default_job(void) const;
	const jobs_type& jobs(void) const;
	const logging_type& logging_level(void) const;
	const logging_type& error_logging_level(void) const;

private:
	bool m_initialized;
	uint16 m_configs_read;
	std::string m_default_file;
	action_type m_action;
	bool m_default;
	class timestamp m_timestamp;
	cfgfiles_type m_cfgfiles;
	std::string m_link_catalog_dir;
	std::string m_log_dir;
	std::string m_rsync_local_path;
	std::string m_delete_command_path;
	std::string m_ssh_local_path;
	uint16 m_rsync_parallel;
	uint16 m_io_poll_interval;
	std::vector<std::string> m_vaults;
	overflow_type m_vault_overflow_behavior;
	uint16 m_vault_overflow_blocks;
	uint16 m_vault_overflow_inodes;
	selection_type m_vault_selection_behavior;
	bool m_vault_locking;
	bool m_delete_old_log_files;
	bool m_delete_old_report_files;
	job m_default_job;
	jobs_type m_jobs;
	logging_type m_logging_level;
	logging_type m_error_logging_level;

	friend class global_parser;
	friend class job_parser;

	void read_config(const std::string& a_path);
	void read_job(const std::string& a_path);
};

void parse_line(
	std::istream& a_in, 
	std::string& a_keyword, 
	std::string& a_value
	);

/** A configuration manager support class:
	Used to parse a configuration file from the perspective of a global context
 */
class global_parser
{
public:
	global_parser(
		const std::string& a_path,
		uint16& a_line,
		std::istream& a_in
		);

private:
	const std::string* m_path;
	std::istream* m_in;
	uint16* m_line;

	const std::string location(void);
	void read_config(const std::string& a_path);
	void read_job(const std::string& a_path, job& a_job);

	void parse(void);
	void parse_default(const std::string& a_value);
	void parse_include(const std::string& a_value);
	void parse_include_job(const std::string& a_value);
	void parse_job(const std::string& a_value);
	void parse_link_catalog_dir(const std::string& a_value);
	void parse_log_dir(const std::string& a_value);
	void parse_delete_old_log_files(const std::string& a_value);
	void parse_delete_old_report_files(const std::string& a_value);
	void parse_logging_level(const std::string& a_value);
	void parse_error_logging_level(const std::string& a_value);
	void parse_rsync_local_path(const std::string& a_value);
	void parse_delete_command_path(const std::string& a_value);
	void parse_ssh_local_path(const std::string& a_value);
	void parse_rsync_parallel(const std::string& a_value);
	void parse_io_poll_interval(const std::string& a_value);
	void parse_timestamp_resolution(const std::string& a_value);
	void parse_vault(const std::string& a_value);
	void parse_vault_overflow_behavior(const std::string& a_value);
	void parse_vault_overflow_blocks(const std::string& a_value);
	void parse_vault_overflow_inodes(const std::string& a_value);
	void parse_vault_selection_behavior(const std::string& a_value);
	void parse_vault_locking(const std::string& a_value);
};

/** A configuration manager support class:
	Used to parse a configuration file from the perspective of a job context
 */
class job_parser
{
public:
	job_parser(
		job * a_job, 
		const std::string& a_path,
		uint16& a_line,
		std::istream& a_in,
		const std::string& a_block_delimiter,
		const bool a_default_context
		);

private:
	job* m_job;
	const std::string* m_path;
	std::istream* m_in;
	uint16* m_line;
	const std::string* m_delimiter;
	bool m_default_context;

	const std::string location(void);
	void read_job(const std::string& a_path);

	void parse(void);
	void parse_archive_path(const std::string& a_value);
	void parse_clear(const std::string& a_value);
	void parse_exclude_from(const std::string& a_value);
	void parse_include_from(const std::string& a_value);
	void parse_groupname(const std::string& a_value);
	void parse_hostname(const std::string& a_value);
	void parse_include(const std::string& a_value);
	void parse_jobname(const std::string& a_value);
	void parse_path(const std::string& a_value);
	void parse_rsync_behavior(const std::string& a_value);
	void parse_rsync_connection_type(const std::string& a_value);
	void parse_rsync_hardlink(const std::string& a_value);
	void parse_rsync_multi_hardlink(const std::string& a_value);
	void parse_rsync_multi_hardlink_max(const std::string& a_value);
	void parse_rsync_options(const std::string& a_value);
	void parse_ssh_options(const std::string& a_value);
	void parse_rsync_options_context(const std::string& a_value);
	void parse_ssh_options_context(const std::string& a_value);
	void parse_rsync_remote_user(const std::string& a_value);
	void parse_rsync_remote_path(const std::string& a_value);
	void parse_rsync_remote_port(const std::string& a_value);
	void parse_rsync_remote_module(const std::string& a_value);
	void parse_rsync_retry_count(const std::string& a_value);
	void parse_rsync_retry_delay(const std::string& a_value);
	void parse_rsync_timeout(const std::string& a_value);
};

extern configuration_manager config;

#endif
