#ifndef __fs_h__
#define __fs_h__

#include <iostream>
#include <string>
#include <vector>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <pwd.h>
#include <grp.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

/*
 * *** NASTY KLUDGE ***
 *
 * _FILE_OFFSET_BITS == 64 is needed for a 64-bit off_t, but on sun-solaris2.8
 * it interferes with struct statvfs64, which needs _LARGE_FILES instead.
 * Surely there is a proper way to do this...
 *
 */
#ifdef HAVE_SYS_STATVFS_H
#	ifdef sun
#		ifdef _FILE_OFFSET_BITS
#			if _FILE_OFFSET_BITS == 64
#				define __HOLD_FILE_OFFSET_BITS_64
#				undef _FILE_OFFSET_BITS
#			endif
#		endif
#	endif
#include <sys/statvfs.h>
#	ifdef __HOLD_FILE_OFFSET_BITS_64
#		define _FILE_OFFSET_BITS 64
#		undef __HOLD_FILE_OFFSET_BITS_64
#	endif
#endif

#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#endif

#include "types.h"

#ifdef HAVE_SYS_STATVFS_H
#define STATFS statvfs
#else
#define STATFS statfs
#endif

#include <string.h>

const std::string cwd(void);
const pid_t pid(void);
const pid_t parent_pid(void);

bool absolute_path(const std::string& a_path);
bool relative_path(const std::string& a_path);
std::string reform_path(const std::string& a_path);
std::string permute_path(const std::string& a_path);
std::string path_basename(const std::string& a_path);
std::string path_dirname(const std::string& a_path);
std::string mk_absolute_path(
	const std::string a_path, 
	const std::string a_rel_path
	);
std::string mk_relative_path(
	const std::string a_path_to,
	const std::string a_path_from
	);
bool exists(const std::string& a_path);
bool readable(const std::string& a_path);
bool writable(const std::string& a_path);
bool executable(const std::string& a_path);
#ifdef S_ISFIFO
bool is_fifo_special(const std::string& a_path);
#endif
#ifdef S_ISCHR
bool is_char_special(const std::string& a_path);
#endif
#ifdef S_ISDIR
bool is_dir(const std::string& a_path);
#endif
#ifdef S_ISREG
bool is_file(const std::string& a_path);
#endif
#ifdef S_ISBLK
bool is_block_special(const std::string& a_path);
#endif
#ifdef S_ISLNK
bool is_link(const std::string& a_path);
#endif
#ifdef S_ISSOCK
bool is_socket(const std::string& a_path);
#endif
#ifdef S_ISDOOR
bool is_door(const std::string& a_path);
#endif
void mk_dir(const std::string& a_path);
void rm_dir(const std::string a_path);
void rm_file(const std::string a_path);
void mk_dirhier(const std::string a_path);
void rename_file(const std::string a_from, const std::string a_to);
void mk_symlink(const std::string a_from, const std::string a_to);
void mk_relative_symlink(const std::string a_from, const std::string a_to);

/** Retrieve information about a file or directory */
class filestatus
{
public:
	enum filetype {
#ifdef S_ISFIFO
		type_fifo_special,
#endif
#ifdef S_ISCHR
		type_character_special,
#endif
#ifdef S_ISDIR
		type_directory,
#endif
#ifdef S_ISREG
		type_regular_file,
#endif
#ifdef S_ISBLK
		type_block_special,
#endif
#ifdef S_ISLNK
		type_link,
#endif
#ifdef S_ISSOCK
		type_socket,
#endif
#ifdef S_ISDOOR
		type_door,
#endif
		type_unknown
	};
	typedef mode_t mode_type;
	typedef ino_t inode_type;
	typedef dev_t device_type;
	typedef uint32 major_type;
	typedef uint32 minor_type;
	typedef nlink_t num_links_type;
	typedef uid_t uid_type;
	typedef gid_t gid_type;
	typedef uint64 size_type;
	typedef time_t time_type;

	filestatus();
	filestatus(const std::string a_path);
	~filestatus();

	void path(const std::string a_path);
	const std::string path(void) const;
	const filetype type(void) const;
	const std::string link(void) const;

	const mode_type mode(void) const;
	const inode_type inode(void) const;
	const device_type dev(void) const;
	const device_type rdev(void) const;
	const major_type get_major(void) const;
	const minor_type get_minor(void) const;
	const num_links_type num_links(void) const;
	const uid_type uid(void) const;
	const gid_type gid(void) const;
	const size_type size(void) const;
	const time_type last_access_time(void) const;
	const time_type last_modification_time(void) const;
	const time_type last_status_change_time(void) const;
	const size_type blocksize(void) const;
	const size_type blocks(void) const;
	const bool uid_is_found(void) const;
	const bool gid_is_found(void) const;
	const std::string uid_name(void) const;
	const std::string gid_name(void) const;

	void clear(void);

#ifdef S_ISFIFO
	const bool is_fifo_special(void) const;
#endif
#ifdef S_ISCHR
	const bool is_character_special(void) const;
#endif
#ifdef S_ISBLK
	const bool is_block_special(void) const;
#endif
#ifdef S_ISLNK
	const bool is_link(void) const;
#endif
#ifdef S_ISSOCK
	const bool is_socket(void) const;
#endif
#ifdef S_ISDOOR
	const bool is_door(void) const;
#endif
#ifdef S_ISDIR
	const bool is_directory(void) const;
#endif
#ifdef S_ISREG
	const bool is_regular_file(void) const;
#endif

#ifdef S_IRUSR
	const bool user_can_read(void) const;
#endif
#ifdef S_IWUSR
	const bool user_can_write(void) const;
#endif
#ifdef S_IXUSR
	const bool user_can_execute(void) const;
#endif
#ifdef S_IRGRP
	const bool group_can_read(void) const;
#endif
#ifdef S_IWGRP
	const bool group_can_write(void) const;
#endif
#ifdef S_IXGRP
	const bool group_can_execute(void) const;
#endif
#ifdef S_IROTH
	const bool other_can_read(void) const;
#endif
#ifdef S_IWOTH
	const bool other_can_write(void) const;
#endif
#ifdef S_IXOTH
	const bool other_can_execute(void) const;
#endif
#ifdef S_ISUID
	const bool is_set_uid(void) const;
#endif
#ifdef S_ISGID
	const bool is_set_gid(void) const;
#endif
#ifdef S_ISVTX
	const bool is_set_sticky(void) const;
#endif

private:
	std::string m_path;
	struct stat m_stat;
	std::string m_link;
	major_type m_major;
	minor_type m_minor;
	bool m_uidfound;
	bool m_gidfound;
	std::string m_uname;
	std::string m_gname;
};

/** Retrieve a list of files in a subdirectory that match a given wildcard
	filename
	
	Pathnames to the files are not included.
 */
class subdirectory : public std::vector<std::string>
{
public:
	typedef std::vector<std::string> type;

	subdirectory();
	subdirectory(const subdirectory& a_class);
	subdirectory(const std::string a_path, const std::string a_filter = "*");
	~subdirectory();

	void assign(const subdirectory& a_class);

	const type&
		path(const std::string a_path, const std::string a_filter = "*");

	subdirectory& operator=(const subdirectory& a_class);

private:
};

void rm_recursive(const std::string a_path);

/** Retrieve a list of pathnames that match a given wildcard path

	Pathnames to the files are included.  Depending on the wildcard path given,
	files listed may be in multiple directories.
 */
class directory : public std::vector<std::string>
{
public:
	typedef std::vector<std::string> type;

	directory();
	directory(const directory& a_class);
	directory(const std::string& a_str);
	~directory();

	const type& path(const std::string& a_path);

private:
};

/** Retrieve information about a filesystem */
class filesystem {
public:
	typedef uint64 size_type;

	filesystem();
	filesystem(const std::string& a_path);

	void clear(void);

	void path(const std::string& a_path);
	const std::string path(void) const;
	const size_type blocksize(void) const;
	const size_type total_blocks(void) const;
	const size_type free_blocks(void) const;
	const size_type used_blocks(void) const;
	const size_type total_inodes(void) const;
	const size_type free_inodes(void) const;
	const size_type used_inodes(void) const;

	filesystem& operator=(const filesystem& a_class);

private:
	std::string m_path;
	struct STATFS m_statfs;
};

/** A simple locking mechanism */
class simple_lock {
public:
	typedef pid_t pid_type;

	simple_lock();
	simple_lock(const std::string& a_lockfile);
	~simple_lock();

	void clear(void);

	void lockfile(const std::string& a_lockfile);
	const std::string lockfile(void) const;

	const pid_type locked_by(void) const;
	const bool is_locked(void) const;
	bool lock(void);
	void unlock(void);
private:
	std::string m_lockfile;
};

#endif
