#ifndef __estring_h__
#define __estring_h__

#include <string>

#include "asserts.h"
#include "types.h"

/* TODO: Make estring more OO

		1) Make an estring_value base class
		2) Make an estring_value class for each type that inherits from the base
		   class
		3) Revamp estring to use smart pointers to allocate the correct version of
		   estring value based on the value being stored.
 */

class estring_value
{
public:
	typedef union {
		unsigned int ui;
		int i;
		unsigned short us;
		short s;
		unsigned long ul;
		long l;
		unsigned long long ull;
		long long ll;
		char const * char_ptr;
		void * void_ptr;
		float f;
		double d;
		} value_type;

	estring_value();

	value_type value;

	void clear(void);
	estring_value& operator=(const estring_value& a_estring_value);

};

/** An extended string class.

	Estring is a derivative of std::string with extra functionality added in
	order to fascilitate (a) the conversion to strings from other types, (b)
	special formatting of those converted strings, and (c) the conversion back
	from a string to some other type.
 */
class estring : public std::string
{
public:
	/** The type from which estring inherits, std::string.  */
	typedef std::string value_type;

	/** The size type.  */
	typedef value_type::size_type size_type;
	
	/** Alignment values for formatted strings.  */
	enum alignment { 
		/** Left-justified */
		left,
		/** Right-justified */
		right,
		/** Centered */
		center
	};

	/** The type last assigned.  */
	enum set_from_type {
		/** std::string */
		type_string,
		/** unsigned int */
		type_unsigned_int,
		/** int */
		type_int,
		/** unsigned short*/
		type_unsigned_short,
		/** short */
		type_short,
		/** unsigned long */
		type_unsigned_long,
		/** long */
		type_long,
		/** unsigned long long */
		type_unsigned_long_long,
		/** long long */
		type_long_long,
		/** float */
		type_float,
		/** double */
		type_double,
		/** char* */
		type_char_ptr,
		/** void* */
		type_void_ptr,
		/** unknown */
		type_unknown
	};

	void init(void);
	void clear(void);
	void reset(void);
	estring();

	//
	// String Formatting
	//
	size_type width(const size_type a_l);
	size_type width(void) const;
	alignment align(const alignment a_alignment);
	alignment align(void) const;
	char left_fillchar(const char a_char);
	char left_fillchar(void) const;
	char right_fillchar(const char a_char);
	char right_fillchar(void) const;
	void fillchar(const char a_char);
	value_type fmt_str(void);
	value_type fmt_str(
		const size_type a_width,
		const alignment a_alignment,
		const char a_left_fill,
		const char a_right_fill
		);

	// String formatting for float and double
	size_type precision(size_type a_p);
	size_type precision(void) const;

	// Formatting for conversion from different bases
	const unsigned int base(const unsigned int a_base);
	const unsigned int base(void) const;

	// estring
	const set_from_type& get_from_type(void) const;
	const estring_value& get_from_value(void) const;
	estring(const estring& a_estr);
	estring& assign(const estring& a_estr);
	estring& operator=(const estring& a_estr);
	estring& lower(void);
	estring& upper(void);

	// char
	estring(const char a_char);
	estring& assign(const char a_char);
	estring& operator=(const char a_char);

	// string
	estring(const value_type& a_string);
	estring& assign(const value_type& a_string);
	estring& operator=(const value_type& a_string);

	// unsigned int
	estring(const unsigned int a_int);
	estring& assign(const unsigned int a_int);
	estring& operator=(const unsigned int a_int);
	operator unsigned int() const;

	// int
	estring(const int a_int);
	estring& assign(const int a_int);
	estring& operator=(const int a_int);
	operator int() const;

	// unsigned short
	estring(const unsigned short a_short);
	estring& assign(const unsigned short a_short);
	estring& operator=(const unsigned short a_short);
	operator unsigned short() const;

	// short
	estring(const short a_short);
	estring& assign(const short a_short);
	estring& operator=(const short a_short);
	operator short() const;

	// unsigned long
	estring(const unsigned long a_long);
	estring& assign(const unsigned long a_long);
	estring& operator=(const unsigned long a_long);
	operator unsigned long() const;

	// long
	estring(const long a_long);
	estring& assign(const long a_long);
	estring& operator=(const long a_long);
	operator long() const;

	// unsigned long long
	estring(const unsigned long long a_long);
	estring& assign(const unsigned long long a_long);
	estring& operator=(const unsigned long long a_long);
	operator unsigned long long() const;

	// long long
	estring(const long long a_long);
	estring& assign(const long long a_long);
	estring& operator=(const long long a_long);
	operator long long() const;

	// char*
	estring(char const * a_ptr);
	estring& assign(char const * a_ptr);
	estring& operator=(char const * a_ptr);
	operator char const *() const;

	// void*
	estring(void* const a_ptr);
	estring& assign(void* const a_ptr);
	estring& operator=(void* const a_ptr);
	operator void*() const;

	// float
	estring(const float a_float);
	estring& assign(const float a_float);
	estring& operator=(const float a_float);
	estring(const float a_float, unsigned a_precision, 
		unsigned int a_base = 10);
	estring& assign(const float a_float, unsigned a_precision, 
		unsigned int a_base = 10);
	operator float() const;

	// double
	estring(const double a_double);
	estring& assign(const double a_double);
	estring& operator=(const double a_double);
	estring(const double a_double, unsigned a_precision, 
		unsigned int a_base = 10);
	estring& assign(const double a_double, unsigned a_precision,
		unsigned int a_base = 10);
	operator double() const;

private:
	/** The current fractional number precision */
	size_type m_precision;
	/** The current numerical base */
	unsigned int m_base;
	/** The alphabet used for any base from 2 to 36 */
	static const char *m_alphabet;
	/** The length of the alphabet */
	static const size_t m_alphabet_len;
	/** The current formatting width */
	size_type m_width;
	/** The current formatting alignment */
	alignment m_alignment;
	/** The current left-hand fill character */
	char m_left_fillchar;
	/** The current right-hand fill character */
	char m_right_fillchar;
	/** The current value type */
	set_from_type m_type;
	/** The current value */
	estring_value m_value;

	template <class T>
	void T_fraction_to_strings(const T& a_t,
		value_type& a_ws, value_type& a_fs);

	template <class T>
	void T_integral_to_string(const T& a_t, value_type& a_str);

	template <class T>
	void T_string_to_integral(const value_type& a_str, T& a_t) const;

	template <class T>
	void T_string_to_signed_integral(const value_type& a_str, T& a_t) const;
	
	template <class T>
	void T_string_to_fractional(const value_type& a_str, T& a_t) const;
};

#endif
