#include "config.h"

#include <iostream>
#include <string>
#include <cstdio>
#include <cctype>
#include <string.h>

#include "asserts.h"
#include "types.h"
#include "error.h"
#include "estring.h"

//----------------------------------------------------------------------------

estring_value::estring_value()
{
	clear();
}

void estring_value::clear(void)
{
	memset(static_cast<void*>(&value), 0, sizeof(value));
}

estring_value& estring_value::operator=(const estring_value& a_estring_value)
{
	value = a_estring_value.value;

	return(*this);
}

//----------------------------------------------------------------------------

/** Alphabet used for base number conversions
 */

const char * estring::m_alphabet = "0123456789abcdefghijklmnopqrstuvwxyz";

/** The length of the alphabet used for base number conversions
 */
const size_t estring::m_alphabet_len = 36;

/** Helper member template function to convert an integral type to an estring.

	Integrals are converted to string form using a default base of 10 unless
	specified otherwise through the base() member function.  For bases greater
	than ten the letters of the alphabet are used, starting with 'a' for eleven,
	'b' for twelve, etc.

	Variables:
		- a_t - the integral type.  Usually uint8, uint16, uint32, or uint64.
		- a_str - the std::string object in which to store the string converstion.
	
	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
 */
template <class T>
void estring::T_integral_to_string(const T& a_t, value_type& a_str)
{
	T num = a_t;
	int digit;
	char ch;

	ASSERT(a_t >= 0);
	ASSERT(m_base >= 2);
	ASSERT(m_base <= m_alphabet_len);

	TRY_nomem(a_str = "");
	if (a_t == 0) {
		TRY_nomem(a_str = "0");
		return;
	}
	while (num > 0) {
		digit = num % m_base;
		ch = m_alphabet[digit];
		TRY_nomem(a_str = ch + a_str);
		num /= m_base;
	}
}

/** Helper member template function to convert a fractional type to an estring.

	Fractions are converted to string form using a default base of 10 unless
	specified otherwise through the base() member function.  For bases greater
	than ten the letters of the alphabet are used, starting with 'a' for eleven,
	'b' for twelve, etc.

	By default a precision of 1 digit to the right of the decimal point is used
	unless specified otherwise by the precision() member function.

	Variables:
		- a_t - the fractional type: float or double.
		- a_ws - the std::string object in which to store the whole part.
		- a_fs - the std::string object in which to store the fractional part.
	
	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
 */
template <class T>
void estring::T_fraction_to_strings(const T& a_t,
	value_type& a_ws, value_type& a_fs)
{
	T fraction = a_t;
	uint64 rounded;
	bool negative = false;
	uint64 whole_part;
	uint64 fractional_part;
	uint64 multiplier;
	size_type c;
	int digit;
	char ch;

	ASSERT(m_base >= 2);
	ASSERT(m_base <= m_alphabet_len);

	TRY_nomem(a_ws = "");
	TRY_nomem(a_fs = "");

	if (fraction < 0.0) {
		negative = true;
		fraction = -fraction;
	}

	multiplier = m_base;
	for (c = 0; c < m_precision; c++) {
		multiplier *= m_base;
	}

	rounded = static_cast<uint64>(fraction * multiplier);
	fractional_part = rounded % multiplier;
	whole_part = rounded / multiplier;

	if ((fractional_part % m_base) + (m_base/2) >= m_base) {
		fractional_part += m_base/2;
		if (fractional_part > (fractional_part % multiplier)) {
			whole_part++;
		}
		fractional_part %= multiplier;
	}

	if (whole_part == 0) {
		TRY_nomem(a_ws = "0");
	}
	else {
		while (whole_part > 0) {
			digit = whole_part % m_base;
			ch = m_alphabet[digit];
			TRY_nomem(a_ws = ch + a_ws);
			whole_part /= m_base;
		}
	}

	fractional_part /= m_base;
	if (fractional_part == 0) {
		TRY_nomem(a_fs = "0");
	}
	else {
		while (fractional_part > 0) {
			digit = fractional_part % m_base;
			ch = m_alphabet[digit];
			TRY_nomem(a_fs = ch +a_fs);
			fractional_part /= m_base;
		}
	}
	if (a_fs.size() > m_precision) {
		a_fs.erase(m_precision);
	}
	for (c = a_fs.size(); c < m_precision; c++) {
		TRY_nomem(a_fs += "0");
	}

	if (negative) {
		TRY_nomem(a_ws = "-" + a_ws);
	}
}

/** Helper member template function to convert a string to an integral type.

	Characters in the string are converted using a default base of 10 unless
	specified otherwise through the base() member function.  For bases greater
	than ten the letters of the alphabet will be assumed, starting with 'a' for
	eleven, 'b' for twelve, etc.
	
	Variables:
		- a_str - the std::string object from which to convert.
		- a_t - a variable of the integral type to which to convert.
	
	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
		- If a symbol is encountered in the string that is not expected then an
			invalid character or invalid base error is thrown.
		- If the type to which to convert the string is not large enough to hold
			the value of the converted string then an overflow error is thrown.
 */
template<class T>
void estring::T_string_to_integral(const value_type& a_str, T& a_t) const
{
	value_type::const_iterator stri;
	T value = 0;
	T digit = 0;
	T overflow_check_1 = 0;
	T overflow_check_2 = 0;
	value_type es;
	value_type alphabet;
	size_type idx;
	const static size_t buffer_len = 256;
	char buffer[buffer_len] = { 0 };
	unsigned int index;

	ASSERT(m_base >= 2);
	ASSERT(m_base <= m_alphabet_len);

	TRY_nomem(alphabet = m_alphabet);
	for (stri = a_str.begin(), index = 0;
		stri != a_str.end();
		++stri, ++index
		)
	{
		idx = alphabet.find(*stri);
		if (idx > m_base) {
			snprintf(buffer, buffer_len, "%u", index);

			TRY_nomem(es = "Parse error at index ");
			TRY_nomem(es += buffer);
			TRY_nomem(es += " (char '");
			TRY_nomem(es += *stri);
			TRY_nomem(es += "')");
			TRY_nomem(es += " converting \"");
			TRY_nomem(es += (*this));
			TRY_nomem(es += "\"");
			if (idx == value_type::npos) {
				TRY_nomem(es += ", invalid character");
			}
			else {
				TRY_nomem(es += ", invalid base or invalid string");
			}
			throw(ERROR(0,es));
		}

		value *= m_base;
		digit = static_cast<unsigned int>(idx);
		value += digit;

		overflow_check_1 *= m_base;
		overflow_check_1 += 1;
		if (overflow_check_1 == overflow_check_2) {
			snprintf(buffer, buffer_len, "%u", index);

			TRY_nomem(es = "Overflow error at index ");
			TRY_nomem(es += buffer);
			TRY_nomem(es += " (char '");
			TRY_nomem(es += *stri);
			TRY_nomem(es += "')");
			TRY_nomem(es += " converting \"");
			TRY_nomem(es += (*this));
			TRY_nomem(es += "\" to unsigned int");
			throw(ERROR(0,es));
		}
		overflow_check_2 = overflow_check_1;
	}
	a_t = value;
}

/** Helper member template function to convert a string to a signed integral.

	Variables:
		- a_str - the std::string to convert from.
		- a_t - a variable of the type to convert to.
	
	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
		- Any exception thrown from the called member template function
			T_string_to_integral().
 */
template <class T>
void estring::T_string_to_signed_integral(const value_type& a_str, T& a_t)
	const
{
	value_type tmp;
	T value = 0;
	bool negative = false;

	if ((value_type::size() > 0) && (this->at(0) == '-')) {
		negative = true;
		TRY_nomem(tmp = (*this).substr(1));
	}
	else {
		TRY_nomem(tmp = (*this));
	}
	T_string_to_integral(tmp,value);
	if (negative)
		value = -value;
	a_t = value;
}

/** Helper member template function to convert a string to a fractional.

	Variables:
		- a_str - the std::string to convert from.
		- a_t - a variable of the type to convert to.
	
	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
		- Any exception thrown by the member template function
			T_string_to_integral().
		- An overflow error is thrown if the wole part of the string is too large
			for the type a_t.
		- An underflow error is thrown if the fractional part of the string is too
			large for the type a_t.
 */
template <class T>
void estring::T_string_to_fractional(const value_type& a_str, T& a_t) const
{
	value_type es;
	value_type tmp;
	value_type whole_string;
	value_type fractional_string;
	size_type idx = 0;
	bool negative = false;;
	uint64 whole_part_converter = 0;
	uint64 fractional_part_converter = 0;
	uint64 fractional_check = 0;
	T whole_part = 0;
	T fractional_part = 0;
	T value =0;
	unsigned int divisor = 0;
	unsigned int c = 0;

	ASSERT(m_base >= 2);
	ASSERT(m_base <= m_alphabet_len);

	if ((value_type::size() > 0) && (this->at(0) == '-')) {
		negative = true;
		TRY_nomem(tmp = (*this).substr(1));
	}
	else {
		TRY_nomem(tmp = (*this));
	}

	idx = tmp.find('.');
	if (idx != value_type::npos) {
		TRY_nomem(whole_string = tmp.substr(0,idx));
		TRY_nomem(fractional_string = tmp.substr(idx+1));
	}
	else {
		TRY_nomem(whole_string = tmp);
		TRY_nomem(fractional_string = "");
	}

	TRY_nomem(es = "Could not convert whole part of estring \"");
	TRY_nomem(es += a_str);
	TRY_nomem(es += "\"");
	TRY(T_string_to_integral(whole_string, whole_part_converter),es);

	TRY_nomem(es = "Could not convert fractional part of estring \"");
	TRY_nomem(es += a_str);
	TRY_nomem(es += "\"");
	TRY(T_string_to_integral(fractional_string, fractional_part_converter),es);

	divisor = 1;
	for (c = 0; c < fractional_string.size(); c++)
		divisor *= m_base;

	whole_part = static_cast<T>(whole_part_converter);
	if (static_cast<uint64>(whole_part) != whole_part_converter) {
		TRY_nomem(es = "Overflow error converting whole part of estring \"");
		TRY_nomem(es += a_str);
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}
	fractional_part = static_cast<T>(fractional_part_converter)/divisor;
	fractional_check = static_cast<uint64>(fractional_part*divisor);
	if (fractional_check != fractional_part_converter) {
		TRY_nomem(es = "Underflow error converting fractional part of estring \"");
		TRY_nomem(es += a_str);
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}

	value = whole_part + fractional_part;

	if (negative) {
		value = -value;
	}
	a_t = value;
}

//---------

/** Initialize the estring object.

	Defaults:
		- A precision of one digit to the right of the decimal.
		- A base of 10.
		- A width of 5 characters for formatted printing.
		- A space as the fill character for padding the left of a string.
		- A space as the fill character for padding the right of a string.
		- Initialize the conversion type to type_unknown.
 */
void estring::init(void)
{
	m_precision = 1;
	m_base = 10;
	m_width = 5;
	m_alignment = left;
	m_left_fillchar = ' ';
	m_right_fillchar = ' ';
	m_type = type_unknown;
}

/** Erase the string value.
 */
void estring::clear(void)
{
	value_type::erase();
}

/** Erase and reinitialize.
 */
void estring::reset(void)
{
	clear();
	init();
}

//---------

/** Default constructor
 */
estring::estring()
{
	init();
}

//---------

/** Set the width of a formatted string.

	Variables:
		- a_l - new width to use for formatting strings.
	
	Returns:
		- size_type - the last width.
 */
estring::size_type estring::width(const size_type a_l)
{
	size_type old;

	old = m_width;
	m_width = a_l;

	return(old);
}

/** Retrieve the set width for formatted strings.
	
	Returns:
		- size_type - the current width.
 */
estring::size_type estring::width(void) const
{
	return(m_width);
}

/** Set the alignment used for formatted strings.

	Vaiables:
		- a_alignment - the type of alignment to use for formatting strings.
 */
estring::alignment estring::align(const alignment a_alignment)
{
	alignment old;

	old = m_alignment;
	m_alignment = a_alignment;

	return(old);
}

/** Retrieve the set alignment for formatted strings.
	
	Returns:
		- enum alignment - the current alignment.
 */
estring::alignment estring::align(void) const
{
	return(m_alignment);
}

/** Set the precision used in converting to/from fractional types.

	Variables:
		- a_p - the number of digits to use to the right of the decimal point when
			converting fractional types.
	
	Returns:
		- size_type - the last precision.

 */
estring::size_type estring::precision(size_type a_p)
{
	size_type old;

	old = m_precision;
	m_precision = a_p;

	if (m_type == type_float)
		assign(m_value.value.f);
	else if (m_type == type_double)
		assign(m_value.value.d);

	return(old);
}

/** Retrieve the set precision used in fractional conversions.
	
	Returns:
		- size_type - the current precision.
 */
estring::size_type estring::precision(void) const
{
	return(m_precision);
}

/** Set the base used in numeric conversions.

	Variables:
		- a_base - The base to use in converting numbers to/from strings.

	Return:
		- const unsigned int - the last base.

	Exceptions:
		- An err_nomem is thrown for memory allocation failure.
		- An invalid base is thrown if the base is smaller than 2 (binary).
		- A base too large is thrown if the base is larger than the alphabet of
			usable symbols.
 */
const unsigned int estring::base(const unsigned int a_base)
{
	unsigned int old = m_base;
	value_type es;
	char str[255] = { 0 };

	if (a_base < 2) {
		sprintf(str, "%u", a_base);
		TRY_nomem(es = "Invalid base: ");
		TRY_nomem(es += str);
		throw(ERROR(0,es));
	}
	if (a_base > m_alphabet_len) {
		sprintf(str, "%u", a_base);
		TRY_nomem(es = "Base too large: ");
		TRY_nomem(es += str);
		throw(ERROR(0,es));
	}
	m_base = a_base;

	if (m_type == type_unsigned_int)
		assign(m_value.value.ui);
	if (m_type == type_int)
		assign(m_value.value.i);
	if (m_type == type_unsigned_short)
		assign(m_value.value.us);
	if (m_type == type_short)
		assign(m_value.value.s);
	if (m_type == type_unsigned_long)
		assign(m_value.value.ul);
	if (m_type == type_long)
		assign(m_value.value.l);
	if (m_type == type_unsigned_long_long)
		assign(m_value.value.ull);
	if (m_type == type_long_long)
		assign(m_value.value.ll);
	if (m_type == type_float)
		assign(m_value.value.f);
	if (m_type == type_double)
		assign(m_value.value.d);

	return(old);
}

/** Retrieve the base used in numeric conversions.

	Returns:
		- const unsigned int - the current numeric base.
 */
const unsigned int estring::base(void) const
{
	return(m_base);
}

/** Set the fill character used to padd the left side of a formatted string.

	Variables:
		- a_char - the character to use as the fill character.
	
	Returns:
		- char - the last fill character.

 */
char estring::left_fillchar(const char a_char)
{
	char old;

	old = m_left_fillchar;
	m_left_fillchar = a_char;

	return(old);
}

/** Retrieve the fill character used to padd the left side of a formatted
		string.
	
	Returns:
		- char - the current fill character.

 */
char estring::left_fillchar(void) const
{
	return(m_left_fillchar);
}

/** Set the fill character used to padd the right side of a formatted string.

	Variables:
		- a_char - the character to use as the fill character.
	
	Returns:
		- char - the last fill character.

 */
char estring::right_fillchar(const char a_char)
{
	char old;

	old = m_right_fillchar;
	m_right_fillchar = a_char;

	return(old);
}

/** Retrieve the fill character used to padd the right side of a formatted
		string.
	
	Returns:
		- char - the current fill character.

 */
char estring::right_fillchar(void) const
{
	return(m_right_fillchar);
}

/** Set the fill character used for padding both the left and right side of a
		formatted string.
	
	Variables:
		- a_char - the character to use as the fill character.
 */
void estring::fillchar(const char a_char)
{
	left_fillchar(a_char);
	right_fillchar(a_char);
}

//---------
		
/** Generate a formatted string.

	Returns:
		- value_type - a formatted rendition of the current string.

	If the string being printed is wider than the assigned width then as many as
	three periods are used ("...") to denote that the contents of the string
	have been truncated.  For strings that use left alignment these three
	periods are printed on the right-hand side of the string, while showing as
	many characters on the left as possible beginning with the left-most
	character.  For strings that use right alignment these three periods are
	printed on the left-hand side of the string, while showing as many
	characters on the right as possible includingn the right-most character.
	For strings that are center-aligned as many characters on both the left and
	the right are printed, including the left-most and the right-most
	characters, while characters in the center of the string are replaced with
	the three periods.
 */
estring::value_type estring::fmt_str(void)
{
	std::string str;
	std::string lstr;
	std::string rstr;
	std::string::size_type c = 0;
	std::string::size_type c_max = 0;
	std::string::size_type offset = 0;
	std::string::size_type length = 0;
	std::string::size_type l_offset = 0;
	std::string::size_type r_offset = 0;
	bool last = true;

	TRY_nomem(str = "");
	TRY_nomem(lstr = "");
	TRY_nomem(rstr = "");

	if (std::string::size() > m_width) {
		if (m_alignment == left) {
			if (m_width >= 3)
				length = m_width-3;
			else
				length = 0;
			TRY_nomem(str = std::string::substr(0,length) + "...");
			TRY_nomem(str = str.substr(0,m_width));
		}
		else if (m_alignment == right) {
			if (m_width >= 3)
				length = m_width-3;
			else
				length = 0;

			offset = std::string::size()-length;
			TRY_nomem(str = "..." + std::string::substr(offset,length));
			TRY_nomem(str = str.substr(str.size()-m_width,m_width));
		}
		else {
			if (m_width < 4) {
				TRY_nomem(str = static_cast<std::string>("....").substr(0,m_width));
				return(str);
			}

			c_max = m_width-3;
			r_offset = std::string::size()-1;
			for (c = 0; c < c_max; c++) {
				if (last) {
					TRY_nomem(lstr += (*this)[l_offset++]);
				}
				else {
					TRY_nomem(rstr = (*this)[r_offset--] + rstr);
				}
				last = !last;
			}

			TRY_nomem(str = lstr);
			TRY_nomem(str += "...");
			TRY_nomem(str += rstr);
		}
		return(str);
	}

	TRY_nomem(str = (*this));
	c_max = m_width - std::string::size();
	for (c = 0; c < c_max; c++) {
		if (m_alignment == right) {
			TRY_nomem(str = m_left_fillchar + str);
		}
		else if (m_alignment == left) {
			TRY_nomem(str += m_right_fillchar);
		}
		else {
			if (last) {
				TRY_nomem(str += m_right_fillchar);
			}
			else {
				TRY_nomem(str = m_left_fillchar + str);
			}
			last = !last;
		}
	}

	return(str);
}

/** Set all the formatting options.

	Variables:
		- a_width - the width of the formatted string.
		- a_alignment - the alignment used to format the string.
		- a_left_fill - the character used for left-hand padding.
		- a_right_fill - the character used for right-hand padding.
	
	Returns:
		- value_type - a formatted rendition of the current string.

 */
estring::value_type estring::fmt_str(
	const size_type a_width,
	const alignment a_alignment,
	const char a_left_fill,
	const char a_right_fill
	)
{
	value_type str;

	width(a_width);
	align(a_alignment);
	left_fillchar(a_left_fill);
	right_fillchar(a_right_fill);

	str = fmt_str();

	return(str);
}

//---------

/** Retrieve the type of value being held by this estring.

	Returns:
		- set_from_type& - the enumeration of the type of value currently assigned.
 */
const estring::set_from_type& estring::get_from_type(void) const
{
	return(m_type);
}

/** Retrieve the typeless_value being held by this estring.

	Returns:
		- typeless_value& - the union that holds the current value.
	
 */
const estring_value& estring::get_from_value(void) const
{
	return(m_value);
}

/** Copy constructor for estring objects.
	
	Variables:
		- a_estr - the estring object to copy.

	Exceptions:
		- Anything thrown by estring::assign(const estring& a_estr).
 */
estring::estring(const estring& a_estr)
{
	init();
	assign(a_estr);
}

/** Assignment for estring objects.
 	
	Variables:
		- a_estr - the estring object to copy.

	Returns:
		- estring& - a reference to self.
	
	Exceptions:
		- An err_nomem is thrown if memory allocation fails.
		- It is possible that an "Invalid base" or "Base too large" may be thrown
			if the estring object is corrupted, but this should never happen.
*/
estring& estring::assign(const estring& a_estr)
{
	TRY_nomem(std::string::assign(a_estr));
	m_precision = a_estr.m_precision;
	m_width = a_estr.m_width;
	m_alignment = a_estr.m_alignment;
	m_left_fillchar = a_estr.m_left_fillchar;
	m_right_fillchar = a_estr.m_right_fillchar;
	m_type = a_estr.m_type;
	m_value = a_estr.m_value;

	return(*this);
}

/** Assignment operator for estring objects.

	Variables:
		- a_estr - the estring object to copy.
	
	Returns:
		- estring& - a reference to self.
	
	Exceptions:
		- Anything thrown by estring::assign(const estring& a_estr).
 */
estring& estring::operator=(const estring& a_estr)
{
	assign(a_estr);

	return(*this);
}

/** Convert all characters to lowercase.

	Returns:
		- estring& - a reference to self.
	
 */
estring& estring::lower(void)
{
	std::string::iterator si;

	for (si = (*this).begin(); si != (*this).end(); ++si) {
		(*si) = tolower((*si));
	}

	return(*this);
}

/** Convert all characters to uppercase.

	Returns:
		- estring& - a reference to self.
	
 */
estring& estring::upper(void)
{
	std::string::iterator si;

	for (si = (*this).begin(); si != (*this).end(); ++si) {
		(*si) = toupper((*si));
	}

	return(*this);
}

//---------

/** Copy constructor for chars.
	
	Exceptions:
		- Anything thrown by estring::assign(const char& a_char).
 */
estring::estring(const char a_char)
{
	init();
	assign(a_char);
}

/** Assignment for chars.
	
	Variables:
		- a_char - the character to assign.
	
	Returns:
		- estring& - a reference to self.
	
	Exceptions:
		- An err_nomem is thrown if memory allocation fails.
 */
estring& estring::assign(const char a_char)
{
	std::string s;

	TRY_nomem(s = a_char);
	TRY_nomem(std::string::assign(s));
	m_type = type_string;
	
	return(*this);
}

/** Assignment operator for chars
	
	Variables:
		- a_char - the character to assign.
	
	Returns:
		- estring& - a reference to self.
	
	Exceptions:
		- Anything thrown by estring::assign(const char& a_char).
 */
estring& estring::operator=(const char a_char)
{
	assign(a_char);

	return(*this);
}

//---------

/** Copy constructor for std::string objects.

	Variables:
		- a_string - the std::string object to copy.

	Exceptions:
		- Anything thrown by estring::assign(const value_type& a_string).
 */
estring::estring(const value_type& a_string)
{
	init();
	assign(a_string);
}

/** Assignment for std::string objects.
	
	Variables:
		- a_string - the std::string object to assign.
	
	Returns:
		- estring& - a reference to self.

	Exceptions:
		- An err_nomem is thrown if memory allocation fails.
 */
estring& estring::assign(const value_type& a_string)
{
	TRY_nomem(std::string::assign(a_string));
	m_type = type_string;

	return(*this);
}

/** Assignment operator for std::string objects.

	Variables:
		- a_string - the std::string object to copy.
	
	Returns:
		- estring& - a reference to self.
	
	Exceptions:
		- An err_nomem is thrown if memory allocation fails.
 */
estring& estring::operator=(const value_type& a_string)
{
	assign(a_string);

	return(*this);
}

//---------

/** Copy constructor for unsigned ints.

	Variables:
		- a_int - unsigned int to copy.
	
	Exceptions:
		- Anything thrown by estring::assign(const unsigned int& a_int).
 */
estring::estring(const unsigned int a_int)
{
	init();
	assign(a_int);
}

/** Assignment for unsigned ints.

	Variables:
		- a_int - unsigned int to assign.

	Returns:
		- estring& - a reference to self.

	Exceptions:
		- err_nomem on memory allocation failure.
		- Anything thrown by 
			T_integral_to_string(const T& a_t, value_type& a_str).
 */
estring& estring::assign(const unsigned int a_int)
{
	std::string s;

	TRY(T_integral_to_string(a_int,s),
		"Could not convert unsigned int to string");
	TRY_nomem(std::string::assign(s));
	m_type = type_unsigned_int;
	m_value.clear();
	m_value.value.ui = a_int;

	return(*this);
}

/** Assignment operator for unsigned ints.
 */
estring& estring::operator=(const unsigned int a_int)
{
	assign(a_int);

	return(*this);
}

/** Implicit conversion operator to an unsigned int.
 */
estring::operator unsigned int() const
{
	unsigned int value = 0;

	TRY(T_string_to_integral((*this),value),
		"Cannot convert string to unsigned int");

	return(value);
}

//---------

estring::estring(const int a_int)
{
	init();
	assign(a_int);
}

estring& estring::assign(const int a_int)
{
	if (a_int < 0) {
		TRY(assign(static_cast<unsigned int>(-a_int)),
			"Coud not convert signed int to string");
		TRY_nomem(std::string::insert(0,"-"));
	}
	else {
		TRY(assign(static_cast<unsigned int>(a_int)),
			"Could not convert signed int to string");
	}
	m_type = type_int;
	m_value.clear();
	m_value.value.i = a_int;

	return(*this);
}

estring& estring::operator=(const int a_int)
{
	assign(a_int);

	return(*this);
}

estring::operator int() const
{
	int value = 0;

	TRY(T_string_to_signed_integral((*this),value),
		"Cannot convert string to signed int");

	return(value);
}

//---------

estring::estring(const unsigned short a_short)
{
	init();
	assign(a_short);
}

estring& estring::assign(const unsigned short a_short)
{
	std::string s;

	TRY(T_integral_to_string(a_short,s),
		"Could not convert unsigned short to string");
	TRY_nomem(std::string::assign(s));
	m_type = type_unsigned_short;
	m_value.clear();
	m_value.value.us = a_short;

	return(*this);
}

estring& estring::operator=(const unsigned short a_short)
{
	assign(a_short);

	return(*this);
}

estring::operator unsigned short() const
{
	unsigned short value = 0;

	TRY(T_string_to_integral((*this),value),
		"Cannot convert string to unsigned short");

	return(value);
}

//---------

estring::estring(const short a_short)
{
	init();
	assign(a_short);
}

estring& estring::assign(const short a_short)
{
	if (a_short < 0) {
		TRY(assign(static_cast<unsigned short>(-a_short)),
			"Could not convert signed short to string");
		TRY_nomem(std::string::insert(0,"-"));
	}
	else {
		TRY(assign(static_cast<unsigned short>(a_short)),
			"Could not convert signed short to string");
	}
	m_type = type_short;
	m_value.clear();
	m_value.value.s = a_short;

	return(*this);
}

estring& estring::operator=(const short a_short)
{
	assign(a_short);

	return(*this);
}

estring::operator short() const
{
	short value = 0;

	TRY(T_string_to_signed_integral((*this),value),
		"Cannot convert string to signed short");

	return(value);
}

//---------

estring::estring(const unsigned long a_long)
{
	init();
	assign(a_long);
}

estring& estring::assign(const unsigned long a_long)
{
	std::string s;

	TRY(T_integral_to_string(a_long,s),
		"Could not convert unsigned long to string");
	TRY_nomem(std::string::assign(s));
	m_type = type_unsigned_long;
	m_value.clear();
	m_value.value.ul = a_long;

	return(*this);
}

estring& estring::operator=(const unsigned long a_long)
{
	assign(a_long);

	return(*this);
}

estring::operator unsigned long() const
{
	unsigned long value = 0;

	TRY(T_string_to_integral((*this),value),
		"Cannot convert string to unsigned long");

	return(value);
}

//---------

estring::estring(const long a_long)
{
	init();
	assign(a_long);
}

estring& estring::assign(const long a_long)
{
	if (a_long < 0) {
		TRY(assign(static_cast<unsigned long>(-a_long)),
			"Could not convert signed long to string");
		TRY_nomem(std::string::insert(0,"-"));
	}
	else {
		TRY(assign(static_cast<unsigned long>(a_long)),
			"Could not convert signed long to string");
	}
	m_type = type_long;
	m_value.clear();
	m_value.value.l = a_long;

	return(*this);
}

estring& estring::operator=(const long a_long)
{
	assign(a_long);

	return(*this);
}

estring::operator long() const
{
	long value = 0;

	TRY(T_string_to_signed_integral((*this),value),
		"Cannot convert string to signed long");

	return(value);
}

//---------

estring::estring(const unsigned long long a_long)
{
	init();
	assign(a_long);
}

estring& estring::assign(const unsigned long long a_long)
{
	std::string s;

	TRY(T_integral_to_string(a_long,s),
		"Could not convert unsigned long long to string");
	TRY_nomem(std::string::assign(s));
	m_type = type_unsigned_long_long;
	m_value.clear();
	m_value.value.ull = a_long;

	return(*this);
}

estring& estring::operator=(const unsigned long long a_long)
{
	assign(a_long);

	return(*this);
}

estring::operator unsigned long long() const
{
	unsigned long long value = 0;

	TRY(T_string_to_integral((*this),value),
		"Cannot convert string to unsigned long long");

	return(value);
}

//---------

estring::estring(const long long a_long)
{
	init();
	assign(a_long);
}

estring& estring::assign(const long long a_long)
{
	if (a_long < 0) {
		TRY(assign(static_cast<unsigned long long>(-a_long)),
			"Could not convert unsigned long long to string");
		TRY_nomem(insert(0,"-"));
	}
	else {
		TRY(assign(static_cast<unsigned long long>(a_long)),
			"Could not convert unsigned long long to string");
	}
	m_type = type_long_long;
	m_value.clear();
	m_value.value.ll = a_long;

	return(*this);
}

estring& estring::operator=(const long long a_long)
{
	assign(a_long);

	return(*this);
}

estring::operator long long() const
{
	long long value = 0;

	TRY(T_string_to_signed_integral((*this),value),
		"Cannot convert string to signed long long");

	return(value);
}

//---------

estring::estring(char const * a_ptr)
{
	init();
	assign(a_ptr);
}

estring& estring::assign(char const * a_ptr)
{
	std::string str;

	TRY_nomem(str = a_ptr);
	TRY_nomem(assign(str));
	m_type = type_char_ptr;
	m_value.clear();
	m_value.value.char_ptr = a_ptr;

	return(*this);
}

estring& estring::operator=(char const * a_ptr)
{
	assign(a_ptr);

	return(*this);
}

estring::operator char const *()const
{
	char const * value = 0;

	if ((m_type != type_char_ptr) && (m_type != type_void_ptr)) {
		throw(ERROR(0,"Value type is not a pointer"));
	}

	value = m_value.value.char_ptr;

	return(value);
}

//---------

estring::estring(void* const a_ptr)
{
	init();
	assign(a_ptr);
}

estring& estring::assign(void* const a_ptr)
{
	static const size_t buffer_len = 32;
	char buffer[buffer_len] = { 0 };
	
	snprintf(buffer, buffer_len, "%p", a_ptr);
	TRY_nomem(std::string::assign(buffer));
	m_type = type_void_ptr;
	m_value.clear();
	m_value.value.void_ptr = a_ptr;

	return(*this);
}

estring& estring::operator=(void* const a_ptr)
{
	assign(a_ptr);

	return(*this);
}

estring::operator void*()const
{
	void * value = 0;

	if ((m_type != type_void_ptr) && (m_type != type_char_ptr)) {
		throw(ERROR(0,"Value type is not a pointer"));
	}

	value = m_value.value.void_ptr;

	return(value);
}

//---------

estring::estring(const float a_float)
{
	init();
	assign(a_float);
}

estring& estring::assign(const float a_float)
{
	std::string ws;
	std::string fs;
	std::string s;

	TRY(T_fraction_to_strings(a_float,ws,fs),
		"Cannot convert float to string");
	TRY_nomem(s = ws);
	if (fs.size() > 0) {
		TRY_nomem(s += ".");
		TRY_nomem(s += fs);
	}
	TRY_nomem(std::string::assign(s));
	m_type = type_float;
	m_value.clear();
	m_value.value.f = a_float;

	return(*this);
}

estring& estring::operator=(const float a_float)
{
	assign(a_float);

	return(*this);
}

estring::estring(const float a_float, unsigned a_precision,
	unsigned int a_base)
{
	init();
	assign(a_float, a_precision, a_base);
}

estring& estring::assign(const float a_float, unsigned a_precision,
	unsigned int a_base)
{
	std::string ws;
	std::string fs;
	std::string s;

	precision(a_precision);
	base(a_base);
	assign(a_float);

	return(*this);
}

estring::operator float() const
{
	float value = 0.0;

	TRY(T_string_to_fractional((*this), value),
		"Cannot convert string to float");

	return(value);
}

//---------

estring::estring(const double a_double)
{
	init();
	assign(a_double);
}

estring& estring::assign(const double a_double)
{
	std::string ws;
	std::string fs;
	std::string s;

	TRY(T_fraction_to_strings(a_double,ws,fs),
		"Cannot convert double to string");
	TRY_nomem(s = ws);
	if (fs.size() > 0) {
		TRY_nomem(s += ".");
		TRY_nomem(s += fs);
	}
	TRY_nomem(std::string::assign(s));
	m_type = type_double;
	m_value.clear();
	m_value.value.d = a_double;

	return(*this);
}

estring& estring::operator=(const double a_double)
{
	assign(a_double);

	return(*this);
}

estring::estring(const double a_double, unsigned a_precision,
	unsigned int a_base)
{
	init();
	assign(a_double, a_precision, a_base);
}

estring& estring::assign(const double a_double, unsigned a_precision,
	unsigned int a_base)
{
	std::string ws;
	std::string fs;
	std::string s;

	precision(a_precision);
	assign(a_base);
	assign(a_double);

	return(*this);
}

estring::operator double() const
{
	double value = 0.0;

	TRY(T_string_to_fractional((*this), value),
		"Cannot convert string to double");

	return(value);
}

