#include "config.h"

#include <iostream>
#include <fstream>
#include <string>
#include <cassert>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <errno.h>

#include "asserts.h"
#include "types.h"
#include "error.h"
#include "estring.h"
#include "fs.h"
#include "tstamp.h"
#include "rconfig.h"
#include "timer.h"

#include "logger.h"

// #define ERR_OUT(e)	std::cerr << e
#define ERR_OUT(e)

#define TEST(code,test) \
	thrown = false; \
	try { \
		code; \
	} \
	catch(error e) { \
		ERR_OUT(e); \
		thrown = true; \
	} \
	catch(...) { \
		assert(0); \
	}


bool make_dir(const std::string& path)
{
	bool thrown;

	thrown = false;
	try {
		mk_dir(path);
	}
	catch(...) {
		thrown = true;
	}

	return(thrown == false);
}

void cleanup(void)
{
	assert(system("rm -fr ./test-logger.dir") == 0);
}

void setup(void)
{
	std::ofstream out;

	assert(make_dir("./test-logger.dir"));
	assert(make_dir("./test-logger.dir/log.dir"));
	assert(make_dir("./test-logger.dir/vault.dir"));

	out.open("./test-logger.dir/rvm.conf");
	assert(out.is_open());
	out << "log-dir ./test-logger.dir/log.dir" << std::endl;
	out << "vault ./test-logger.dir/vault.dir" << std::endl;
	out << "<job>" << std::endl;
	out << "	archive-path pathname" << std::endl;
	out << "	rsync-connection-type local" << std::endl;
	out << "	path /var/spool" << std::endl;
	out << "</job>" << std::endl;
	out.close();
}

const bool same(std::string s1, std::string s2)
{
	s1[11] = 'x'; s2[11] = 'x';
	s1[12] = 'x'; s2[12] = 'x';
	s1[14] = 'x'; s2[14] = 'x';
	s1[15] = 'x'; s2[15] = 'x';
	s1[17] = 'x'; s2[17] = 'x';
	s1[18] = 'x'; s2[18] = 'x';

	if (s1 == s2) {
		return(true);
	}
	return(false);
}

void test(void)
{
	char const * argv[256] = { 0 };
	int argc = 0;
	bool thrown;

	thrown = false;
	TEST(
		logger.init(),
		assert(e.num() == 0);
		assert(!e.internal());
		assert(e.size() == 1);
		assert(e[0].what() == "Configuration manager is not initialized");
		);
	assert(thrown);

	argv[argc++] = "<program>";
	argv[argc++] = "--archive";
	config.default_file("./test-logger.dir/rvm.conf");
	config.init(argc, argv);

	logger.clear();
	TEST(logger.init(),);
	assert(!thrown);

	logger.write("Testing... ");
	logger.write("1... ");
	logger.write("2... ");
	logger.write("3... ");
	logger.write("\n");
	logger.write("Testing...\n");
	logger.write("1...\n",1);
	logger.write("2...\n",2);
	logger.write("3...\n",3);
	logger.write("Testing...\n1...\n2...\n3...\n",1);
	logger.write("PID is ");
	logger.write(estring(static_cast<unsigned long>(pid())));
	logger.write("\n");
	logger.clear();

	std::ifstream in;
	std::string filename;
	std::string str;
	char line[1024];

	filename = "./test-logger.dir/log.dir/";
	filename += config.timestamp().str();
	filename += ".log";
	in.open(filename.c_str());
	assert(in.is_open());
	
	in.get(line,1024);
	str = stamp(pid());
	str += "Rsync Vault Manager - ";
	str += estring(VERSION);
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "Testing... 1... 2... 3... ";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "Testing...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "  1...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "    2...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "      3...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "  Testing...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "  1...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "  2...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "  3...";
	assert(same(str,line));

	in.ignore();

	in.get(line,1024);
	str = stamp(pid());
	str += "PID is ";
	str += estring(static_cast<unsigned long>(pid()));
	assert(same(str,line));

	in.close();
}

int main(int argc, char const * argv[])
{
	cleanup();
	setup();
	try {
		test();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	cleanup();
	return(0);
}

