#include "config.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <iostream>
#include <fstream>
#include <string>
#include <cassert>
#include <cerrno>

#include "asserts.h"
#include "error.h"

void test1(void)
{
	error_instance ei;

	assert(ei.what() == "");
#ifdef __GNUC__
	assert(ei.where() == "");
#endif
	assert(ei.file() == "");
	assert(ei.line() == 0);
}

void test2(void)
{
	error e(0);

	e = ERROR(0,"Simulated error instance");
	assert(e.num() == 0);
	assert(!e.internal());
	assert(e.size() == 1);
	assert(e[0].what() == "Simulated error instance");

	e = INTERNAL_ERROR(0,"Simulated internal error instance");
	assert(e.num() == 0);
	assert(e.internal());
	assert(e.size() == 1);
	assert(e[0].what() == "Simulated internal error instance");
}

void test3(void)
{
	error e(0);

	e = ERROR(1,"Simulated error instance");
	assert(e.num() == 1);
	assert(!e.internal());
	assert(e.size() == 1);
	assert(e[0].what() == "Simulated error instance");

	e = INTERNAL_ERROR(0,"Simulated internal error instance");
	assert(e.num() == 0);
	assert(e.internal());
	assert(e.size() == 1);
	assert(e[0].what() == "Simulated internal error instance");

	e = INTERNAL_ERROR(1,"Simulated internal error instance");
	assert(e.num() == 1);
	assert(e.internal());
	assert(e.size() == 1);
	assert(e[0].what() == "Simulated internal error instance");
}

void test4b(void)
{
	throw(ERROR(1,"Simulated error for test4b"));
}

void test4a(void)
{
	try {
		test4b();
	}
	catch(error e) {
		e.push_back(ERROR_INSTANCE("Failure in test4a"));
		throw(e);
	}
}

void test4(void)
{
	bool thrown = false;

	try {
		test4a();
	}
	catch(error e) {
		e.push_back(ERROR_INSTANCE("Failure in test4"));
		assert(e.num() == 1);
		assert(!e.internal());
		assert(e.size() == 3);
		assert(e[0].what() == "Simulated error for test4b");
		assert(e[1].what() == "Failure in test4a");
		assert(e[2].what() == "Failure in test4");
		thrown = true;
	}

	assert(thrown);
}

void test5b(void)
{
	throw(INTERNAL_ERROR(1,"Simulated error for test5b"));
}

void test5a(void)
{
	try {
		test5b();
	}
	catch(error e) {
		e.push_back(ERROR_INSTANCE("Failure in test5a"));
		throw(e);
	}
}

void test5(void)
{
	bool thrown = false;

	try {
		test5a();
	}
	catch(error e) {
		e.push_back(ERROR_INSTANCE("Failure in test5"));
		assert(e.num() == 1);
		assert(e.internal());
		assert(e.size() == 3);
		assert(e[0].what() == "Simulated error for test5b");
		assert(e[1].what() == "Failure in test5a");
		assert(e[2].what() == "Failure in test5");
		thrown = true;
	}

	assert(thrown);
}

void test6(void)
{
	error e1(0), e2(0);

	e1.push_back(ERROR_INSTANCE("Testing:"));
	e1.push_back(ERROR_INSTANCE("One..."));
	e1.push_back(ERROR_INSTANCE("Two..."));
	e1.push_back(ERROR_INSTANCE("Three..."));
	
	assert(e1.size() == 4);
	assert(e1[0].what() == "Testing:");
	assert(e1[1].what() == "One...");
	assert(e1[2].what() == "Two...");
	assert(e1[3].what() == "Three...");

	e2 = e1;

	assert(e2.size() == 4);
	assert(e2[0].what() == "Testing:");
	assert(e2[1].what() == "One...");
	assert(e2[2].what() == "Two...");
	assert(e2[3].what() == "Three...");
}

#ifndef ASSUME_STL_MEMORY_EXCEPTION
void test7a(void)
{
	std::string str = "a";

	while (true) {
		TRY(str += str,"Could not append to string");
	}
}

void test7(void)
{
	bool thrown = false;
	bool identified = false;

	try {
		test7a();
	}
	catch(error e) {
		// std::cerr << e;
		thrown = true;
		identified = true;
	}
	catch(...) {
		// std::cerr << err_unknown;
		assert(errno == ENOMEM);
		thrown = true;
	}

	assert(thrown);
	assert(identified);
}
#endif

void test8(void)
{
	error e(err_nomem);

	assert(e.num() == ENOMEM);
	assert(e.size() == 1);
	assert(e[0].what() == "Out of memory");
}

int main(int argc, char const * argv[])
{
	try {
		test1();
		test2();
		test3();
		test4();
		test5();
		test6();
#ifndef ASSUME_STL_MEMORY_EXCEPTION
#ifdef HAVE_EXCEPTION_ON_ALLOC_FAILURE
		test7();
#endif
#endif
		test8();
	}
	catch(error e) {
		std::cerr << e;
		assert(0);
	}
	catch(...) {
		std::cerr << err_unknown;
		assert(0);
	}
	return(0);
}
