#ifndef __reporter_h__
#define __reporter_h__

#include <string>
#include <vector>

#include "asserts.h"
#include "types.h"
#include "timer.h"
#include "table.h"

/** Vault stats report */
class vault_stats_report
{
public:
	vault_stats_report();
	vault_stats_report(const vault_stats_report& a_class);
	vault_stats_report(
		const std::string& a_message,
		const filesystem& a_class
		);
	~vault_stats_report();

	void clear(void);

	void assign(
		const vault_stats_report& a_class
		);
	void assign(
		const std::string& a_message,
		const filesystem& a_class
		);
	void assign(
		const std::string& a_message,
		const uint64 a_total_blocks,
		const uint64 a_free_blocks,
		const uint64 a_total_inodes,
		const uint64 a_free_inodes
		);
	void assign(
		const std::string& a_message,
		const std::string& a_time,
		const uint64 a_total_blocks,
		const uint64 a_free_blocks,
		const uint64 a_total_inodes,
		const uint64 a_free_inodes
		);

	const std::string& time(void) const;
	const std::string& message(void) const;
	const uint64 total_blocks(void) const;
	const uint64 free_blocks(void) const;
	const uint64 total_inodes(void) const;
	const uint64 free_inodes(void) const;

	vault_stats_report& operator=(const vault_stats_report& a_class);

private:
	std::string m_time;
	std::string m_message;
	uint64 m_total_blocks;
	uint64 m_free_blocks;
	uint64 m_total_inodes;
	uint64 m_free_inodes;
};

/** Vault selection and usage report */
class vault_report
{
public:
	vault_report();
	~vault_report();

	void clear(void);

	void add_report(const vault_stats_report& a_class);
	
	void write_report(std::ostream& a_out);

	void format_synopsis(table& a_table);
private:
	std::vector<vault_stats_report> m_reports;
};

/** Job report for an rsync run on a single path */
class job_path_report
{
public:
	job_path_report();
	job_path_report(const job_path_report& a_class);
	job_path_report(
		const std::string a_source,
		const timer a_time,
		const uint16 a_exit_code,
		const uint16 a_signal_num,
		const rsync_behavior::value_type a_behavior,
		const std::string a_error_message
		);
	~job_path_report();

	void clear(void);

	void assign(const job_path_report& a_class);
	void assign(
		const std::string a_source,
		const timer a_time,
		const uint16 a_exit_code,
		const uint16 a_signal_num,
		const rsync_behavior::value_type a_behavior,
		const std::string a_error_message
		);
	
	const bool status(void) const;
	void source(const std::string& a_class);
	const std::string& source(void) const;
	void time(const timer& a_class);
	const timer& time(void) const;
	void exit_code(const int a_exit_code);
	const int exit_code(void) const;
	void signal_num(const int a_signal_num);
	const int signal_num(void) const;
	const rsync_behavior::value_type behavior(void) const;
	void error_msg(const std::string& a_class);
	const std::string& error_msg(void) const;

	job_path_report& operator=(const job_path_report& a_class);

private:
	std::string m_source;
	timer m_time;
	int m_exit_code;
	int m_signal_num;
	std::string m_error_msg;
	rsync_behavior::value_type m_behavior;
};

/** Submit or parse a job path report */
class reportio
{
public:
	enum source_type {
		rsync,
		report
	};
	static const char *tags[];

	void write_rsync_out(const std::string& a_str);
	void write_rsync_err(const std::string& a_str);
	void write_report(
		const std::string a_source,
		const timer& a_timer,
		const int a_exit_code,
		const int a_signal_num,
		const rsync_behavior::value_type& a_behavior,
		const std::string& a_error_msg
		);

	job_path_report parse(const std::string& a_str);

	bool is_report(const std::string& a_str);
};

/** A single job report */
class single_job_report
{
public:
	single_job_report();
	~single_job_report();

	void clear(void);

	void add_report(const job_path_report& a_class);
	const std::vector<job_path_report>& reports(void) const;

	void id(const std::string& a_str);
	const std::string& id(void) const;

	const bool status(void) const;

private:
	std::vector<job_path_report> m_reports;
	std::string m_id;
};

/** All job reports */
class jobs_report
{
public:
	jobs_report();
	~jobs_report();

	void clear(void);

	void add_report(const single_job_report& a_class);
	const std::vector<single_job_report>& reports(void) const;

	void write_report(std::ostream& a_out);

	void format_synopsis(table& a_table);

private:
	std::vector<single_job_report> m_jobs;
};

/** The report manager

	Process reports coming in both from the parent and from children, then
	format and print a finalized report.
 */
class report_manager
{
public:
	report_manager();
	~report_manager();

	void init(void);
	const bool initialized(void) const;
	void clear(void);

	void set_total_time(const timer& a_class);

	vault_report& vault(void);
	jobs_report& jobs(void);

	void print_report(void);
	void file_report(void);
	void write_report(std::ostream& a_out);

	void format_synopsis(table& a_table);

private:
	bool m_initialized;
	timer m_total_time;
	vault_report m_vault;
	jobs_report m_jobs;

	void mf_write_header(std::ostream& a_out);
	void mf_write_synopsis(std::ostream& a_out);
};

extern report_manager reporter;

#endif
