#include "config.h"

#include <iostream>
#include <fstream>

#ifdef STAT_MACROS_BROKEN
#error Sorry, S_ISDIR, S_ISREG et. al. appear to be broken on this system.
#endif

#include <cstring>
#include <cerrno>

#ifdef HAVE_SYS_MKDEV_H
#include <sys/mkdev.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef TIME_WITH_SYS_TIME
	#include <sys/time.h>
	#include <time.h>
#else
	#ifdef HAVE_SYS_TIME_H
		#include <sys/time.h>
	#else
		#include <time.h>
	#endif
#endif

#ifdef HAVE_DIRENT_H
	#include <dirent.h>
	#define NAMELEN(dirent) strlen((dirent)->d_name)
#else
	#define dirent direct
	#define NAMELEN(dirent) (dirent)->d_namlen
	#ifdef HAVE_SYS_NDIR_H
		#include <sys/ndir.h>
	#endif
	#ifdef HAVE_SYS_DIR_H
		#include <sys/dir.h>
	#endif
	#ifdef HAVE_NDIR_H
		#include <ndir.h>
	#endif
#endif

#ifdef HAVE_FNMATCH_H
#include <fnmatch.h>
#endif

#include <stdio.h>

#include "asserts.h"
#include "error.h"
#include "estring.h"
#include "fs.h"

//
// Define certain file attribute elements if they don't already exist
//
#ifndef S_IFMT
#define S_IFMT	(S_IFREG|S_IFCHR|S_IFBLK|S_IFIFO)
#endif

#ifndef S_IAMB
#define S_IAMB	(S_ISUID|S_ISGID|S_ISVTX\
		|S_IRUSR|S_IWUSR|S_IXUSR\
		|S_IRGRP|S_IWGRP|S_IXGRP\
		|S_IROTH|S_IWOTH|S_IXOTH\
		)

#endif

#ifdef S_IFIFO
#ifndef S_ISFIFO
#define S_ISFIFO(mode)	(((mode) & S_IFMT) == S_IFIFO)
#endif
#endif

#ifdef S_IFCHR
#ifndef S_ISCHR
#define S_ISCHR(mode)	(((mode) & S_IFMT) == S_IFCHR)
#endif
#endif

#ifdef S_IFDIR
#ifndef S_ISDIR
#define S_ISDIR(mode)	(((mode) & S_IFMT) == S_IFDIR)
#endif
#endif

#ifdef S_IFBLK
#ifndef S_ISBLK
#define S_ISBLK(mode)	(((mode) & S_IFMT) == S_IFBLK)
#endif
#endif

#ifdef S_IFREG
#ifndef S_ISREG
#define S_ISREG(mode)	(((mode) & S_IFMT) == S_IFREG)
#endif
#endif

#ifdef S_IFLNK
#ifndef S_ISLNK
#define S_ISLNK(mode)	(((mode) & S_IFMT) == S_IFLNK)
#endif
#endif

#ifdef S_IFSOCK
#ifndef S_ISSOCK
#define S_ISSOCK(mode)	(((mode) & S_IFMT) == S_IFSOCK)
#endif
#endif

#ifdef S_IFDOOR
#ifndef S_ISDOOR
#define S_ISDOOR(mode)	(((mode) & S_IFMT) == S_IFDOOR)
#endif
#endif

#ifndef S_IRWXU
#define S_IRWXU	(S_IRUSR|S_IWUSR|S_IXUSR)
#endif

#ifndef S_IRWXG
#define S_IRWXG	(S_IRGRP|S_IWGRP|S_IXGRP)
#endif

#ifndef S_IRWXO
#define S_IRWXO	(S_IROTH|S_IWOTH|S_IXOTH)
#endif

#ifndef ACCESSPERMS
#define ACCESSPERMS (S_IRWXU|S_IRWXG|S_IRWXO)
#endif

//----------------------------------------------------------------------------

/** Return the current working directory */
const std::string cwd(void)
{
	std::string s;
	char buf[PATH_MAX] = { 0 };

	if (getcwd(buf, PATH_MAX) == 0)
		throw(ERROR(errno,"Could not determine current directory"));

	TRY_nomem(s = buf);

	return(s);
}

/** Return the PID of this process */
const pid_t pid(void)
{
	pid_t pid;

	pid = getpid();
	if (pid == -1)
		throw(ERROR(errno,"Could not determine PID"));

	return(pid);
}

/** Return the PID of the parent process */
const pid_t parent_pid(void)
{
	pid_t pid;

	pid = getppid();
	if (pid == -1)
		throw(ERROR(errno,"Could not determine PID"));

	return(pid);
}

/** Return true if the string looks like an absolute path */
bool absolute_path(const std::string& a_path)
{
	if ((a_path.size() > 0) && (a_path[0] == '/')) {
		return(true);
	}
	return(false);
}

/** Return true if the string looks like a relative path */
bool relative_path(const std::string& a_path)
{
	bool value;

	value = !absolute_path(a_path);

	return(value);
}

/** Reformat a path to remove double slashes */
std::string reform_path(const std::string& a_path)
{
	std::string::size_type idx;
	std::string str;

	TRY_nomem(str = a_path);
	idx = str.find('/');
	while (idx != std::string::npos) {
		while ((idx != str.size()-1) && (str[idx+1] == '/'))
			str.erase(idx+1,1);
		idx = str.find('/',idx+1);
	}
	
	return(str);
}

/** Reformat a path to remove the begining and trailing slashes, and replace
	all other slashes with underscores
 */
std::string permute_path(const std::string& a_path)
{
	std::string path;
	std::string::size_type idx;

	TRY_nomem(path = reform_path(a_path));
	idx = path.find('/');
	while (idx != std::string::npos) {
		if ((idx == 0) || (idx == path.size()-1))
			path.erase(idx,1);
		else
			path[idx] = '-';
		idx = path.find('/');
	}
	if (path.size() == 0)
		TRY_nomem(path = "root");
	if (path.substr(0,2) == ".-")
		path.erase(0,2);

	return(path);
}

/** Return everything after the last slash from a path */
std::string path_basename(const std::string& a_path)
{
	std::string path;
	std::string::size_type idx;

	idx = a_path.rfind('/');
	if (idx == std::string::npos) {
		TRY_nomem(path = a_path);
	}
	else {
		TRY_nomem(path = a_path.substr(idx+1));
	}
	
	return(path);
}

/** Return everything up to the last slash from a path */
std::string path_dirname(const std::string& a_path)
{
	std::string path;
	std::string::size_type idx;

	idx = a_path.rfind('/');
	if (idx == std::string::npos) {
		TRY_nomem(path = ".");
	}
	else {
		TRY_nomem(path = a_path.substr(0,idx));
	}
	
	return(path);
}

/** Make the path a_rel_path absolute with respect to a_path, where a_rel_path
 * and a_path are directory names */
std::string mk_absolute_path(
	const std::string a_path,
	const std::string a_rel_path
	)
{
	std::string es;
	std::string path;

	if (a_rel_path.size() == 0) {
		TRY_nomem(es = "Invalid relative path: \"");
		TRY_nomem(es += a_rel_path);
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}
	if (relative_path(a_path)) {
		TRY_nomem(es = "Invalid absolute reference path: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(INTERNAL_ERROR(0,es));
	}
	if (relative_path(a_rel_path)) {
		TRY_nomem(path = a_path);
		TRY_nomem(path += "/");
	}
	TRY_nomem(path += a_rel_path);
	path = reform_path(path);

	return(path);
}

/** Make the path a_path_to relative from a_path_from, where a_path_to and
 * a_path_from are directory names */
std::string mk_relative_path(
	const std::string a_path_to,
	const std::string a_path_from
	)
{
	std::string::size_type idx_to, idx_from, idx;
	std::string path_to, path_from, path;
	std::string path_to_substr, path_from_substr;

	TRY_nomem(path_to = a_path_to);
	TRY_nomem(path_from = a_path_from);
	TRY_nomem(path = "");

	idx_to = path_to.find("/");
	if (idx_to != std::string::npos) {
		TRY_nomem(path_to_substr = path_to.substr(0,idx_to));
	}
	else {
		TRY_nomem(path_to_substr = path_to);
	}
	idx_from = path_from.find("/");
	if (idx_from != std::string::npos) {
		TRY_nomem(path_from_substr = path_from.substr(0,idx_from));
	}
	else {
		TRY_nomem(path_from_substr = path_from);
	}

	while (
		(path_to.size() > 0 && path_from.size() > 0)
		&& (path_to_substr == path_from_substr)
		)
	{
		path_to.erase(0,path_to_substr.size());
		if ((path_to.size() > 0) && (path_to[0] == '/'))
			path_to.erase(0,1);
		path_from.erase(0,path_from_substr.size());
		if ((path_from.size() > 0) && (path_from[0] == '/'))
			path_from.erase(0,1);

		idx_to = path_to.find("/");
		if (idx_to != std::string::npos) {
			TRY_nomem(path_to_substr = path_to.substr(0,idx_to));
		}
		else {
			TRY_nomem(path_to_substr = path_to);
		}
		idx_from = path_from.find("/");
		if (idx_from != std::string::npos) {
			TRY_nomem(path_from_substr = path_from.substr(0,idx_from));
		}
		else {
			TRY_nomem(path_from_substr = path_from);
		}
	}

	while (path_from.size() > 0) {
		TRY_nomem(path += "../");
		idx = path_from.find('/');
		if (idx == std::string::npos)
			path_from.erase();
		else
			path_from.erase(0,idx+1);
	}
	TRY_nomem(path += path_to);
	TRY_nomem(path = reform_path(path));

	return(path);
}

/** Return true if the file or directory exists */
bool exists(const std::string& a_path)
{
	if (access(a_path.c_str(), F_OK) == 0) return(true);
	errno = 0;
#ifdef S_ISFIFO
	if (is_fifo_special(a_path)) return(true);
#endif
#ifdef S_ISCHR
	if (is_char_special(a_path)) return(true);
#endif
#ifdef S_ISDIR
	if (is_dir(a_path)) return(true);
#endif
#ifdef S_ISREG
	if (is_file(a_path)) return(true);
#endif
#ifdef S_ISBLK
	if (is_block_special(a_path)) return(true);
#endif
#ifdef S_ISLNK
	if (is_link(a_path)) return(true);
#endif
#ifdef S_ISSOCK
	if (is_socket(a_path)) return(true);
#endif
#ifdef S_ISDOOR
	if (is_door(a_path)) return(true);
#endif
	return(false);
}

/** Return true if the file or directory exists and is readable */
bool readable(const std::string& a_path)
{
	if (access(a_path.c_str(), R_OK) != 0) {
		errno = 0;
		return(false);
	}
	return(true);
}

/** Return true if the file or directory exists and is writable */
bool writable(const std::string& a_path)
{
	if (access(a_path.c_str(), W_OK) != 0) {
		errno = 0;
		return(false);
	}
	return(true);
}

/** Return true if the file or directory exists and is executable */
bool executable(const std::string& a_path)
{
	if (access(a_path.c_str(), X_OK) != 0) {
		errno = 0;
		return(false);
	}
	return(true);
}

#ifdef S_ISFIFO
/** Return true if the file is a fifo-special */
bool is_fifo_special(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_fifo_special()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISCHR
/** Return true if the file is a char-special */
bool is_char_special(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_character_special()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISDIR
/** Return true if the file is a directory */
bool is_dir(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_directory()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISREG
/** Return true if the file is a regular file */
bool is_file(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_regular_file()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISBLK
/** Return true if the file is a block-special */
bool is_block_special(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_block_special()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISLNK
/** Return true is the file is a link */
bool is_link(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_link()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISSOCK
/** Return true if the file is a socket */
bool is_socket(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_socket()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

#ifdef S_ISDOOR
/** Return true if the file is a door */
bool is_door(const std::string& a_path)
{
	filestatus fstat;

	try {
		fstat.path(a_path);
		if (fstat.is_door()) {
			return(true);
		}
	}
	catch(...) {
		return(false);
	}
	return(false);
}
#endif

/** Create a directory */
void mk_dir(const std::string& a_path)
{
	std::string es;

	if (mkdir(a_path.c_str(), ACCESSPERMS) != 0) {
		TRY_nomem(es = "Could not create directory: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
}

/** Remove a directory */
void rm_dir(const std::string a_path)
{
	if (!exists(a_path)) return;
	if (rmdir(a_path.c_str()) != 0) {
		std::string es;

		TRY_nomem(es = "Could not remove directory: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");

		if (exists(a_path)) {
			if (writable(a_path)) {
				throw(ERROR(errno,es));
			}
			else {
				error e = ERROR(errno,es);

				e.push_back(ERROR_INSTANCE("No write permissions"));
				throw(e);
			}
		}
	}
}

/** Remove a file */
void rm_file(const std::string a_path)
{
	if (!exists(a_path)) return;
	if (unlink(a_path.c_str()) != 0) {
		std::string es;

		TRY_nomem(es = "Could not remove file: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");

		if (exists(a_path)) {
			if (writable(a_path)) {
				throw(ERROR(errno,es));
			}
			else {
				error e = ERROR(errno,es);

				e.push_back(ERROR_INSTANCE("No write permissions"));
				throw(e);
			}
		}
	}
}

/** Recursively create a directory heirarchy */
void mk_dirhier_recursive_(const std::string a_path)
{
	std::string parent_dir;
	int ce;

	if (a_path.size() == 0)
		return;
	if (exists(a_path))
		return;
	
	for (ce = a_path.size()-1; ((ce > 0) && (a_path[ce] != '/')); ce--);

	if (ce > 0) {
		TRY_nomem(parent_dir = a_path.substr(0,ce));
		mk_dirhier_recursive_(parent_dir);
	}
	if (!exists(a_path))
		mk_dir(a_path);
}

/** Recursively create a directory heirarchy */
void mk_dirhier(const std::string a_path)
{
	if (a_path.size() == 0)
		return;
	if (exists(a_path))
		return;
	
	try {
		mk_dirhier_recursive_(a_path);
	}
	catch(error e) {
		std::string es;

		TRY_nomem(es = "Could not create directory hierarchy: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");

		e.push_back(ERROR_INSTANCE(es));
		throw(e);
	}
	catch(...) {
		throw(err_unknown);
	}
}

/** Rename a file or directory */
void rename_file(const std::string a_from, const std::string a_to)
{
	std::string es;

	if (a_from.size() == 0) {
		TRY_nomem(es = "Illegal from filename: \"");
		TRY_nomem(es += a_from);
		TRY_nomem(es += "\"");
		throw(INTERNAL_ERROR(0,es));
	}
	if (!exists(a_from)) {
		TRY_nomem(es = "From filename does not exist: \"");
		TRY_nomem(es += a_from);
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}
	if (a_to.size() == 0) {
		TRY_nomem(es = "Illegal to filename: \"");
		TRY_nomem(es += a_to);
		TRY_nomem(es += "\"");
		throw(INTERNAL_ERROR(0,es));
	}
	if (exists(a_to)) {
		TRY_nomem(es = "To filename already exists: \"");
		TRY_nomem(es += a_to);
		TRY_nomem(es += "\"");
		throw(ERROR(0,es));
	}
	if (rename(a_from.c_str(), a_to.c_str()) != 0) {
		TRY_nomem(es = "Could not rename file: \"");
		TRY_nomem(es += a_from);
		TRY_nomem(es += "\" to \"");
		TRY_nomem(es += a_to);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
}

/** Create a symbolic link */
void mk_symlink(const std::string a_from, const std::string a_to)
{
	if (symlink(a_from.c_str(), a_to.c_str()) != 0) {
		std::string es;

		TRY_nomem(es = "Could not link: \"");
		TRY_nomem(es += a_from);
		TRY_nomem(es += "\" to: \"");
		TRY_nomem(es += a_to);
		TRY_nomem(es += "\"");

		throw(ERROR(errno,es));
	}
}

/** Given a from and to path, create a relative symbolic link */
void mk_relative_symlink(const std::string a_from, const std::string a_to)
{
	std::string from_dirname, to_dirname, from_basename;
	std::string rel;

	TRY_nomem(from_dirname = path_dirname(a_from));
	TRY_nomem(to_dirname = path_dirname(a_to));
	TRY_nomem(from_basename = path_basename(a_from));
	TRY_nomem(rel = mk_relative_path(from_dirname, to_dirname));
	if (rel.size() != 0) {
		TRY_nomem(rel += "/");
	}
	TRY_nomem(rel += from_basename);
	TRY_nomem(rel = reform_path(rel));

	try {
		mk_symlink(rel,a_to);
	}
	catch(error e) {
		std::string es;

		TRY_nomem(es = "Could not link: \"");
		TRY_nomem(es += a_to);
		TRY_nomem(es += "\" as: \"");
		TRY_nomem(es += rel);
		TRY_nomem(es += "\"");

		e.push_back(ERROR_INSTANCE(es));
		throw(e);
	}
	catch(...) {
		throw(err_unknown);
	}
}

//----------------------------------------------------------------------------

/** C'tor */
filestatus::filestatus()
{
}

/** C'tor */
filestatus::filestatus(const std::string a_path)
{
	path(a_path);
}

/** D'tor */
filestatus::~filestatus()
{
}

/** Retrieve information about a pathname */
void filestatus::path(const std::string a_path)
{
	std::string es;
	struct passwd *passwd_ptr = 0;
	struct group *group_ptr = 0;
	struct stat statbuf;
	char path_buf[PATH_MAX] = { 0 };

	clear();
	TRY_nomem(m_path = reform_path(a_path));
	if (lstat(m_path.c_str(), &statbuf) < 0) {
		TRY_nomem(es = "For path: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
	if (lstat(m_path.c_str(), &m_stat) == -1) {
		TRY_nomem(es = "For path: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
#ifdef S_ISFIFO
	if (S_ISFIFO(m_stat.st_mode)) {
		m_major = major(m_stat.st_rdev);
		m_minor = minor(m_stat.st_rdev);
	}
#endif
#ifdef S_ISCHR
	if (S_ISCHR(m_stat.st_mode)) {
		m_major = major(m_stat.st_rdev);
		m_minor = minor(m_stat.st_rdev);
	}
#endif
#ifdef S_ISBLK
	if (S_ISBLK(m_stat.st_mode)) {
		m_major = major(m_stat.st_rdev);
		m_minor = minor(m_stat.st_rdev);
	}
#endif
#ifdef S_ISLNK
	if (S_ISLNK(m_stat.st_mode)) {
		if (readlink(a_path.c_str(), path_buf, PATH_MAX) < 0) {
			TRY_nomem(es = "Could not read path's link: \"");
			TRY_nomem(es += a_path);
			TRY_nomem(es += "\"");
			throw(ERROR(errno,es));
		}
		TRY_nomem(m_link = path_buf);
	}
#endif
	passwd_ptr = getpwuid(m_stat.st_uid);
	if (passwd_ptr != 0) {
		m_uidfound = true;
		TRY_nomem(m_uname = passwd_ptr->pw_name);
	}
	group_ptr = getgrgid(m_stat.st_gid);
	if (group_ptr != 0) {
		m_gidfound = true;
		TRY_nomem(m_gname = group_ptr->gr_name);
	}
}

/** Return the pathname that this filestatus object has information about */
const std::string filestatus::path(void) const
{
	return(m_path);
}

/** Return the type of file */
const filestatus::filetype filestatus::type(void) const
{
#ifdef S_ISFIFO
	if (S_ISFIFO(m_stat.st_mode)) {
		return(type_fifo_special);
	}
#endif
#ifdef S_ISCHR
	if (S_ISCHR(m_stat.st_mode)) {
		return(type_character_special);
	}
#endif
#ifdef S_ISDIR
	if (S_ISDIR(m_stat.st_mode)) {
		return(type_directory);
	}
#endif
#ifdef S_ISBLK
	if (S_ISBLK(m_stat.st_mode)) {
		return(type_block_special);
	}
#endif
#ifdef S_ISREG
	if (S_ISREG(m_stat.st_mode)) {
		return(type_regular_file);
	}
#endif
#ifdef S_ISLNK
	if (S_ISLNK(m_stat.st_mode)) {
		return(type_link);
	}
#endif
#ifdef S_ISSOCK
	if (S_ISSOCK(m_stat.st_mode)) {
		return(type_socket);
	}
#endif
#ifdef S_ISDOOR
	if (S_ISDOOR(m_stat.st_mode)) {
		return(type_door);
	}
#endif
	return(type_unknown);
}

/** If the pathname is a link, return the path it is linked to */
const std::string filestatus::link(void) const
{
	return(m_link);
}

/** Return the file mode */
const filestatus::mode_type filestatus::mode(void) const
{
	return(m_stat.st_mode);
}

/** Return the file inode */
const filestatus::inode_type filestatus::inode(void) const
{
	return(m_stat.st_ino);
}

/** Return the file's device */
const filestatus::device_type filestatus::dev(void) const
{
	return(m_stat.st_dev);
}

/** Return the file's raw device */
const filestatus::device_type filestatus::rdev(void) const
{
	return(m_stat.st_rdev);
}

/** If the pathname is a special file, return it's major number */
const filestatus::major_type filestatus::get_major(void) const
{
	return(m_major);
}

/** If the pathname is a special file, return it's minor number */
const filestatus::minor_type filestatus::get_minor(void) const
{
	return(m_minor);
}

/** Return the number of links to this file */
const filestatus::num_links_type filestatus::num_links(void) const
{
	return(m_stat.st_nlink);
}

/** Return the file's owner's UID */
const filestatus::uid_type filestatus::uid(void) const
{
	return(m_stat.st_uid);
}

/** Return the file's owner's GID */
const filestatus::gid_type filestatus::gid(void) const
{
	return(m_stat.st_gid);
}

/** Return the file size in bytes */
const filestatus::size_type filestatus::size(void) const
{
	size_type value;

	value = static_cast<uint64>(m_stat.st_size);

	return(value);
}

/** Return the last access time of this file */
const filestatus::time_type filestatus::last_access_time(void) const
{
	return(m_stat.st_atime);
}

/** Return the last modification time of this file */
const filestatus::time_type filestatus::last_modification_time(void) const
{
	return(m_stat.st_mtime);
}

/** Return the last status change time of this file */
const filestatus::time_type filestatus::last_status_change_time(void) const
{
	return(m_stat.st_ctime);
}

/** Return the blocksize used to store this file */
const filestatus::size_type filestatus::blocksize(void) const
{
	size_type value;

	value = static_cast<uint64>(m_stat.st_blksize);

	return(value);
}

/** Return the number of blocks used to store this file */
const filestatus::size_type filestatus::blocks(void) const
{
	size_type value;

	value = static_cast<uint64>(m_stat.st_blocks);

	return(value);
}

/** If the file's owner's UID is found in the passwd file, return true */
const bool filestatus::uid_is_found(void) const
{
	return(m_uidfound);
}

/** If the file's owner's GID is found in the passwd file, return true */
const bool filestatus::gid_is_found(void) const
{
	return(m_gidfound);
}

/** Return the file's owner's user name (from UID) */
const std::string filestatus::uid_name(void) const
{
	return(m_uname);
}

/** Return the file's owner's group name (from UID) */
const std::string filestatus::gid_name(void) const
{
	return(m_gname);
}

#ifdef S_ISFIFO
/** Return true if the file is a fifo-special */
const bool filestatus::is_fifo_special(void) const
{
	bool value;

	value = (type() == type_fifo_special);

	return(value);
}
#endif

#ifdef S_ISCHR
/** Return true if the file is a char-special */
const bool filestatus::is_character_special(void) const
{
	bool value;

	value = (type() == type_character_special);

	return(value);
}
#endif

#ifdef S_ISBLK
/** Return true if the file is a block-special */
const bool filestatus::is_block_special(void) const
{
	bool value;

	value = (type() == type_block_special);

	return(value);
}
#endif

#ifdef S_ISLNK
/** Return true if the file is a link */
const bool filestatus::is_link(void) const
{
	bool value;

	value = (type() == type_link);

	return(value);
}
#endif

#ifdef S_ISSOCK
/** Return true if the file is a socket */
const bool filestatus::is_socket(void) const
{
	bool value;

	value = (type() == type_socket);

	return(value);
}
#endif

#ifdef S_ISDOOR
/** Return true if the file is a door */
const bool filestatus::is_door(void) const
{
	bool value;

	value = (type() == type_door);

	return(value);
}
#endif

#ifdef S_ISDIR
/** Return true if the file is a directory */
const bool filestatus::is_directory(void) const
{
	bool value;

	value = (type() == type_directory);

	return(value);
}
#endif

#ifdef S_ISREG
/** Return true if the file is a regular file */
const bool filestatus::is_regular_file(void) const
{
	bool value;

	value = (type() == type_regular_file);

	return(value);
}
#endif

#ifdef S_IRUSR
/** Return true if the file is readable by it's owner */
const bool filestatus::user_can_read(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IRUSR) != 0);

	return(value);
}
#endif

#ifdef S_IWUSR
/** Return true if the file is writable by it's owner */
const bool filestatus::user_can_write(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IWUSR) != 0);

	return(value);
}
#endif

#ifdef S_IXUSR
/** Return true if the file is executable by it's owner */
const bool filestatus::user_can_execute(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IXUSR) != 0);

	return(value);
}
#endif

#ifdef S_IRGRP
/** Return true if the file is readable by users in the same group */
const bool filestatus::group_can_read(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IRGRP) != 0);

	return(value);
}
#endif

#ifdef S_IWGRP
/** Return true if the file is writable by users in the same group */
const bool filestatus::group_can_write(void) const
{
	bool value;
	
	value = ((m_stat.st_mode & S_IWGRP) != 0);

	return(value);
}
#endif

#ifdef S_IXGRP
/** Return true if the file is executable by users in the same group */
const bool filestatus::group_can_execute(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IXGRP) != 0);

	return(value);
}
#endif

#ifdef S_IROTH
/** Return true if the file is readable by others */
const bool filestatus::other_can_read(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IROTH) != 0);

	return(value);
}
#endif

#ifdef S_IWOTH
/** Return true if the file is writable by others */
const bool filestatus::other_can_write(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IWOTH) != 0);

	return(value);
}
#endif

#ifdef S_IXOTH
/** Return true if the file is executable by others */
const bool filestatus::other_can_execute(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_IXOTH) != 0);

	return(value);
}
#endif

#ifdef S_ISUID
/** Return true if the file's mode has it's set-uid bit set */
const bool filestatus::is_set_uid(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_ISUID) != 0);

	return(value);
}
#endif

#ifdef S_ISGID
/** Return true if the file's mode has it's set-gid bit set */
const bool filestatus::is_set_gid(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_ISGID) != 0);

	return(value);
}
#endif

#ifdef S_ISVTX
/** Return true if the file's mode has it's sticky bit set */
const bool filestatus::is_set_sticky(void) const
{
	bool value;

	value = ((m_stat.st_mode & S_ISVTX) != 0);

	return(value);
}
#endif

/** Clear all values */
void filestatus::clear(void)
{
	TRY_nomem(m_path = "");
	memset(&m_stat, 0, sizeof(m_stat));
	m_major = 0;
	m_minor = 0;
	m_uidfound = false;
	m_gidfound = false;
	TRY_nomem(m_uname = "");
	TRY_nomem(m_gname = "");
}

//----------------------------------------------------------------------------

/** C'tor */
subdirectory::subdirectory()
{
}

/** C'tor */
subdirectory::subdirectory(const subdirectory& a_class)
{
	assign(a_class);
}

/** C'or */
subdirectory::subdirectory(
	const std::string a_path, const std::string a_filter)
{
	path(a_path, a_filter);
}

/** D'tor */
subdirectory::~subdirectory()
{
}

/** Assign the contents of a given subdirectory to this subdirectory */
void subdirectory::assign(const subdirectory& a_class)
{
	type::assign(a_class.begin(), a_class.end());
}

/** Return a vector of strings of a list of files in a subdirectory

	Files in the list do not contain the pathname to the file.  Only filenames
	that match the given wildcard filter will be listed.
 */
const subdirectory::type& 
	subdirectory::path(const std::string a_path, const std::string a_filter)
{
	std::string es;
	std::string name;
	DIR *dp = 0;
	struct dirent *dirp = 0;
	filestatus filestat;

	clear();

	TRY(filestat.path(a_path),"Could not stat directory");

	dp = opendir(a_path.c_str());
	if (dp == 0) {
		TRY_nomem(es = "For path: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
	while ( (dirp = readdir(dp)) != 0 ) {
		name = dirp->d_name;
		if (
			((a_filter != ".") && (a_filter != "..")) 
			&& 
			((name == ".") || (name == ".."))
			)
			continue;
		if (fnmatch(a_filter.c_str(), name.c_str(), 0) == 0) {
			TRY_nomem(type::push_back(name));
		}
	}
	if (closedir(dp) < 0) {
		TRY_nomem(es = "Error closing directory \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}

	std::sort(begin(), end());

	return(*this);
}

subdirectory& subdirectory::operator=(const subdirectory& a_class)
{
	assign(a_class);

	return(*this);
}

//----------------------------------------------------------------------------

/** Recursively delete the contents of a directory */
void rm_recursive(const std::string a_path)
{
	subdirectory d;
	subdirectory::const_iterator di;
	filestatus f;
	std::string pathstr;
	std::string es;

	TRY_nomem(es = "Could not recursively delete: \"");
	TRY_nomem(es += a_path);
	TRY_nomem(es += "\"");

	if (!exists(a_path)) {
		if (is_link(a_path)) {
			TRY(rm_file(a_path),es);
		}
		return;
	}
	TRY(f.path(a_path),es);
	if (f.is_directory()) {
		TRY(d.path(a_path),es);
		if (d.size() != 0) {
	
			for (di = d.begin(); di != d.end(); ++di) {
				TRY_nomem(
					pathstr = static_cast<std::string>(a_path)
						+ static_cast<std::string>("/")
						+ static_cast<std::string>(*di)
						);
				TRY(f.path(pathstr),es);
				if (f.is_directory()) {
					TRY(rm_recursive(pathstr),es);
				}
				else {
					rm_file(pathstr);
				}
			}
		}
		rm_dir(a_path);
	}
	else {
		rm_file(a_path);
	}
}

//----------------------------------------------------------------------------

/** C'tor */
directory::directory()
{
}

/** C'tor */
directory::directory(const directory& a_class)
{
	// TODO: Isn't this supposed to do something!?
}

/** C'tor */
directory::directory(const std::string& a_str)
{
	path(a_str);
}

/** D'tor */
directory::~directory()
{
}

/** Retrieve a list of paths that match the wildcard path given */
const directory::type& directory::path(const std::string& a_path)
{
	std::string es;
	std::string::size_type idx;
	std::string path;
	std::string subdir_path;
	std::string new_subdir_path;
	subdirectory subdir;
	subdirectory::const_iterator sdi;
	filestatus filestat;
	const_iterator di;
	type list;

	TRY_nomem(path = reform_path(a_path));
	if (path.size() == 0) {
		return(*this);
	}
	
	idx = path.find('/');
	if (idx != std::string::npos) {
		TRY_nomem(subdir_path = path.substr(0,idx));
		path.erase(0,idx+1);
	}
	else {
		TRY_nomem(subdir_path = path);
		path.erase();
	}
	if (subdir_path.size() == 0)
		TRY_nomem(subdir_path = "/");
	
	if (!exists(subdir_path)) {
		return(*this);
	}
	TRY_nomem(push_back(subdir_path));

	while (path.size() != 0) {
		idx = path.find('/');
		if (idx != std::string::npos) {
			TRY_nomem(subdir_path = path.substr(0,idx));
			path.erase(0,idx+1);
		}
		else {
			TRY_nomem(subdir_path = path);
			path.erase();
		}
		list.clear();
		for (di = begin(); di != end(); di++) {
			TRY_nomem(list.push_back(*di));
		}
		clear();
		for (di = list.begin(); di != list.end(); di++) {
			filestat.path(*di);
			if (!filestat.is_directory())
				continue;
			subdir.path(*di, subdir_path);
			for (sdi = subdir.begin(); sdi != subdir.end(); sdi++) {
				TRY_nomem(new_subdir_path = *di);
				TRY_nomem(new_subdir_path += "/");
				TRY_nomem(new_subdir_path += *sdi);
				TRY_nomem(new_subdir_path = reform_path(new_subdir_path));
				TRY_nomem(push_back(new_subdir_path));
			}
		}
	}
	return(*this);
}

//----------------------------------------------------------------------------

/** C'tor */
filesystem::filesystem()
{
	clear();
}

/** C'tor */
filesystem::filesystem(const std::string& a_path)
{
	clear();
	path(a_path);
}

/** Clear the filesystem object */
void filesystem::clear(void)
{
	TRY_nomem(m_path = "");
	memset(&m_statfs, 0, sizeof(m_statfs));
}

/** Retrieve information about the filesystem on which the given path resides */
void filesystem::path(const std::string& a_path)
{
	std::string es;

	TRY_nomem(m_path = reform_path(a_path));
	if (STATFS(a_path.c_str(), &m_statfs) != 0) {
		TRY_nomem(es = "Could not stat filesystem: \"");
		TRY_nomem(es += a_path);
		TRY_nomem(es += "\"");
		throw(ERROR(errno,es));
	}
}

/** Return the path from which this filesystem information was obtained */
const std::string filesystem::path(void) const
{
	return(m_path);
}

/** Return the filesystem block size */
const filesystem::size_type filesystem::blocksize(void) const
{
	size_type value;

	value = static_cast<uint64>(m_statfs.f_bsize);

	return(value);
}

/** Return the filesystem's total number of blocks */
const filesystem::size_type filesystem::total_blocks(void) const
{
	size_type value;

	value = static_cast<uint64>(m_statfs.f_blocks);

	return(value);
}

/** Return the filesystem's number of free blocks */
const filesystem::size_type filesystem::free_blocks(void) const
{
	size_type value;

	value = static_cast<uint64>(m_statfs.f_bfree);

	return(value);
}

/** Return the filesystem's number of used blocks */
const filesystem::size_type filesystem::used_blocks(void) const
{
	size_type value;

	value = total_blocks() - free_blocks();

	return(value);
}

/** Return the filesystem's total number of inodes, if supported by the
	filesystem, otherwise the result is system-dependent, but usually 0.
 */
const filesystem::size_type filesystem::total_inodes(void) const
{
	size_type value;

	value = static_cast<uint64>(m_statfs.f_files);

	return(value);
}

/** Return the filesystem's total number of free inodes, if supported by the
	filesystem, otherwise the result is system-dependent, but usually 0.
 */
const filesystem::size_type filesystem::free_inodes(void) const
{
	size_type value;
	
	value = static_cast<uint64>(m_statfs.f_ffree);

	return(value);
}

/** Return the filesystem's number of used inodes */
const filesystem::size_type filesystem::used_inodes(void) const
{
	size_type value;

	value = total_inodes() - free_inodes();

	return(value);
}

/** Copy values from another instance */
filesystem& filesystem::operator=(const filesystem& a_class)
{
	TRY_nomem(m_path = a_class.m_path);
	memcpy(&m_statfs, &a_class.m_statfs, sizeof(m_statfs));
	
	return(*this);
}

//----------------------------------------------------------------------------

/** C'tor */
simple_lock::simple_lock()
{
	clear();
}

/** C'tor */
simple_lock::simple_lock(const std::string& a_lockfile)
{
	clear();
	lockfile(a_lockfile);
}

/** D'tor */
simple_lock::~simple_lock()
{
	clear();
}

/** Clear the simple_lock object */
void simple_lock::clear(void)
{
	if (is_locked() && (locked_by() == getpid())) {
		unlock();
	}
	m_lockfile.clear();
}

/** Set the lockfile path */
void simple_lock::lockfile(const std::string& a_lockfile)
{
	clear();
	m_lockfile = a_lockfile;
}

/** Get the lockfile path */
const std::string simple_lock::lockfile(void) const
{
	return(m_lockfile);
}

/** Get the PID of the locking process */
const simple_lock::pid_type simple_lock::locked_by(void) const
{
	simple_lock::pid_type pid;
	std::ifstream in;

	pid = 0;
	if (m_lockfile.size() == 0) return(0);
	if (!exists(m_lockfile)) return(0);
	in.open(m_lockfile.c_str());
	if (!in.is_open()) return(0);
	in >> pid;
	if (!in) {
		in.close();
		return(0);
	}
	return(pid);
}

/** Find out whether or not the lock is in place */
const bool simple_lock::is_locked(void) const
{
	simple_lock::pid_type pid;

	pid = locked_by();
	if (pid == 0) return(false);
	if (pid == getpid()) return(true);
	if (kill(pid,0) >= 0) return(true);
	errno = 0;
	return(false);
}

/** Lock */
bool simple_lock::lock(void)
{
	std::ofstream out;
	simple_lock::pid_type pid;

	if (is_locked()) return(false);
	if (m_lockfile.size() == 0) return(false);
	out.open(m_lockfile.c_str());
	if (!out.is_open()) return(false);
	pid = getpid();
	out << pid << std::endl;
	if (!out) {
		clear();
		return(false);
	}
	out.close();
	return(true);
}

/** Unlock */
void simple_lock::unlock(void)
{
	std::string es;

	if (m_lockfile.size() == 0) return;
	if (!exists(m_lockfile)) return;
	TRY_nomem(es = "Cannot unlock: \"");
	TRY_nomem(es += m_lockfile);
	TRY_nomem(es += "\"");
	TRY(rm_file(m_lockfile),es);
}

//----------------------------------------------------------------------------

