use std::fmt::Formatter;

use crate::registry::{Linter, Rule};

#[derive(PartialEq, Eq, PartialOrd, Ord)]
pub struct NoqaCode(&'static str, &'static str);

impl std::fmt::Debug for NoqaCode {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        std::fmt::Display::fmt(self, f)
    }
}

impl std::fmt::Display for NoqaCode {
    fn fmt(&self, f: &mut Formatter<'_>) -> Result<(), std::fmt::Error> {
        write!(f, "{}{}", self.0, self.1)
    }
}

impl PartialEq<&str> for NoqaCode {
    fn eq(&self, other: &&str) -> bool {
        match other.strip_prefix(self.0) {
            Some(suffix) => suffix == self.1,
            None => false,
        }
    }
}

#[derive(Debug, Copy, Clone)]
pub enum RuleGroup {
    /// The rule has not been assigned to any specific group.
    Unspecified,
    /// The rule is still under development, and must be enabled explicitly.
    Nursery,
}

#[ruff_macros::map_codes]
pub fn code_to_rule(linter: Linter, code: &str) -> Option<(RuleGroup, Rule)> {
    #[allow(clippy::enum_glob_use)]
    use Linter::*;

    #[rustfmt::skip]
    Some(match (linter, code) {
        // pycodestyle errors
        (Pycodestyle, "E101") => (RuleGroup::Unspecified, Rule::MixedSpacesAndTabs),
        (Pycodestyle, "E111") => (RuleGroup::Nursery, Rule::IndentationWithInvalidMultiple),
        (Pycodestyle, "E112") => (RuleGroup::Nursery, Rule::NoIndentedBlock),
        (Pycodestyle, "E113") => (RuleGroup::Nursery, Rule::UnexpectedIndentation),
        (Pycodestyle, "E114") => (RuleGroup::Nursery, Rule::IndentationWithInvalidMultipleComment),
        (Pycodestyle, "E115") => (RuleGroup::Nursery, Rule::NoIndentedBlockComment),
        (Pycodestyle, "E116") => (RuleGroup::Nursery, Rule::UnexpectedIndentationComment),
        (Pycodestyle, "E117") => (RuleGroup::Nursery, Rule::OverIndented),
        (Pycodestyle, "E201") => (RuleGroup::Nursery, Rule::WhitespaceAfterOpenBracket),
        (Pycodestyle, "E202") => (RuleGroup::Nursery, Rule::WhitespaceBeforeCloseBracket),
        (Pycodestyle, "E203") => (RuleGroup::Nursery, Rule::WhitespaceBeforePunctuation),
        (Pycodestyle, "E211") => (RuleGroup::Nursery, Rule::WhitespaceBeforeParameters),
        (Pycodestyle, "E221") => (RuleGroup::Nursery, Rule::MultipleSpacesBeforeOperator),
        (Pycodestyle, "E222") => (RuleGroup::Nursery, Rule::MultipleSpacesAfterOperator),
        (Pycodestyle, "E223") => (RuleGroup::Nursery, Rule::TabBeforeOperator),
        (Pycodestyle, "E224") => (RuleGroup::Nursery, Rule::TabAfterOperator),
        (Pycodestyle, "E225") => (RuleGroup::Nursery, Rule::MissingWhitespaceAroundOperator),
        (Pycodestyle, "E226") => (RuleGroup::Nursery, Rule::MissingWhitespaceAroundArithmeticOperator),
        (Pycodestyle, "E227") => (RuleGroup::Nursery, Rule::MissingWhitespaceAroundBitwiseOrShiftOperator),
        (Pycodestyle, "E228") => (RuleGroup::Nursery, Rule::MissingWhitespaceAroundModuloOperator),
        (Pycodestyle, "E231") => (RuleGroup::Nursery, Rule::MissingWhitespace),
        (Pycodestyle, "E251") => (RuleGroup::Nursery, Rule::UnexpectedSpacesAroundKeywordParameterEquals),
        (Pycodestyle, "E252") => (RuleGroup::Nursery, Rule::MissingWhitespaceAroundParameterEquals),
        (Pycodestyle, "E261") => (RuleGroup::Nursery, Rule::TooFewSpacesBeforeInlineComment),
        (Pycodestyle, "E262") => (RuleGroup::Nursery, Rule::NoSpaceAfterInlineComment),
        (Pycodestyle, "E265") => (RuleGroup::Nursery, Rule::NoSpaceAfterBlockComment),
        (Pycodestyle, "E266") => (RuleGroup::Nursery, Rule::MultipleLeadingHashesForBlockComment),
        (Pycodestyle, "E271") => (RuleGroup::Nursery, Rule::MultipleSpacesAfterKeyword),
        (Pycodestyle, "E272") => (RuleGroup::Nursery, Rule::MultipleSpacesBeforeKeyword),
        (Pycodestyle, "E273") => (RuleGroup::Nursery, Rule::TabAfterKeyword),
        (Pycodestyle, "E274") => (RuleGroup::Nursery, Rule::TabBeforeKeyword),
        (Pycodestyle, "E275") => (RuleGroup::Nursery, Rule::MissingWhitespaceAfterKeyword),
        (Pycodestyle, "E401") => (RuleGroup::Unspecified, Rule::MultipleImportsOnOneLine),
        (Pycodestyle, "E402") => (RuleGroup::Unspecified, Rule::ModuleImportNotAtTopOfFile),
        (Pycodestyle, "E501") => (RuleGroup::Unspecified, Rule::LineTooLong),
        (Pycodestyle, "E701") => (RuleGroup::Unspecified, Rule::MultipleStatementsOnOneLineColon),
        (Pycodestyle, "E702") => (RuleGroup::Unspecified, Rule::MultipleStatementsOnOneLineSemicolon),
        (Pycodestyle, "E703") => (RuleGroup::Unspecified, Rule::UselessSemicolon),
        (Pycodestyle, "E711") => (RuleGroup::Unspecified, Rule::NoneComparison),
        (Pycodestyle, "E712") => (RuleGroup::Unspecified, Rule::TrueFalseComparison),
        (Pycodestyle, "E713") => (RuleGroup::Unspecified, Rule::NotInTest),
        (Pycodestyle, "E714") => (RuleGroup::Unspecified, Rule::NotIsTest),
        (Pycodestyle, "E721") => (RuleGroup::Unspecified, Rule::TypeComparison),
        (Pycodestyle, "E722") => (RuleGroup::Unspecified, Rule::BareExcept),
        (Pycodestyle, "E731") => (RuleGroup::Unspecified, Rule::LambdaAssignment),
        (Pycodestyle, "E741") => (RuleGroup::Unspecified, Rule::AmbiguousVariableName),
        (Pycodestyle, "E742") => (RuleGroup::Unspecified, Rule::AmbiguousClassName),
        (Pycodestyle, "E743") => (RuleGroup::Unspecified, Rule::AmbiguousFunctionName),
        (Pycodestyle, "E902") => (RuleGroup::Unspecified, Rule::IOError),
        (Pycodestyle, "E999") => (RuleGroup::Unspecified, Rule::SyntaxError),

        // pycodestyle warnings
        (Pycodestyle, "W191") => (RuleGroup::Unspecified, Rule::TabIndentation),
        (Pycodestyle, "W291") => (RuleGroup::Unspecified, Rule::TrailingWhitespace),
        (Pycodestyle, "W292") => (RuleGroup::Unspecified, Rule::MissingNewlineAtEndOfFile),
        (Pycodestyle, "W293") => (RuleGroup::Unspecified, Rule::BlankLineWithWhitespace),
        (Pycodestyle, "W505") => (RuleGroup::Unspecified, Rule::DocLineTooLong),
        (Pycodestyle, "W605") => (RuleGroup::Unspecified, Rule::InvalidEscapeSequence),

        // pyflakes
        (Pyflakes, "401") => (RuleGroup::Unspecified, Rule::UnusedImport),
        (Pyflakes, "402") => (RuleGroup::Unspecified, Rule::ImportShadowedByLoopVar),
        (Pyflakes, "403") => (RuleGroup::Unspecified, Rule::UndefinedLocalWithImportStar),
        (Pyflakes, "404") => (RuleGroup::Unspecified, Rule::LateFutureImport),
        (Pyflakes, "405") => (RuleGroup::Unspecified, Rule::UndefinedLocalWithImportStarUsage),
        (Pyflakes, "406") => (RuleGroup::Unspecified, Rule::UndefinedLocalWithNestedImportStarUsage),
        (Pyflakes, "407") => (RuleGroup::Unspecified, Rule::FutureFeatureNotDefined),
        (Pyflakes, "501") => (RuleGroup::Unspecified, Rule::PercentFormatInvalidFormat),
        (Pyflakes, "502") => (RuleGroup::Unspecified, Rule::PercentFormatExpectedMapping),
        (Pyflakes, "503") => (RuleGroup::Unspecified, Rule::PercentFormatExpectedSequence),
        (Pyflakes, "504") => (RuleGroup::Unspecified, Rule::PercentFormatExtraNamedArguments),
        (Pyflakes, "505") => (RuleGroup::Unspecified, Rule::PercentFormatMissingArgument),
        (Pyflakes, "506") => (RuleGroup::Unspecified, Rule::PercentFormatMixedPositionalAndNamed),
        (Pyflakes, "507") => (RuleGroup::Unspecified, Rule::PercentFormatPositionalCountMismatch),
        (Pyflakes, "508") => (RuleGroup::Unspecified, Rule::PercentFormatStarRequiresSequence),
        (Pyflakes, "509") => (RuleGroup::Unspecified, Rule::PercentFormatUnsupportedFormatCharacter),
        (Pyflakes, "521") => (RuleGroup::Unspecified, Rule::StringDotFormatInvalidFormat),
        (Pyflakes, "522") => (RuleGroup::Unspecified, Rule::StringDotFormatExtraNamedArguments),
        (Pyflakes, "523") => (RuleGroup::Unspecified, Rule::StringDotFormatExtraPositionalArguments),
        (Pyflakes, "524") => (RuleGroup::Unspecified, Rule::StringDotFormatMissingArguments),
        (Pyflakes, "525") => (RuleGroup::Unspecified, Rule::StringDotFormatMixingAutomatic),
        (Pyflakes, "541") => (RuleGroup::Unspecified, Rule::FStringMissingPlaceholders),
        (Pyflakes, "601") => (RuleGroup::Unspecified, Rule::MultiValueRepeatedKeyLiteral),
        (Pyflakes, "602") => (RuleGroup::Unspecified, Rule::MultiValueRepeatedKeyVariable),
        (Pyflakes, "621") => (RuleGroup::Unspecified, Rule::ExpressionsInStarAssignment),
        (Pyflakes, "622") => (RuleGroup::Unspecified, Rule::MultipleStarredExpressions),
        (Pyflakes, "631") => (RuleGroup::Unspecified, Rule::AssertTuple),
        (Pyflakes, "632") => (RuleGroup::Unspecified, Rule::IsLiteral),
        (Pyflakes, "633") => (RuleGroup::Unspecified, Rule::InvalidPrintSyntax),
        (Pyflakes, "634") => (RuleGroup::Unspecified, Rule::IfTuple),
        (Pyflakes, "701") => (RuleGroup::Unspecified, Rule::BreakOutsideLoop),
        (Pyflakes, "702") => (RuleGroup::Unspecified, Rule::ContinueOutsideLoop),
        (Pyflakes, "704") => (RuleGroup::Unspecified, Rule::YieldOutsideFunction),
        (Pyflakes, "706") => (RuleGroup::Unspecified, Rule::ReturnOutsideFunction),
        (Pyflakes, "707") => (RuleGroup::Unspecified, Rule::DefaultExceptNotLast),
        (Pyflakes, "722") => (RuleGroup::Unspecified, Rule::ForwardAnnotationSyntaxError),
        (Pyflakes, "811") => (RuleGroup::Unspecified, Rule::RedefinedWhileUnused),
        (Pyflakes, "821") => (RuleGroup::Unspecified, Rule::UndefinedName),
        (Pyflakes, "822") => (RuleGroup::Unspecified, Rule::UndefinedExport),
        (Pyflakes, "823") => (RuleGroup::Unspecified, Rule::UndefinedLocal),
        (Pyflakes, "841") => (RuleGroup::Unspecified, Rule::UnusedVariable),
        (Pyflakes, "842") => (RuleGroup::Unspecified, Rule::UnusedAnnotation),
        (Pyflakes, "901") => (RuleGroup::Unspecified, Rule::RaiseNotImplemented),

        // pylint
        (Pylint, "C0414") => (RuleGroup::Unspecified, Rule::UselessImportAlias),
        (Pylint, "C1901") => (RuleGroup::Unspecified, Rule::CompareToEmptyString),
        (Pylint, "C3002") => (RuleGroup::Unspecified, Rule::UnnecessaryDirectLambdaCall),
        (Pylint, "E0100") => (RuleGroup::Unspecified, Rule::YieldInInit),
        (Pylint, "E0101") => (RuleGroup::Unspecified, Rule::ReturnInInit),
        (Pylint, "E0116") => (RuleGroup::Unspecified, Rule::ContinueInFinally),
        (Pylint, "E0117") => (RuleGroup::Unspecified, Rule::NonlocalWithoutBinding),
        (Pylint, "E0118") => (RuleGroup::Unspecified, Rule::LoadBeforeGlobalDeclaration),
        (Pylint, "E0241") => (RuleGroup::Unspecified, Rule::DuplicateBases),
        (Pylint, "E0302") => (RuleGroup::Unspecified, Rule::UnexpectedSpecialMethodSignature),
        (Pylint, "E0604") => (RuleGroup::Unspecified, Rule::InvalidAllObject),
        (Pylint, "E0605") => (RuleGroup::Unspecified, Rule::InvalidAllFormat),
        (Pylint, "E1142") => (RuleGroup::Unspecified, Rule::AwaitOutsideAsync),
        (Pylint, "E1205") => (RuleGroup::Unspecified, Rule::LoggingTooManyArgs),
        (Pylint, "E1206") => (RuleGroup::Unspecified, Rule::LoggingTooFewArgs),
        (Pylint, "E1307") => (RuleGroup::Unspecified, Rule::BadStringFormatType),
        (Pylint, "E1310") => (RuleGroup::Unspecified, Rule::BadStrStripCall),
        (Pylint, "E1507") => (RuleGroup::Unspecified, Rule::InvalidEnvvarValue),
        (Pylint, "E2502") => (RuleGroup::Unspecified, Rule::BidirectionalUnicode),
        (Pylint, "E2510") => (RuleGroup::Unspecified, Rule::InvalidCharacterBackspace),
        (Pylint, "E2512") => (RuleGroup::Unspecified, Rule::InvalidCharacterSub),
        (Pylint, "E2513") => (RuleGroup::Unspecified, Rule::InvalidCharacterEsc),
        (Pylint, "E2514") => (RuleGroup::Unspecified, Rule::InvalidCharacterNul),
        (Pylint, "E2515") => (RuleGroup::Unspecified, Rule::InvalidCharacterZeroWidthSpace),
        (Pylint, "R0133") => (RuleGroup::Unspecified, Rule::ComparisonOfConstant),
        (Pylint, "R0206") => (RuleGroup::Unspecified, Rule::PropertyWithParameters),
        (Pylint, "R0402") => (RuleGroup::Unspecified, Rule::ManualFromImport),
        (Pylint, "R0911") => (RuleGroup::Unspecified, Rule::TooManyReturnStatements),
        (Pylint, "R0912") => (RuleGroup::Unspecified, Rule::TooManyBranches),
        (Pylint, "R0913") => (RuleGroup::Unspecified, Rule::TooManyArguments),
        (Pylint, "R0915") => (RuleGroup::Unspecified, Rule::TooManyStatements),
        (Pylint, "R1701") => (RuleGroup::Unspecified, Rule::RepeatedIsinstanceCalls),
        (Pylint, "R1711") => (RuleGroup::Unspecified, Rule::UselessReturn),
        (Pylint, "R1722") => (RuleGroup::Unspecified, Rule::SysExitAlias),
        (Pylint, "R2004") => (RuleGroup::Unspecified, Rule::MagicValueComparison),
        (Pylint, "R5501") => (RuleGroup::Unspecified, Rule::CollapsibleElseIf),
        (Pylint, "W0120") => (RuleGroup::Unspecified, Rule::UselessElseOnLoop),
        (Pylint, "W0129") => (RuleGroup::Unspecified, Rule::AssertOnStringLiteral),
        (Pylint, "W0406") => (RuleGroup::Unspecified, Rule::ImportSelf),
        (Pylint, "W0602") => (RuleGroup::Unspecified, Rule::GlobalVariableNotAssigned),
        (Pylint, "W0603") => (RuleGroup::Unspecified, Rule::GlobalStatement),
        (Pylint, "W0711") => (RuleGroup::Unspecified, Rule::BinaryOpException),
        (Pylint, "W1508") => (RuleGroup::Unspecified, Rule::InvalidEnvvarDefault),
        (Pylint, "W2901") => (RuleGroup::Unspecified, Rule::RedefinedLoopName),
        (Pylint, "W3301") => (RuleGroup::Unspecified, Rule::NestedMinMax),

        // flake8-async
        (Flake8Async, "100") => (RuleGroup::Unspecified, Rule::BlockingHttpCallInAsyncFunction),
        (Flake8Async, "101") => (RuleGroup::Unspecified, Rule::OpenSleepOrSubprocessInAsyncFunction),
        (Flake8Async, "102") => (RuleGroup::Unspecified, Rule::BlockingOsCallInAsyncFunction),

        // flake8-builtins
        (Flake8Builtins, "001") => (RuleGroup::Unspecified, Rule::BuiltinVariableShadowing),
        (Flake8Builtins, "002") => (RuleGroup::Unspecified, Rule::BuiltinArgumentShadowing),
        (Flake8Builtins, "003") => (RuleGroup::Unspecified, Rule::BuiltinAttributeShadowing),

        // flake8-bugbear
        (Flake8Bugbear, "002") => (RuleGroup::Unspecified, Rule::UnaryPrefixIncrement),
        (Flake8Bugbear, "003") => (RuleGroup::Unspecified, Rule::AssignmentToOsEnviron),
        (Flake8Bugbear, "004") => (RuleGroup::Unspecified, Rule::UnreliableCallableCheck),
        (Flake8Bugbear, "005") => (RuleGroup::Unspecified, Rule::StripWithMultiCharacters),
        (Flake8Bugbear, "006") => (RuleGroup::Unspecified, Rule::MutableArgumentDefault),
        (Flake8Bugbear, "007") => (RuleGroup::Unspecified, Rule::UnusedLoopControlVariable),
        (Flake8Bugbear, "008") => (RuleGroup::Unspecified, Rule::FunctionCallInDefaultArgument),
        (Flake8Bugbear, "009") => (RuleGroup::Unspecified, Rule::GetAttrWithConstant),
        (Flake8Bugbear, "010") => (RuleGroup::Unspecified, Rule::SetAttrWithConstant),
        (Flake8Bugbear, "011") => (RuleGroup::Unspecified, Rule::AssertFalse),
        (Flake8Bugbear, "012") => (RuleGroup::Unspecified, Rule::JumpStatementInFinally),
        (Flake8Bugbear, "013") => (RuleGroup::Unspecified, Rule::RedundantTupleInExceptionHandler),
        (Flake8Bugbear, "014") => (RuleGroup::Unspecified, Rule::DuplicateHandlerException),
        (Flake8Bugbear, "015") => (RuleGroup::Unspecified, Rule::UselessComparison),
        (Flake8Bugbear, "016") => (RuleGroup::Unspecified, Rule::CannotRaiseLiteral),
        (Flake8Bugbear, "017") => (RuleGroup::Unspecified, Rule::AssertRaisesException),
        (Flake8Bugbear, "018") => (RuleGroup::Unspecified, Rule::UselessExpression),
        (Flake8Bugbear, "019") => (RuleGroup::Unspecified, Rule::CachedInstanceMethod),
        (Flake8Bugbear, "020") => (RuleGroup::Unspecified, Rule::LoopVariableOverridesIterator),
        (Flake8Bugbear, "021") => (RuleGroup::Unspecified, Rule::FStringDocstring),
        (Flake8Bugbear, "022") => (RuleGroup::Unspecified, Rule::UselessContextlibSuppress),
        (Flake8Bugbear, "023") => (RuleGroup::Unspecified, Rule::FunctionUsesLoopVariable),
        (Flake8Bugbear, "024") => (RuleGroup::Unspecified, Rule::AbstractBaseClassWithoutAbstractMethod),
        (Flake8Bugbear, "025") => (RuleGroup::Unspecified, Rule::DuplicateTryBlockException),
        (Flake8Bugbear, "026") => (RuleGroup::Unspecified, Rule::StarArgUnpackingAfterKeywordArg),
        (Flake8Bugbear, "027") => (RuleGroup::Unspecified, Rule::EmptyMethodWithoutAbstractDecorator),
        (Flake8Bugbear, "028") => (RuleGroup::Unspecified, Rule::NoExplicitStacklevel),
        (Flake8Bugbear, "029") => (RuleGroup::Unspecified, Rule::ExceptWithEmptyTuple),
        (Flake8Bugbear, "030") => (RuleGroup::Unspecified, Rule::ExceptWithNonExceptionClasses),
        (Flake8Bugbear, "031") => (RuleGroup::Unspecified, Rule::ReuseOfGroupbyGenerator),
        (Flake8Bugbear, "032") => (RuleGroup::Unspecified, Rule::UnintentionalTypeAnnotation),
        (Flake8Bugbear, "904") => (RuleGroup::Unspecified, Rule::RaiseWithoutFromInsideExcept),
        (Flake8Bugbear, "905") => (RuleGroup::Unspecified, Rule::ZipWithoutExplicitStrict),

        // flake8-blind-except
        (Flake8BlindExcept, "001") => (RuleGroup::Unspecified, Rule::BlindExcept),

        // flake8-comprehensions
        (Flake8Comprehensions, "00") => (RuleGroup::Unspecified, Rule::UnnecessaryGeneratorList),
        (Flake8Comprehensions, "01") => (RuleGroup::Unspecified, Rule::UnnecessaryGeneratorSet),
        (Flake8Comprehensions, "02") => (RuleGroup::Unspecified, Rule::UnnecessaryGeneratorDict),
        (Flake8Comprehensions, "03") => (RuleGroup::Unspecified, Rule::UnnecessaryListComprehensionSet),
        (Flake8Comprehensions, "04") => (RuleGroup::Unspecified, Rule::UnnecessaryListComprehensionDict),
        (Flake8Comprehensions, "05") => (RuleGroup::Unspecified, Rule::UnnecessaryLiteralSet),
        (Flake8Comprehensions, "06") => (RuleGroup::Unspecified, Rule::UnnecessaryLiteralDict),
        (Flake8Comprehensions, "08") => (RuleGroup::Unspecified, Rule::UnnecessaryCollectionCall),
        (Flake8Comprehensions, "09") => (RuleGroup::Unspecified, Rule::UnnecessaryLiteralWithinTupleCall),
        (Flake8Comprehensions, "10") => (RuleGroup::Unspecified, Rule::UnnecessaryLiteralWithinListCall),
        (Flake8Comprehensions, "11") => (RuleGroup::Unspecified, Rule::UnnecessaryListCall),
        (Flake8Comprehensions, "13") => (RuleGroup::Unspecified, Rule::UnnecessaryCallAroundSorted),
        (Flake8Comprehensions, "14") => (RuleGroup::Unspecified, Rule::UnnecessaryDoubleCastOrProcess),
        (Flake8Comprehensions, "15") => (RuleGroup::Unspecified, Rule::UnnecessarySubscriptReversal),
        (Flake8Comprehensions, "16") => (RuleGroup::Unspecified, Rule::UnnecessaryComprehension),
        (Flake8Comprehensions, "17") => (RuleGroup::Unspecified, Rule::UnnecessaryMap),
        (Flake8Comprehensions, "18") => (RuleGroup::Unspecified, Rule::UnnecessaryLiteralWithinDictCall),
        (Flake8Comprehensions, "19") => (RuleGroup::Unspecified, Rule::UnnecessaryComprehensionAnyAll),

        // flake8-debugger
        (Flake8Debugger, "0") => (RuleGroup::Unspecified, Rule::Debugger),

        // mccabe
        (McCabe, "1") => (RuleGroup::Unspecified, Rule::ComplexStructure),

        // flake8-tidy-imports
        (Flake8TidyImports, "251") => (RuleGroup::Unspecified, Rule::BannedApi),
        (Flake8TidyImports, "252") => (RuleGroup::Unspecified, Rule::RelativeImports),

        // flake8-return
        (Flake8Return, "501") => (RuleGroup::Unspecified, Rule::UnnecessaryReturnNone),
        (Flake8Return, "502") => (RuleGroup::Unspecified, Rule::ImplicitReturnValue),
        (Flake8Return, "503") => (RuleGroup::Unspecified, Rule::ImplicitReturn),
        (Flake8Return, "504") => (RuleGroup::Unspecified, Rule::UnnecessaryAssign),
        (Flake8Return, "505") => (RuleGroup::Unspecified, Rule::SuperfluousElseReturn),
        (Flake8Return, "506") => (RuleGroup::Unspecified, Rule::SuperfluousElseRaise),
        (Flake8Return, "507") => (RuleGroup::Unspecified, Rule::SuperfluousElseContinue),
        (Flake8Return, "508") => (RuleGroup::Unspecified, Rule::SuperfluousElseBreak),

        // flake8-gettext
        (Flake8GetText, "001") => (RuleGroup::Unspecified, Rule::FStringInGetTextFuncCall),
        (Flake8GetText, "002") => (RuleGroup::Unspecified, Rule::FormatInGetTextFuncCall),
        (Flake8GetText, "003") => (RuleGroup::Unspecified, Rule::PrintfInGetTextFuncCall),

        // flake8-implicit-str-concat
        (Flake8ImplicitStrConcat, "001") => (RuleGroup::Unspecified, Rule::SingleLineImplicitStringConcatenation),
        (Flake8ImplicitStrConcat, "002") => (RuleGroup::Unspecified, Rule::MultiLineImplicitStringConcatenation),
        (Flake8ImplicitStrConcat, "003") => (RuleGroup::Unspecified, Rule::ExplicitStringConcatenation),

        // flake8-print
        (Flake8Print, "1") => (RuleGroup::Unspecified, Rule::Print),
        (Flake8Print, "3") => (RuleGroup::Unspecified, Rule::PPrint),

        // flake8-quotes
        (Flake8Quotes, "000") => (RuleGroup::Unspecified, Rule::BadQuotesInlineString),
        (Flake8Quotes, "001") => (RuleGroup::Unspecified, Rule::BadQuotesMultilineString),
        (Flake8Quotes, "002") => (RuleGroup::Unspecified, Rule::BadQuotesDocstring),
        (Flake8Quotes, "003") => (RuleGroup::Unspecified, Rule::AvoidableEscapedQuote),

        // flake8-annotations
        (Flake8Annotations, "001") => (RuleGroup::Unspecified, Rule::MissingTypeFunctionArgument),
        (Flake8Annotations, "002") => (RuleGroup::Unspecified, Rule::MissingTypeArgs),
        (Flake8Annotations, "003") => (RuleGroup::Unspecified, Rule::MissingTypeKwargs),
        (Flake8Annotations, "101") => (RuleGroup::Unspecified, Rule::MissingTypeSelf),
        (Flake8Annotations, "102") => (RuleGroup::Unspecified, Rule::MissingTypeCls),
        (Flake8Annotations, "201") => (RuleGroup::Unspecified, Rule::MissingReturnTypeUndocumentedPublicFunction),
        (Flake8Annotations, "202") => (RuleGroup::Unspecified, Rule::MissingReturnTypePrivateFunction),
        (Flake8Annotations, "204") => (RuleGroup::Unspecified, Rule::MissingReturnTypeSpecialMethod),
        (Flake8Annotations, "205") => (RuleGroup::Unspecified, Rule::MissingReturnTypeStaticMethod),
        (Flake8Annotations, "206") => (RuleGroup::Unspecified, Rule::MissingReturnTypeClassMethod),
        (Flake8Annotations, "401") => (RuleGroup::Unspecified, Rule::AnyType),

        // flake8-future-annotations
        (Flake8FutureAnnotations, "100") => (RuleGroup::Unspecified, Rule::MissingFutureAnnotationsImport),

        // flake8-2020
        (Flake82020, "101") => (RuleGroup::Unspecified, Rule::SysVersionSlice3),
        (Flake82020, "102") => (RuleGroup::Unspecified, Rule::SysVersion2),
        (Flake82020, "103") => (RuleGroup::Unspecified, Rule::SysVersionCmpStr3),
        (Flake82020, "201") => (RuleGroup::Unspecified, Rule::SysVersionInfo0Eq3),
        (Flake82020, "202") => (RuleGroup::Unspecified, Rule::SixPY3),
        (Flake82020, "203") => (RuleGroup::Unspecified, Rule::SysVersionInfo1CmpInt),
        (Flake82020, "204") => (RuleGroup::Unspecified, Rule::SysVersionInfoMinorCmpInt),
        (Flake82020, "301") => (RuleGroup::Unspecified, Rule::SysVersion0),
        (Flake82020, "302") => (RuleGroup::Unspecified, Rule::SysVersionCmpStr10),
        (Flake82020, "303") => (RuleGroup::Unspecified, Rule::SysVersionSlice1),

        // flake8-simplify
        (Flake8Simplify, "101") => (RuleGroup::Unspecified, Rule::DuplicateIsinstanceCall),
        (Flake8Simplify, "102") => (RuleGroup::Unspecified, Rule::CollapsibleIf),
        (Flake8Simplify, "103") => (RuleGroup::Unspecified, Rule::NeedlessBool),
        (Flake8Simplify, "105") => (RuleGroup::Unspecified, Rule::SuppressibleException),
        (Flake8Simplify, "107") => (RuleGroup::Unspecified, Rule::ReturnInTryExceptFinally),
        (Flake8Simplify, "108") => (RuleGroup::Unspecified, Rule::IfElseBlockInsteadOfIfExp),
        (Flake8Simplify, "109") => (RuleGroup::Unspecified, Rule::CompareWithTuple),
        (Flake8Simplify, "110") => (RuleGroup::Unspecified, Rule::ReimplementedBuiltin),
        (Flake8Simplify, "112") => (RuleGroup::Unspecified, Rule::UncapitalizedEnvironmentVariables),
        (Flake8Simplify, "114") => (RuleGroup::Unspecified, Rule::IfWithSameArms),
        (Flake8Simplify, "115") => (RuleGroup::Unspecified, Rule::OpenFileWithContextHandler),
        (Flake8Simplify, "116") => (RuleGroup::Unspecified, Rule::IfElseBlockInsteadOfDictLookup),
        (Flake8Simplify, "117") => (RuleGroup::Unspecified, Rule::MultipleWithStatements),
        (Flake8Simplify, "118") => (RuleGroup::Unspecified, Rule::InDictKeys),
        (Flake8Simplify, "201") => (RuleGroup::Unspecified, Rule::NegateEqualOp),
        (Flake8Simplify, "202") => (RuleGroup::Unspecified, Rule::NegateNotEqualOp),
        (Flake8Simplify, "208") => (RuleGroup::Unspecified, Rule::DoubleNegation),
        (Flake8Simplify, "210") => (RuleGroup::Unspecified, Rule::IfExprWithTrueFalse),
        (Flake8Simplify, "211") => (RuleGroup::Unspecified, Rule::IfExprWithFalseTrue),
        (Flake8Simplify, "212") => (RuleGroup::Unspecified, Rule::IfExprWithTwistedArms),
        (Flake8Simplify, "220") => (RuleGroup::Unspecified, Rule::ExprAndNotExpr),
        (Flake8Simplify, "221") => (RuleGroup::Unspecified, Rule::ExprOrNotExpr),
        (Flake8Simplify, "222") => (RuleGroup::Unspecified, Rule::ExprOrTrue),
        (Flake8Simplify, "223") => (RuleGroup::Unspecified, Rule::ExprAndFalse),
        (Flake8Simplify, "300") => (RuleGroup::Unspecified, Rule::YodaConditions),
        (Flake8Simplify, "401") => (RuleGroup::Unspecified, Rule::IfElseBlockInsteadOfDictGet),
        (Flake8Simplify, "910") => (RuleGroup::Unspecified, Rule::DictGetWithNoneDefault),

        // pyupgrade
        (Pyupgrade, "001") => (RuleGroup::Unspecified, Rule::UselessMetaclassType),
        (Pyupgrade, "003") => (RuleGroup::Unspecified, Rule::TypeOfPrimitive),
        (Pyupgrade, "004") => (RuleGroup::Unspecified, Rule::UselessObjectInheritance),
        (Pyupgrade, "005") => (RuleGroup::Unspecified, Rule::DeprecatedUnittestAlias),
        (Pyupgrade, "006") => (RuleGroup::Unspecified, Rule::NonPEP585Annotation),
        (Pyupgrade, "007") => (RuleGroup::Unspecified, Rule::NonPEP604Annotation),
        (Pyupgrade, "008") => (RuleGroup::Unspecified, Rule::SuperCallWithParameters),
        (Pyupgrade, "009") => (RuleGroup::Unspecified, Rule::UTF8EncodingDeclaration),
        (Pyupgrade, "010") => (RuleGroup::Unspecified, Rule::UnnecessaryFutureImport),
        (Pyupgrade, "011") => (RuleGroup::Unspecified, Rule::LRUCacheWithoutParameters),
        (Pyupgrade, "012") => (RuleGroup::Unspecified, Rule::UnnecessaryEncodeUTF8),
        (Pyupgrade, "013") => (RuleGroup::Unspecified, Rule::ConvertTypedDictFunctionalToClass),
        (Pyupgrade, "014") => (RuleGroup::Unspecified, Rule::ConvertNamedTupleFunctionalToClass),
        (Pyupgrade, "015") => (RuleGroup::Unspecified, Rule::RedundantOpenModes),
        (Pyupgrade, "017") => (RuleGroup::Unspecified, Rule::DatetimeTimezoneUTC),
        (Pyupgrade, "018") => (RuleGroup::Unspecified, Rule::NativeLiterals),
        (Pyupgrade, "019") => (RuleGroup::Unspecified, Rule::TypingTextStrAlias),
        (Pyupgrade, "020") => (RuleGroup::Unspecified, Rule::OpenAlias),
        (Pyupgrade, "021") => (RuleGroup::Unspecified, Rule::ReplaceUniversalNewlines),
        (Pyupgrade, "022") => (RuleGroup::Unspecified, Rule::ReplaceStdoutStderr),
        (Pyupgrade, "023") => (RuleGroup::Unspecified, Rule::DeprecatedCElementTree),
        (Pyupgrade, "024") => (RuleGroup::Unspecified, Rule::OSErrorAlias),
        (Pyupgrade, "025") => (RuleGroup::Unspecified, Rule::UnicodeKindPrefix),
        (Pyupgrade, "026") => (RuleGroup::Unspecified, Rule::DeprecatedMockImport),
        (Pyupgrade, "027") => (RuleGroup::Unspecified, Rule::UnpackedListComprehension),
        (Pyupgrade, "028") => (RuleGroup::Unspecified, Rule::YieldInForLoop),
        (Pyupgrade, "029") => (RuleGroup::Unspecified, Rule::UnnecessaryBuiltinImport),
        (Pyupgrade, "030") => (RuleGroup::Unspecified, Rule::FormatLiterals),
        (Pyupgrade, "031") => (RuleGroup::Unspecified, Rule::PrintfStringFormatting),
        (Pyupgrade, "032") => (RuleGroup::Unspecified, Rule::FString),
        (Pyupgrade, "033") => (RuleGroup::Unspecified, Rule::LRUCacheWithMaxsizeNone),
        (Pyupgrade, "034") => (RuleGroup::Unspecified, Rule::ExtraneousParentheses),
        (Pyupgrade, "035") => (RuleGroup::Unspecified, Rule::DeprecatedImport),
        (Pyupgrade, "036") => (RuleGroup::Unspecified, Rule::OutdatedVersionBlock),
        (Pyupgrade, "037") => (RuleGroup::Unspecified, Rule::QuotedAnnotation),
        (Pyupgrade, "038") => (RuleGroup::Unspecified, Rule::NonPEP604Isinstance),

        // pydocstyle
        (Pydocstyle, "100") => (RuleGroup::Unspecified, Rule::UndocumentedPublicModule),
        (Pydocstyle, "101") => (RuleGroup::Unspecified, Rule::UndocumentedPublicClass),
        (Pydocstyle, "102") => (RuleGroup::Unspecified, Rule::UndocumentedPublicMethod),
        (Pydocstyle, "103") => (RuleGroup::Unspecified, Rule::UndocumentedPublicFunction),
        (Pydocstyle, "104") => (RuleGroup::Unspecified, Rule::UndocumentedPublicPackage),
        (Pydocstyle, "105") => (RuleGroup::Unspecified, Rule::UndocumentedMagicMethod),
        (Pydocstyle, "106") => (RuleGroup::Unspecified, Rule::UndocumentedPublicNestedClass),
        (Pydocstyle, "107") => (RuleGroup::Unspecified, Rule::UndocumentedPublicInit),
        (Pydocstyle, "200") => (RuleGroup::Unspecified, Rule::FitsOnOneLine),
        (Pydocstyle, "201") => (RuleGroup::Unspecified, Rule::NoBlankLineBeforeFunction),
        (Pydocstyle, "202") => (RuleGroup::Unspecified, Rule::NoBlankLineAfterFunction),
        (Pydocstyle, "203") => (RuleGroup::Unspecified, Rule::OneBlankLineBeforeClass),
        (Pydocstyle, "204") => (RuleGroup::Unspecified, Rule::OneBlankLineAfterClass),
        (Pydocstyle, "205") => (RuleGroup::Unspecified, Rule::BlankLineAfterSummary),
        (Pydocstyle, "206") => (RuleGroup::Unspecified, Rule::IndentWithSpaces),
        (Pydocstyle, "207") => (RuleGroup::Unspecified, Rule::UnderIndentation),
        (Pydocstyle, "208") => (RuleGroup::Unspecified, Rule::OverIndentation),
        (Pydocstyle, "209") => (RuleGroup::Unspecified, Rule::NewLineAfterLastParagraph),
        (Pydocstyle, "210") => (RuleGroup::Unspecified, Rule::SurroundingWhitespace),
        (Pydocstyle, "211") => (RuleGroup::Unspecified, Rule::BlankLineBeforeClass),
        (Pydocstyle, "212") => (RuleGroup::Unspecified, Rule::MultiLineSummaryFirstLine),
        (Pydocstyle, "213") => (RuleGroup::Unspecified, Rule::MultiLineSummarySecondLine),
        (Pydocstyle, "214") => (RuleGroup::Unspecified, Rule::SectionNotOverIndented),
        (Pydocstyle, "215") => (RuleGroup::Unspecified, Rule::SectionUnderlineNotOverIndented),
        (Pydocstyle, "300") => (RuleGroup::Unspecified, Rule::TripleSingleQuotes),
        (Pydocstyle, "301") => (RuleGroup::Unspecified, Rule::EscapeSequenceInDocstring),
        (Pydocstyle, "400") => (RuleGroup::Unspecified, Rule::EndsInPeriod),
        (Pydocstyle, "401") => (RuleGroup::Unspecified, Rule::NonImperativeMood),
        (Pydocstyle, "402") => (RuleGroup::Unspecified, Rule::NoSignature),
        (Pydocstyle, "403") => (RuleGroup::Unspecified, Rule::FirstLineCapitalized),
        (Pydocstyle, "404") => (RuleGroup::Unspecified, Rule::DocstringStartsWithThis),
        (Pydocstyle, "405") => (RuleGroup::Unspecified, Rule::CapitalizeSectionName),
        (Pydocstyle, "406") => (RuleGroup::Unspecified, Rule::NewLineAfterSectionName),
        (Pydocstyle, "407") => (RuleGroup::Unspecified, Rule::DashedUnderlineAfterSection),
        (Pydocstyle, "408") => (RuleGroup::Unspecified, Rule::SectionUnderlineAfterName),
        (Pydocstyle, "409") => (RuleGroup::Unspecified, Rule::SectionUnderlineMatchesSectionLength),
        (Pydocstyle, "410") => (RuleGroup::Unspecified, Rule::NoBlankLineAfterSection),
        (Pydocstyle, "411") => (RuleGroup::Unspecified, Rule::NoBlankLineBeforeSection),
        (Pydocstyle, "412") => (RuleGroup::Unspecified, Rule::BlankLinesBetweenHeaderAndContent),
        (Pydocstyle, "413") => (RuleGroup::Unspecified, Rule::BlankLineAfterLastSection),
        (Pydocstyle, "414") => (RuleGroup::Unspecified, Rule::EmptyDocstringSection),
        (Pydocstyle, "415") => (RuleGroup::Unspecified, Rule::EndsInPunctuation),
        (Pydocstyle, "416") => (RuleGroup::Unspecified, Rule::SectionNameEndsInColon),
        (Pydocstyle, "417") => (RuleGroup::Unspecified, Rule::UndocumentedParam),
        (Pydocstyle, "418") => (RuleGroup::Unspecified, Rule::OverloadWithDocstring),
        (Pydocstyle, "419") => (RuleGroup::Unspecified, Rule::EmptyDocstring),

        // pep8-naming
        (PEP8Naming, "801") => (RuleGroup::Unspecified, Rule::InvalidClassName),
        (PEP8Naming, "802") => (RuleGroup::Unspecified, Rule::InvalidFunctionName),
        (PEP8Naming, "803") => (RuleGroup::Unspecified, Rule::InvalidArgumentName),
        (PEP8Naming, "804") => (RuleGroup::Unspecified, Rule::InvalidFirstArgumentNameForClassMethod),
        (PEP8Naming, "805") => (RuleGroup::Unspecified, Rule::InvalidFirstArgumentNameForMethod),
        (PEP8Naming, "806") => (RuleGroup::Unspecified, Rule::NonLowercaseVariableInFunction),
        (PEP8Naming, "807") => (RuleGroup::Unspecified, Rule::DunderFunctionName),
        (PEP8Naming, "811") => (RuleGroup::Unspecified, Rule::ConstantImportedAsNonConstant),
        (PEP8Naming, "812") => (RuleGroup::Unspecified, Rule::LowercaseImportedAsNonLowercase),
        (PEP8Naming, "813") => (RuleGroup::Unspecified, Rule::CamelcaseImportedAsLowercase),
        (PEP8Naming, "814") => (RuleGroup::Unspecified, Rule::CamelcaseImportedAsConstant),
        (PEP8Naming, "815") => (RuleGroup::Unspecified, Rule::MixedCaseVariableInClassScope),
        (PEP8Naming, "816") => (RuleGroup::Unspecified, Rule::MixedCaseVariableInGlobalScope),
        (PEP8Naming, "817") => (RuleGroup::Unspecified, Rule::CamelcaseImportedAsAcronym),
        (PEP8Naming, "818") => (RuleGroup::Unspecified, Rule::ErrorSuffixOnExceptionName),
        (PEP8Naming, "999") => (RuleGroup::Unspecified, Rule::InvalidModuleName),

        // isort
        (Isort, "001") => (RuleGroup::Unspecified, Rule::UnsortedImports),
        (Isort, "002") => (RuleGroup::Unspecified, Rule::MissingRequiredImport),

        // eradicate
        (Eradicate, "001") => (RuleGroup::Unspecified, Rule::CommentedOutCode),

        // flake8-bandit
        (Flake8Bandit, "101") => (RuleGroup::Unspecified, Rule::Assert),
        (Flake8Bandit, "102") => (RuleGroup::Unspecified, Rule::ExecBuiltin),
        (Flake8Bandit, "103") => (RuleGroup::Unspecified, Rule::BadFilePermissions),
        (Flake8Bandit, "104") => (RuleGroup::Unspecified, Rule::HardcodedBindAllInterfaces),
        (Flake8Bandit, "105") => (RuleGroup::Unspecified, Rule::HardcodedPasswordString),
        (Flake8Bandit, "106") => (RuleGroup::Unspecified, Rule::HardcodedPasswordFuncArg),
        (Flake8Bandit, "107") => (RuleGroup::Unspecified, Rule::HardcodedPasswordDefault),
        (Flake8Bandit, "108") => (RuleGroup::Unspecified, Rule::HardcodedTempFile),
        (Flake8Bandit, "110") => (RuleGroup::Unspecified, Rule::TryExceptPass),
        (Flake8Bandit, "112") => (RuleGroup::Unspecified, Rule::TryExceptContinue),
        (Flake8Bandit, "113") => (RuleGroup::Unspecified, Rule::RequestWithoutTimeout),
        (Flake8Bandit, "301") => (RuleGroup::Unspecified, Rule::SuspiciousPickleUsage),
        (Flake8Bandit, "302") => (RuleGroup::Unspecified, Rule::SuspiciousMarshalUsage),
        (Flake8Bandit, "303") => (RuleGroup::Unspecified, Rule::SuspiciousInsecureHashUsage),
        (Flake8Bandit, "304") => (RuleGroup::Unspecified, Rule::SuspiciousInsecureCipherUsage),
        (Flake8Bandit, "305") => (RuleGroup::Unspecified, Rule::SuspiciousInsecureCipherModeUsage),
        (Flake8Bandit, "306") => (RuleGroup::Unspecified, Rule::SuspiciousMktempUsage),
        (Flake8Bandit, "307") => (RuleGroup::Unspecified, Rule::SuspiciousEvalUsage),
        (Flake8Bandit, "308") => (RuleGroup::Unspecified, Rule::SuspiciousMarkSafeUsage),
        (Flake8Bandit, "310") => (RuleGroup::Unspecified, Rule::SuspiciousURLOpenUsage),
        (Flake8Bandit, "311") => (RuleGroup::Unspecified, Rule::SuspiciousNonCryptographicRandomUsage),
        (Flake8Bandit, "312") => (RuleGroup::Unspecified, Rule::SuspiciousTelnetUsage),
        (Flake8Bandit, "313") => (RuleGroup::Unspecified, Rule::SuspiciousXMLCElementTreeUsage),
        (Flake8Bandit, "314") => (RuleGroup::Unspecified, Rule::SuspiciousXMLElementTreeUsage),
        (Flake8Bandit, "315") => (RuleGroup::Unspecified, Rule::SuspiciousXMLExpatReaderUsage),
        (Flake8Bandit, "316") => (RuleGroup::Unspecified, Rule::SuspiciousXMLExpatBuilderUsage),
        (Flake8Bandit, "317") => (RuleGroup::Unspecified, Rule::SuspiciousXMLSaxUsage),
        (Flake8Bandit, "318") => (RuleGroup::Unspecified, Rule::SuspiciousXMLMiniDOMUsage),
        (Flake8Bandit, "319") => (RuleGroup::Unspecified, Rule::SuspiciousXMLPullDOMUsage),
        (Flake8Bandit, "320") => (RuleGroup::Unspecified, Rule::SuspiciousXMLETreeUsage),
        (Flake8Bandit, "321") => (RuleGroup::Unspecified, Rule::SuspiciousFTPLibUsage),
        (Flake8Bandit, "323") => (RuleGroup::Unspecified, Rule::SuspiciousUnverifiedContextUsage),
        (Flake8Bandit, "324") => (RuleGroup::Unspecified, Rule::HashlibInsecureHashFunction),
        (Flake8Bandit, "501") => (RuleGroup::Unspecified, Rule::RequestWithNoCertValidation),
        (Flake8Bandit, "506") => (RuleGroup::Unspecified, Rule::UnsafeYAMLLoad),
        (Flake8Bandit, "508") => (RuleGroup::Unspecified, Rule::SnmpInsecureVersion),
        (Flake8Bandit, "509") => (RuleGroup::Unspecified, Rule::SnmpWeakCryptography),
        (Flake8Bandit, "602") => (RuleGroup::Unspecified, Rule::SubprocessPopenWithShellEqualsTrue),
        (Flake8Bandit, "603") => (RuleGroup::Unspecified, Rule::SubprocessWithoutShellEqualsTrue),
        (Flake8Bandit, "604") => (RuleGroup::Unspecified, Rule::CallWithShellEqualsTrue),
        (Flake8Bandit, "605") => (RuleGroup::Unspecified, Rule::StartProcessWithAShell),
        (Flake8Bandit, "606") => (RuleGroup::Unspecified, Rule::StartProcessWithNoShell),
        (Flake8Bandit, "607") => (RuleGroup::Unspecified, Rule::StartProcessWithPartialPath),
        (Flake8Bandit, "608") => (RuleGroup::Unspecified, Rule::HardcodedSQLExpression),
        (Flake8Bandit, "612") => (RuleGroup::Unspecified, Rule::LoggingConfigInsecureListen),
        (Flake8Bandit, "701") => (RuleGroup::Unspecified, Rule::Jinja2AutoescapeFalse),

        // flake8-boolean-trap
        (Flake8BooleanTrap, "001") => (RuleGroup::Unspecified, Rule::BooleanPositionalArgInFunctionDefinition),
        (Flake8BooleanTrap, "002") => (RuleGroup::Unspecified, Rule::BooleanDefaultValueInFunctionDefinition),
        (Flake8BooleanTrap, "003") => (RuleGroup::Unspecified, Rule::BooleanPositionalValueInFunctionCall),

        // flake8-unused-arguments
        (Flake8UnusedArguments, "001") => (RuleGroup::Unspecified, Rule::UnusedFunctionArgument),
        (Flake8UnusedArguments, "002") => (RuleGroup::Unspecified, Rule::UnusedMethodArgument),
        (Flake8UnusedArguments, "003") => (RuleGroup::Unspecified, Rule::UnusedClassMethodArgument),
        (Flake8UnusedArguments, "004") => (RuleGroup::Unspecified, Rule::UnusedStaticMethodArgument),
        (Flake8UnusedArguments, "005") => (RuleGroup::Unspecified, Rule::UnusedLambdaArgument),

        // flake8-import-conventions
        (Flake8ImportConventions, "001") => (RuleGroup::Unspecified, Rule::UnconventionalImportAlias),
        (Flake8ImportConventions, "002") => (RuleGroup::Unspecified, Rule::BannedImportAlias),
        (Flake8ImportConventions, "003") => (RuleGroup::Unspecified, Rule::BannedImportFrom),

        // flake8-datetimez
        (Flake8Datetimez, "001") => (RuleGroup::Unspecified, Rule::CallDatetimeWithoutTzinfo),
        (Flake8Datetimez, "002") => (RuleGroup::Unspecified, Rule::CallDatetimeToday),
        (Flake8Datetimez, "003") => (RuleGroup::Unspecified, Rule::CallDatetimeUtcnow),
        (Flake8Datetimez, "004") => (RuleGroup::Unspecified, Rule::CallDatetimeUtcfromtimestamp),
        (Flake8Datetimez, "005") => (RuleGroup::Unspecified, Rule::CallDatetimeNowWithoutTzinfo),
        (Flake8Datetimez, "006") => (RuleGroup::Unspecified, Rule::CallDatetimeFromtimestamp),
        (Flake8Datetimez, "007") => (RuleGroup::Unspecified, Rule::CallDatetimeStrptimeWithoutZone),
        (Flake8Datetimez, "011") => (RuleGroup::Unspecified, Rule::CallDateToday),
        (Flake8Datetimez, "012") => (RuleGroup::Unspecified, Rule::CallDateFromtimestamp),

        // pygrep-hooks
        (PygrepHooks, "001") => (RuleGroup::Unspecified, Rule::Eval),
        (PygrepHooks, "002") => (RuleGroup::Unspecified, Rule::DeprecatedLogWarn),
        (PygrepHooks, "003") => (RuleGroup::Unspecified, Rule::BlanketTypeIgnore),
        (PygrepHooks, "004") => (RuleGroup::Unspecified, Rule::BlanketNOQA),
        (PygrepHooks, "005") => (RuleGroup::Unspecified, Rule::InvalidMockAccess),

        // pandas-vet
        (PandasVet, "002") => (RuleGroup::Unspecified, Rule::PandasUseOfInplaceArgument),
        (PandasVet, "003") => (RuleGroup::Unspecified, Rule::PandasUseOfDotIsNull),
        (PandasVet, "004") => (RuleGroup::Unspecified, Rule::PandasUseOfDotNotNull),
        (PandasVet, "007") => (RuleGroup::Unspecified, Rule::PandasUseOfDotIx),
        (PandasVet, "008") => (RuleGroup::Unspecified, Rule::PandasUseOfDotAt),
        (PandasVet, "009") => (RuleGroup::Unspecified, Rule::PandasUseOfDotIat),
        (PandasVet, "010") => (RuleGroup::Unspecified, Rule::PandasUseOfDotPivotOrUnstack),
        (PandasVet, "011") => (RuleGroup::Unspecified, Rule::PandasUseOfDotValues),
        (PandasVet, "012") => (RuleGroup::Unspecified, Rule::PandasUseOfDotReadTable),
        (PandasVet, "013") => (RuleGroup::Unspecified, Rule::PandasUseOfDotStack),
        (PandasVet, "015") => (RuleGroup::Unspecified, Rule::PandasUseOfPdMerge),
        (PandasVet, "901") => (RuleGroup::Unspecified, Rule::PandasDfVariableName),

        // flake8-errmsg
        (Flake8ErrMsg, "101") => (RuleGroup::Unspecified, Rule::RawStringInException),
        (Flake8ErrMsg, "102") => (RuleGroup::Unspecified, Rule::FStringInException),
        (Flake8ErrMsg, "103") => (RuleGroup::Unspecified, Rule::DotFormatInException),

        // flake8-pyi
        (Flake8Pyi, "001") => (RuleGroup::Unspecified, Rule::UnprefixedTypeParam),
        (Flake8Pyi, "006") => (RuleGroup::Unspecified, Rule::BadVersionInfoComparison),
        (Flake8Pyi, "007") => (RuleGroup::Unspecified, Rule::UnrecognizedPlatformCheck),
        (Flake8Pyi, "008") => (RuleGroup::Unspecified, Rule::UnrecognizedPlatformName),
        (Flake8Pyi, "009") => (RuleGroup::Unspecified, Rule::PassStatementStubBody),
        (Flake8Pyi, "010") => (RuleGroup::Unspecified, Rule::NonEmptyStubBody),
        (Flake8Pyi, "011") => (RuleGroup::Unspecified, Rule::TypedArgumentDefaultInStub),
        (Flake8Pyi, "012") => (RuleGroup::Unspecified, Rule::PassInClassBody),
        (Flake8Pyi, "014") => (RuleGroup::Unspecified, Rule::ArgumentDefaultInStub),
        (Flake8Pyi, "015") => (RuleGroup::Unspecified, Rule::AssignmentDefaultInStub),
        (Flake8Pyi, "016") => (RuleGroup::Unspecified, Rule::DuplicateUnionMember),
        (Flake8Pyi, "020") => (RuleGroup::Unspecified, Rule::QuotedAnnotationInStub),
        (Flake8Pyi, "021") => (RuleGroup::Unspecified, Rule::DocstringInStub),
        (Flake8Pyi, "033") => (RuleGroup::Unspecified, Rule::TypeCommentInStub),
        (Flake8Pyi, "042") => (RuleGroup::Unspecified, Rule::SnakeCaseTypeAlias),
        (Flake8Pyi, "043") => (RuleGroup::Unspecified, Rule::TSuffixedTypeAlias),
        (Flake8Pyi, "052") => (RuleGroup::Unspecified, Rule::UnannotatedAssignmentInStub),

        // flake8-pytest-style
        (Flake8PytestStyle, "001") => (RuleGroup::Unspecified, Rule::PytestFixtureIncorrectParenthesesStyle),
        (Flake8PytestStyle, "002") => (RuleGroup::Unspecified, Rule::PytestFixturePositionalArgs),
        (Flake8PytestStyle, "003") => (RuleGroup::Unspecified, Rule::PytestExtraneousScopeFunction),
        (Flake8PytestStyle, "004") => (RuleGroup::Unspecified, Rule::PytestMissingFixtureNameUnderscore),
        (Flake8PytestStyle, "005") => (RuleGroup::Unspecified, Rule::PytestIncorrectFixtureNameUnderscore),
        (Flake8PytestStyle, "006") => (RuleGroup::Unspecified, Rule::PytestParametrizeNamesWrongType),
        (Flake8PytestStyle, "007") => (RuleGroup::Unspecified, Rule::PytestParametrizeValuesWrongType),
        (Flake8PytestStyle, "008") => (RuleGroup::Unspecified, Rule::PytestPatchWithLambda),
        (Flake8PytestStyle, "009") => (RuleGroup::Unspecified, Rule::PytestUnittestAssertion),
        (Flake8PytestStyle, "010") => (RuleGroup::Unspecified, Rule::PytestRaisesWithoutException),
        (Flake8PytestStyle, "011") => (RuleGroup::Unspecified, Rule::PytestRaisesTooBroad),
        (Flake8PytestStyle, "012") => (RuleGroup::Unspecified, Rule::PytestRaisesWithMultipleStatements),
        (Flake8PytestStyle, "013") => (RuleGroup::Unspecified, Rule::PytestIncorrectPytestImport),
        (Flake8PytestStyle, "015") => (RuleGroup::Unspecified, Rule::PytestAssertAlwaysFalse),
        (Flake8PytestStyle, "016") => (RuleGroup::Unspecified, Rule::PytestFailWithoutMessage),
        (Flake8PytestStyle, "017") => (RuleGroup::Unspecified, Rule::PytestAssertInExcept),
        (Flake8PytestStyle, "018") => (RuleGroup::Unspecified, Rule::PytestCompositeAssertion),
        (Flake8PytestStyle, "019") => (RuleGroup::Unspecified, Rule::PytestFixtureParamWithoutValue),
        (Flake8PytestStyle, "020") => (RuleGroup::Unspecified, Rule::PytestDeprecatedYieldFixture),
        (Flake8PytestStyle, "021") => (RuleGroup::Unspecified, Rule::PytestFixtureFinalizerCallback),
        (Flake8PytestStyle, "022") => (RuleGroup::Unspecified, Rule::PytestUselessYieldFixture),
        (Flake8PytestStyle, "023") => (RuleGroup::Unspecified, Rule::PytestIncorrectMarkParenthesesStyle),
        (Flake8PytestStyle, "024") => (RuleGroup::Unspecified, Rule::PytestUnnecessaryAsyncioMarkOnFixture),
        (Flake8PytestStyle, "025") => (RuleGroup::Unspecified, Rule::PytestErroneousUseFixturesOnFixture),
        (Flake8PytestStyle, "026") => (RuleGroup::Unspecified, Rule::PytestUseFixturesWithoutParameters),

        // flake8-pie
        (Flake8Pie, "790") => (RuleGroup::Unspecified, Rule::UnnecessaryPass),
        (Flake8Pie, "794") => (RuleGroup::Unspecified, Rule::DuplicateClassFieldDefinition),
        (Flake8Pie, "796") => (RuleGroup::Unspecified, Rule::NonUniqueEnums),
        (Flake8Pie, "800") => (RuleGroup::Unspecified, Rule::UnnecessarySpread),
        (Flake8Pie, "804") => (RuleGroup::Unspecified, Rule::UnnecessaryDictKwargs),
        (Flake8Pie, "807") => (RuleGroup::Unspecified, Rule::ReimplementedListBuiltin),
        (Flake8Pie, "810") => (RuleGroup::Unspecified, Rule::MultipleStartsEndsWith),

        // flake8-commas
        (Flake8Commas, "812") => (RuleGroup::Unspecified, Rule::MissingTrailingComma),
        (Flake8Commas, "818") => (RuleGroup::Unspecified, Rule::TrailingCommaOnBareTuple),
        (Flake8Commas, "819") => (RuleGroup::Unspecified, Rule::ProhibitedTrailingComma),

        // flake8-no-pep420
        (Flake8NoPep420, "001") => (RuleGroup::Unspecified, Rule::ImplicitNamespacePackage),

        // flake8-executable
        (Flake8Executable, "001") => (RuleGroup::Unspecified, Rule::ShebangNotExecutable),
        (Flake8Executable, "002") => (RuleGroup::Unspecified, Rule::ShebangMissingExecutableFile),
        (Flake8Executable, "003") => (RuleGroup::Unspecified, Rule::ShebangMissingPython),
        (Flake8Executable, "004") => (RuleGroup::Unspecified, Rule::ShebangLeadingWhitespace),
        (Flake8Executable, "005") => (RuleGroup::Unspecified, Rule::ShebangNotFirstLine),

        // flake8-type-checking
        (Flake8TypeChecking, "001") => (RuleGroup::Unspecified, Rule::TypingOnlyFirstPartyImport),
        (Flake8TypeChecking, "002") => (RuleGroup::Unspecified, Rule::TypingOnlyThirdPartyImport),
        (Flake8TypeChecking, "003") => (RuleGroup::Unspecified, Rule::TypingOnlyStandardLibraryImport),
        (Flake8TypeChecking, "004") => (RuleGroup::Unspecified, Rule::RuntimeImportInTypeCheckingBlock),
        (Flake8TypeChecking, "005") => (RuleGroup::Unspecified, Rule::EmptyTypeCheckingBlock),

        // tryceratops
        (Tryceratops, "002") => (RuleGroup::Unspecified, Rule::RaiseVanillaClass),
        (Tryceratops, "003") => (RuleGroup::Unspecified, Rule::RaiseVanillaArgs),
        (Tryceratops, "004") => (RuleGroup::Unspecified, Rule::TypeCheckWithoutTypeError),
        (Tryceratops, "200") => (RuleGroup::Unspecified, Rule::ReraiseNoCause),
        (Tryceratops, "201") => (RuleGroup::Unspecified, Rule::VerboseRaise),
        (Tryceratops, "300") => (RuleGroup::Unspecified, Rule::TryConsiderElse),
        (Tryceratops, "301") => (RuleGroup::Unspecified, Rule::RaiseWithinTry),
        (Tryceratops, "302") => (RuleGroup::Unspecified, Rule::UselessTryExcept),
        (Tryceratops, "400") => (RuleGroup::Unspecified, Rule::ErrorInsteadOfException),
        (Tryceratops, "401") => (RuleGroup::Unspecified, Rule::VerboseLogMessage),

        // flake8-use-pathlib
        (Flake8UsePathlib, "100") => (RuleGroup::Unspecified, Rule::OsPathAbspath),
        (Flake8UsePathlib, "101") => (RuleGroup::Unspecified, Rule::OsChmod),
        (Flake8UsePathlib, "102") => (RuleGroup::Unspecified, Rule::OsMkdir),
        (Flake8UsePathlib, "103") => (RuleGroup::Unspecified, Rule::OsMakedirs),
        (Flake8UsePathlib, "104") => (RuleGroup::Unspecified, Rule::OsRename),
        (Flake8UsePathlib, "105") => (RuleGroup::Unspecified, Rule::PathlibReplace),
        (Flake8UsePathlib, "106") => (RuleGroup::Unspecified, Rule::OsRmdir),
        (Flake8UsePathlib, "107") => (RuleGroup::Unspecified, Rule::OsRemove),
        (Flake8UsePathlib, "108") => (RuleGroup::Unspecified, Rule::OsUnlink),
        (Flake8UsePathlib, "109") => (RuleGroup::Unspecified, Rule::OsGetcwd),
        (Flake8UsePathlib, "110") => (RuleGroup::Unspecified, Rule::OsPathExists),
        (Flake8UsePathlib, "111") => (RuleGroup::Unspecified, Rule::OsPathExpanduser),
        (Flake8UsePathlib, "112") => (RuleGroup::Unspecified, Rule::OsPathIsdir),
        (Flake8UsePathlib, "113") => (RuleGroup::Unspecified, Rule::OsPathIsfile),
        (Flake8UsePathlib, "114") => (RuleGroup::Unspecified, Rule::OsPathIslink),
        (Flake8UsePathlib, "115") => (RuleGroup::Unspecified, Rule::OsReadlink),
        (Flake8UsePathlib, "116") => (RuleGroup::Unspecified, Rule::OsStat),
        (Flake8UsePathlib, "117") => (RuleGroup::Unspecified, Rule::OsPathIsabs),
        (Flake8UsePathlib, "118") => (RuleGroup::Unspecified, Rule::OsPathJoin),
        (Flake8UsePathlib, "119") => (RuleGroup::Unspecified, Rule::OsPathBasename),
        (Flake8UsePathlib, "120") => (RuleGroup::Unspecified, Rule::OsPathDirname),
        (Flake8UsePathlib, "121") => (RuleGroup::Unspecified, Rule::OsPathSamefile),
        (Flake8UsePathlib, "122") => (RuleGroup::Unspecified, Rule::OsPathSplitext),
        (Flake8UsePathlib, "123") => (RuleGroup::Unspecified, Rule::BuiltinOpen),
        (Flake8UsePathlib, "124") => (RuleGroup::Unspecified, Rule::PyPath),

        // flake8-logging-format
        (Flake8LoggingFormat, "001") => (RuleGroup::Unspecified, Rule::LoggingStringFormat),
        (Flake8LoggingFormat, "002") => (RuleGroup::Unspecified, Rule::LoggingPercentFormat),
        (Flake8LoggingFormat, "003") => (RuleGroup::Unspecified, Rule::LoggingStringConcat),
        (Flake8LoggingFormat, "004") => (RuleGroup::Unspecified, Rule::LoggingFString),
        (Flake8LoggingFormat, "010") => (RuleGroup::Unspecified, Rule::LoggingWarn),
        (Flake8LoggingFormat, "101") => (RuleGroup::Unspecified, Rule::LoggingExtraAttrClash),
        (Flake8LoggingFormat, "201") => (RuleGroup::Unspecified, Rule::LoggingExcInfo),
        (Flake8LoggingFormat, "202") => (RuleGroup::Unspecified, Rule::LoggingRedundantExcInfo),

        // flake8-raise
        (Flake8Raise, "102") => (RuleGroup::Unspecified, Rule::UnnecessaryParenOnRaiseException),

        // flake8-self
        (Flake8Self, "001") => (RuleGroup::Unspecified, Rule::PrivateMemberAccess),

        // numpy
        (Numpy, "001") => (RuleGroup::Unspecified, Rule::NumpyDeprecatedTypeAlias),
        (Numpy, "002") => (RuleGroup::Unspecified, Rule::NumpyLegacyRandom),

        // ruff
        (Ruff, "001") => (RuleGroup::Unspecified, Rule::AmbiguousUnicodeCharacterString),
        (Ruff, "002") => (RuleGroup::Unspecified, Rule::AmbiguousUnicodeCharacterDocstring),
        (Ruff, "003") => (RuleGroup::Unspecified, Rule::AmbiguousUnicodeCharacterComment),
        (Ruff, "005") => (RuleGroup::Unspecified, Rule::CollectionLiteralConcatenation),
        (Ruff, "006") => (RuleGroup::Unspecified, Rule::AsyncioDanglingTask),
        (Ruff, "007") => (RuleGroup::Unspecified, Rule::PairwiseOverZipped),
        (Ruff, "008") => (RuleGroup::Unspecified, Rule::MutableDataclassDefault),
        (Ruff, "009") => (RuleGroup::Unspecified, Rule::FunctionCallInDataclassDefaultArgument),
        (Ruff, "010") => (RuleGroup::Unspecified, Rule::ExplicitFStringTypeConversion),
        (Ruff, "100") => (RuleGroup::Unspecified, Rule::UnusedNOQA),

        // flake8-django
        (Flake8Django, "001") => (RuleGroup::Unspecified, Rule::DjangoNullableModelStringField),
        (Flake8Django, "003") => (RuleGroup::Unspecified, Rule::DjangoLocalsInRenderFunction),
        (Flake8Django, "006") => (RuleGroup::Unspecified, Rule::DjangoExcludeWithModelForm),
        (Flake8Django, "007") => (RuleGroup::Unspecified, Rule::DjangoAllWithModelForm),
        (Flake8Django, "008") => (RuleGroup::Unspecified, Rule::DjangoModelWithoutDunderStr),
        (Flake8Django, "012") => (RuleGroup::Unspecified, Rule::DjangoUnorderedBodyContentInModel),
        (Flake8Django, "013") => (RuleGroup::Unspecified, Rule::DjangoNonLeadingReceiverDecorator),

        // flynt
        // Reserved: (Flynt, "001") => (RuleGroup::Unspecified, Rule::StringConcatenationToFString),
        (Flynt, "002") => (RuleGroup::Unspecified, Rule::StaticJoinToFString),

        // flake8-todos
        (Flake8Todo, "001") => (RuleGroup::Unspecified, Rule::InvalidTodoTag),
        (Flake8Todo, "002") => (RuleGroup::Unspecified, Rule::MissingTodoAuthor),
        (Flake8Todo, "003") => (RuleGroup::Unspecified, Rule::MissingTodoLink),
        (Flake8Todo, "004") => (RuleGroup::Unspecified, Rule::MissingTodoColon),
        (Flake8Todo, "005") => (RuleGroup::Unspecified, Rule::MissingTodoDescription),
        (Flake8Todo, "006") => (RuleGroup::Unspecified, Rule::InvalidTodoCapitalization),
        (Flake8Todo, "007") => (RuleGroup::Unspecified, Rule::MissingSpaceAfterTodoColon),

        _ => return None,
    })
}
