module TapKit

	class Entity
		attr_accessor :model, :name, :class_name, :attributes, :fetch_spec, \
			:class_properties, :locking_attributes, :parent, :class_property_names, \
			:locking_attribute_names, :primary_key_attribute_names, :external_name
		attr_reader :relationships, :primary_key_attributes, \
			:class_description, :sub_entities

		def initialize( list = {}, model = nil )
			@model              = model
			@attributes         = []
			@class_properties   = []
			@locking_attributes = []
			@relationships      = []
			@sub_entities       = []
			@name               = list['name']
			@class_name         = list['class_name'] || 'GenericRecord'
			@external_name      = list['external_name']
			@fetch_spec         = list['fetch_spec'] # unsupported
			@primary_key_attributes = []
			@primary_key_attribute_names  = list['primary_key_attributes'] || []
			@class_property_names = list['class_properties'] || []
			@locking_attribute_names = list['locking_attributes']

			list['attributes'] ||= []
			list['attributes'].each do |attr_list|
				name = attr_list['name']
				attr = Attribute.new(attr_list, self)
				add_attribute attr
			end

			@class_description = EntityClassDescription.new self
		end


		##
		## properties
		##

		def beautify_name
			if external_name then
				names = external_name.split '_'
				names.each { |name| name.capitalize! }
				@name = names.join
			end
		end

		# get attribute for name
		def attribute( name )
			attr = nil
			@attributes.each do |_attr|
				if _attr.name == name then
					attr = _attr
				end
			end
			attr
		end

		def add_attribute( attribute )
			if class_property? attribute.name then
				@class_properties << attribute
			end

			if primary_key_attribute? attribute.name then
				@primary_key_attributes << attribute
			end

			@attributes << attribute
		end

		def remove_attribute( attribute )
			@attributes.delete attribute
			@primary_key_attribute_names.delete attribute.name
		end

		def relationship( name )
			result = nil
			@relationships.each do |re|
				if re.name == name then
					result = re
				end
			end
			result
		end

		def add_relationship( relationship )
			if class_property? relationship.name then
				@class_properties << relationship
			end

			@relationships << relationship
		end

		def remove_relationship( relationship )
			@relationships.delete relationship
		end

		def add_sub_entity( child )
			child.parent = self
			@sub_entities << child
		end

		def remove_sub_entity( child )
			child.parent = nil
			@sub_entities.delete child
		end

		def gid( row )
			pk_values = primary_key row
			KeyGlobalID.new(@name, pk_values)
		end

		# Property must be an instance of EOAttribtue or EORelationship.
		def class_property?( property )
			if @class_properties.include? property then
				true
			elsif @class_property_names.include? property then
				true
			else
				false
			end
		end

		def class_property_attributes
			attrs = []
			attributes.each do |attr|
				if class_properties.include? attr then
					attrs << attr
				end
			end
			attrs
		end

		def class_property_relationships
			relations = []
			relationships.each do |relation|
				if class_properties.include? relation then
					relations << relation
				end
			end
			relations
		end

		def validate_required_attributes
			msg = "Entity requires attributes: 'name', 'external_name', " +
				"'attributes', 'class_properties', 'primary_key_attributes'"

			if @name.nil? or @external_name.nil? or @attributes.empty? or \
				@class_properties.empty? or @primary_key_attributes.empty? then
				key = @apapetr_name || :model
				error = ValidationError.new(msg, key)
				raise error
			end
		end


		##
		## primary key
		##

		def primary_key( gid_or_row )
			values = key_values = nil

			case gid_or_row
			when KeyGlobalID
				key_values = gid_or_row.key_values
			when Hash
				key_values = gid_or_row
			end

			if key_values then
				values = {}
				key_values.each do |key, value|
					if primary_key_attribute? key then
						values[key] = value
					end
				end
			end
	
			values
		end

		# Attribute or name
		def primary_key_attribute?( attribute )
			if @primary_key_attributes.include? attribute then
				true
			elsif @primary_key_attribute_names.include? attribute then
				true
			else
				false
			end
		end

		def qualifier_for_primary_key( row )
			values = {}
			@primary_key_attribute_names.each do |name|
				if row[name] then
					values[name] = row[name]
				end
			end

			# new_to_match_all_values
			if values then
				Qualifier.new_to_match_all_values values
			else
				nil
			end
		end

		def qualifier_for_primary_key?( qualifier )
			case qualifier
			when KeyValueQualifier
				_key_qualifier_for_primary_key? qualifier
			when AndQualifier, OrQualifier
				_and_or_qualifier_for_primary_key? qualifier
			else
				false
			end
		end

		private

		def _key_qualifier_for_primary_key?( qualifier )
			if primary_key_attribute? qualifier.key then
				true
			end
		end

		def _and_or_qualifier_for_primary_key?( qualifiers )
			result = false
			qualifiers.each do |qualifier|
				case qualifier
				when KeyValueQualifier
					result = _key_qualifier_for_primary_key? qualifier
				when AndQualifier
					result = _and_or_qualifier_for_primary_key? qualifier
				end
			end
			result
		end

		public

		##
		## destination objects
		##

		def destination_entity( keypath )
			_destination_object(keypath, Entity)
		end

		def destination_attribute( keypath )
			_destination_object(keypath, Attribute)
		end

		def destination_relationship( keypath )
			_destination_object(keypath, Relationship)
		end

		private

		def _destination_object( keypath, klass )
			paths             = keypath.split '.'
			last_entity       = self
			last_attribute    = nil
			last_relationship = nil
			last              = nil

			paths.each_with_index do |path, index|
				if (index + 1) == paths.size then
					if attribute = last_entity.attribute(path) then
						last_attribute = attribute
						last           = Attribute
					elsif relationship = last_entity.relationship(path) then
						last_relationship = relationship
						last_entity       = last_relationship.destination_entity
						last              = Relationship
					end
				else
					last_relationship = last_entity.relationship path
					last_entity       = last_relationship.destination_entity
				end
			end

			object = nil
			if klass == Entity then
				object = last_entity
			elsif (klass == Attribute) and (last == Attribute) then
				object = last_attribute
			elsif klass == Relationship
				object = last_relationship
			end

			object
		end

		public

		def to_h
			attrs      = []
			rels       = []
			properties = []
			pk_attrs   = []
			@attributes.each       { |attr| attrs << attr.to_h }
			@relationships.each    { |rel| rels << rel.to_h }
			@class_properties.each { |prop| properties << prop.name }
			@primary_key_attributes.each { |attr| pk_attrs << attr.name }

			property = {
				'name'                   => @name,
				'class_name'             => @class_name,
				'external_name'          => @external_name,
				'attributes'             => attrs,
				'class_properties'       => properties,
				'primary_key_attributes' => pk_attrs
			}
			property['relationships'] = rels unless rels.empty?

			property
		end
	end
end
