require 'dbi'
require 'tapkit/access/adapters/dbi'

module TapKit

	class PostgreSQLAdapter < DBIAdapter
		class << self
			def expression_class
				PostgreSQLExpression
			end

			# Internal types for PostgreSQL.
			# PostgreSQLAdapter supports basic data types.
			# Geometry, money, list data types are not supported.
			def internal_types
				types = super
				types['boolean']   = String
				types['bigint']    = Integer
				types['bigserial'] = Integer
				types['serial']    = Integer
				types['bytea']     = String
				types['text']      = String
				types['cidr']      = String
				types['inet']      = String
				types['macaddr']   = String
				types
			end
		end

		def expression_factory
			PostgreSQLExpressionFactory.new self
		end
	end

	class PostgreSQLChannel < DBIChannel
		def auto_commit
			true
		end

		def evaluate_for_aggregate_spec( agg_spec, expression )
			state = self.evaluate expression
			rows = []
			state.fetch_all.each do |row|
				new_row = {}
				row.each_with_index do |value, index|
					key = agg_spec.attributes[index][:key]
					new_row[key] = value
				end
				rows << new_row
			end
			rows
		end
	end

	class PostgreSQLContext < DBIContext
		def create_channel
			channel = PostgreSQLChannel.new self
			@channels << channel
			channel
		end
	end

	class PostgreSQLExpression < DBIExpression
		class << self
			def adapter_class
				PostgreSQLAdapter
			end
		end

		def initialize( entity, encoding = nil )
			super
			@join_on = []
		end

		# CILIKE -> ILIKE
		def sql_for_case_insensitive_like( value, key )
			"#{key} ILIKE #{value}"
		end

		def add_select_list_with_function( attribute, function, as = nil )
			column = sql_for_attribute attribute
			item   = format_sql_string(column, attribute.read_format)
			item   = sql_for_function(function, item)
			item << " AS #{as}" if as
			append_item item
		end

		def join_clause
			if @join_entities.empty? then
				return ''
			end

			clause = ''
			semantic = "#@join_semantic JOIN"
			@join_entities.each_with_index do |entity, index|
				name    = entity[0].external_name
				another = entity[1]
				clause << "#{semantic} #{name} #{another} ON #{@join_on[index]} "
			end
			clause.chop!
			clause
		end

		def add_join( left, right, semantic )
			clause = assemble_join(left, right, semantic)

			unless @join_semantic then
				case semantic
				when Relationship::INNER_JOIN
					@join_semantic = "INNER"
				when Relationship::FULL_OUTER_JOIN
					@join_semantic = "FULL OUTER"
				when Relationship::LEFT_OUTER_JOIN
					@join_semantic = "LEFT OUTER"
				when Relationship::RIGHT_OUTER_JOIN
					@join_semantic = "RIGHT OUTER"
				end
			end

			@join_on << clause
			clause
		end
	end

	class PostgreSQLExpressionFactory < DBIExpressionFactory
		def expression_class
			PostgreSQLExpression
		end
	end

end
