/******************************************************************************
 * unix.c (uptime.c)
 *
 * Author:
 *    Daniel Berger
 *
 * sys-uptime source code for most *nix platforms
 *****************************************************************************/
#include "ruby.h"
#include "lib/version.h"

#if defined (__FreeBSD__)
#include <sys/time.h>
#include <sys/sysctl.h>

#else
#include <sys/times.h>
#include <unistd.h>
#include <time.h>

#ifdef HAVE_UTMPX_H
#include <utmpx.h>
#endif

#ifdef HAVE_SYS_LOADAVG_H
#include <sys/loadavg.h>
#endif

#ifdef _SC_CLK_TCK
#define TICKS sysconf(_SC_CLK_TCK)
#else
#define TICKS sysconf(CLOCKS_PER_SEC)
#endif

#endif

#define MAXSTRINGSIZE 32 /* reasonable limit */

VALUE cUptimeError;

time_t get_uptime_secs()
{
#if defined (__FreeBSD__)
   struct timeval tv;
   size_t tvlen = sizeof(tv);
   int mib[2];

   mib[0] = CTL_KERN;
   mib[1] = KERN_BOOTTIME;
   if(sysctl(mib, 2, &tv, &tvlen, NULL, 0)){
      rb_raise(cUptimeError,"sysctl() call failed");
   }
   return time(NULL) - tv.tv_sec;
#elif HAVE_UTMPX_H
   time_t uptime;
   struct utmpx id;
   struct utmpx* u;
   setutxent();
   id.ut_type = BOOT_TIME;
   u = getutxid(&id); 
   if(u == NULL)
      rb_raise(cUptimeError,"failed to extract uptime");

   uptime = time(NULL) - u->ut_tv.tv_sec;
   return uptime;
#else
   struct tms tms;
   time_t seconds;
   seconds = times(&tms) / TICKS;
   
   if(-1 == seconds)
      rb_raise(cUptimeError,"times() function failed");
      
   if(seconds < 0)
      rb_raise(cUptimeError,"value returned larger than type could handle");
   
   return seconds;
#endif
}

static VALUE uptime_seconds()
{
   return UINT2NUM(get_uptime_secs());
}

static VALUE uptime_minutes()
{
   return UINT2NUM(get_uptime_secs() / 60);
}

static VALUE uptime_hours()
{
   return UINT2NUM(get_uptime_secs() / 3600);
}

static VALUE uptime_days()
{
   return UINT2NUM(get_uptime_secs() / 86400);
}

static VALUE uptime_uptime()
{
   char c_string[MAXSTRINGSIZE];
   long seconds, days, hours, minutes;
 
   seconds = get_uptime_secs();
   days = seconds/86400;
   seconds -= days*86400;
   hours = seconds/3600;
   seconds -= hours*3600;
   minutes = seconds/60;
   seconds -= minutes*60;

   sprintf(c_string, "%ld:%ld:%ld:%ld", days, hours, minutes, seconds);
   return rb_str_new2(c_string);
}

static VALUE uptime_dhms()
{
   VALUE a = rb_ary_new2(4);
   long s, m, h, d;

   s = get_uptime_secs();
   d =  s                / (24*60*60);
   h = (s -= d*24*60*60) / (   60*60);
	m = (s -= h*   60*60) /        60;
	s      -= m*      60;

	rb_ary_push(a, INT2FIX(d));
	rb_ary_push(a, INT2FIX(h));
	rb_ary_push(a, INT2FIX(m));
	rb_ary_push(a, INT2FIX(s));
	return a;
}

static VALUE uptime_btime(){
#if defined (__FreeBSD__)
   struct timeval tv;
   size_t tvlen = sizeof(tv);
   int mib[2];

   mib[0] = CTL_KERN;
   mib[1] = KERN_BOOTTIME;
   if(sysctl(mib, 2, &tv, &tvlen, NULL, 0)){
      rb_raise(cUptimeError,"sysctl() call failed");
   }
   return rb_time_new(tv.tv_sec,tv.tv_usec);
#else
#ifdef HAVE_UTMPX_H
   struct utmpx* ent;
   setutxent();
 
   while( (ent = getutxent()) ){
      if(ent->ut_type == BOOT_TIME){
         return rb_time_new(ent->ut_tv.tv_sec,ent->ut_tv.tv_usec);
      }      
   }
#else
   rb_raise(cUptimeError,"boot_time() not implemented on this platform");
#endif
#endif
   return Qnil;
}

void Init_uptime()
{
   VALUE mSys, cUptime;

   /* Modules and Classes */
   mSys = rb_define_module("Sys");
   cUptime = rb_define_class_under(mSys, "Uptime", rb_cObject);
   cUptimeError = rb_define_class_under(mSys, "UptimeError", rb_eStandardError);
   
   /* Constants */
   rb_define_const(cUptime,"VERSION",rb_str_new2(SYS_UPTIME_VERSION));

   /* Class Methods */
   rb_define_singleton_method(cUptime, "seconds", uptime_seconds, 0);
   rb_define_singleton_method(cUptime, "minutes", uptime_minutes, 0);
   rb_define_singleton_method(cUptime, "hours",   uptime_hours,   0);
   rb_define_singleton_method(cUptime, "days",    uptime_days,    0);
   rb_define_singleton_method(cUptime, "uptime",  uptime_uptime,  0);
   rb_define_singleton_method(cUptime, "dhms",    uptime_dhms,    0);
   rb_define_singleton_method(cUptime, "boot_time", uptime_btime, 0);
}
