#
# How to use:
#
# db = NFSStore.new("/tmp/foo")
# db.transaction do
#   p db.roots
#   ary = db["root"] = [1,2,3,4]
#   ary[0] = [1,1.5]
# end

# db.transaction do
#   p db["root"]
# end

  require 'ftools'
  require 'digest/md5'
  require 'socket'
  require 'tempfile'
  require 'fcntl'
  require 'fileutils'

#
#
# lockfile.rb provides nfs resistent lockfiles 
#
#
  require 'lockfile.rb'

#
#
# NFSStore: an nfs resistent pstore
#
#

  class NFSStore
#{{{
  #
  #
  # following constants and class methods were added
  #
  #
    VERSION = '0.5.0'
    HOSTNAME = Socket::gethostname
    DONT_USE_NFSLOCK = false

    class << self
#{{{
      attr :dont_use_nfslock, true
      attr :uncache_attempts, true
      attr :load_attempts, true
      attr :load_timeout, true
      attr :hostname, true
      def init
#{{{
        @hostname = HOSTNAME
        @dont_use_nfslock = DONT_USE_NFSLOCK
        @load_attempts = 16
        @load_timeout = 0.01
        @uncache_attempts = 4
#}}}
      end
#}}}
    end
    self.init

    def initialize(file, opts = {})
#{{{
      @opts = opts
      dir = File::dirname(file)
      unless File::directory? dir
        raise NFSStore::Error, format("directory %s does not exist", dir)
      end
      if File::exist? file and not File::readable? file
        raise NFSStore::Error, format("file %s not readable", file)
      end
      @transaction = false
      @filename = file
      @dirname, @basename = File.dirname(@filename), File.basename(@filename)
      @tempfile = Tempfile.new(@filename.gsub(%r|/+|o,'_'))
      @tempfile.close
      @local = @tempfile.path
      @backup = "#{ @filename }~"
      @backupbackup = File.join(@dirname, ".#{ @basename }~")
      @lockfile = getopt('lockfile', @opts) || Lockfile.new("#{ @filename }.lock")
      @dont_use_nfslock = getopt('dont_use_nfslock', @opts) || klass.dont_use_nfslock 
      @load_attempts = getopt('load_attempts', @opts) || klass.load_attempts 
      @load_timeout = getopt('load_timeout', @opts) || klass.load_timeout 
      @uncache_attempts = getopt('uncache_attempts', @opts) || klass.uncache_attempts 
      @abort = false
#}}}
    end

    class Error < StandardError; end
    def in_transaction
#{{{
      raise NFSStore::Error, "not in transaction" unless @transaction
#}}}
    end
    private :in_transaction
    def [](name)
#{{{
      in_transaction
      @table[name]
#}}}
    end
    def fetch(name, default=NFSStore::Error)
#{{{
      unless @table.key? name
        if default==NFSStore::Error
          raise NFSStore::Error, format("undefined root name `%s'", name)
        else
          default
        end
      end
      self[name]
#}}}
    end
    def []=(name, value)
#{{{
      in_transaction
      @table[name] = value
#}}}
    end
    def delete(name)
#{{{
      in_transaction
      @table.delete name
#}}}
    end
    def roots
#{{{
      in_transaction
      @table.keys
#}}}
    end
    def root?(name)
#{{{
      in_transaction
      @table.key? name
#}}}
    end
    def path
#{{{
      @filename
#}}}
    end
    def commit
#{{{
      in_transaction
      @abort = false
      throw :pstore_abort_transaction
#}}}
    end
    def abort
#{{{
      in_transaction
      @abort = true
      throw :pstore_abort_transaction
#}}}
    end
    def transaction(read_only=false)
#{{{
      raise NFSStore::Error, 'nested transaction' if @transaction

      value = nil
      content = nil
      orig = nil
      triedbackup = false
      reload = true
      attempts = 0
      md5 = nil
      size = nil
      backedup = nil

    #
    # start transaction
    #
      begin
      #
      # obtain lock
      #
        lock do
          @table = nil
          @transaction = true
        #
        # load data from @filename or backup
        #
          begin
          #
          # actual read of data
          #
            begin
              begin
                uncache @filename
                File::copy @filename, @local
                orig = true
              rescue Errno::ENOENT
                orig = false 
                if read_only
                  reload = false
                  raise
                end
                FileUtils::touch @local
              end

              open(@local, 'rb') do |f|
                if read_only
                  @table = Marshal::load(f)
                elsif orig and (content = f.read) != ""
                  @table = Marshal::load(content)
                  size = content.size
                  md5 = Digest::MD5.digest(content)
                  content = nil	# unreference huge data
                else
                  @table = {}
                end
              end

            rescue => e
              raise unless reload
              raise if attempts >= @load_attempts 
              attempts += 1
              warn "load failure <#{ attempts }>\n#{ errmsg e }" 
              sleep @load_timeout
              retry
            end # reading

          rescue => e
            raise if triedbackup
            raise unless File.exist? @backup
            warn "failback - <#{ @backup }> --> <#{ @filename }>" 
            File::mv @backup, @filename
            triedbackup = true
            retry
          end # loading
        #
        # confirm load status
        #
          raise "error loading <#{ @filename }>" unless @table
        #
        # yield to user callback and upload changes - iff any
        #
          begin
            catch(:pstore_abort_transaction){ value = yield self }
          rescue Exception
            @abort = true
            raise
          ensure
            if not read_only and not @abort
              content = Marshal::dump(@table)
              if not md5 or size != content.size or md5 != Digest::MD5.digest(content)
                open(@local,'w'){|f| f.write content}
                nfstmp = tmpnam @dirname, @basename
                File::copy @local, nfstmp
                if orig
                  File::mv @filename, @backup
                  File::copy @backup, @backupbackup
                  backedup = true
                end
                begin
                  raise "invalid lock!" unless @lockfile.validlock?
                  File::mv nfstmp, @filename
                  uncache @filename
                  raise "failed update" unless File.stat(@local).size == File.stat(@filename).size
                  raise "invalid lock!" unless @lockfile.validlock?
                rescue
                  if backedup
                    uncache @backup rescue nil
                    uncache @filename rescue nil
                    File::mv @backup, @filename rescue nil
                  end
                  raise
                end
              end
            end
            File.rm_f(@filename) if @abort and not orig
            @abort = false
          end
        end # lock
      ensure
        @table = nil
        @transaction = false
        content = nil # unreference huge data
      end # transaction

      return value
#}}}
    end
    def lock
#{{{
      ret = nil
      @lockfile.lock do
        open(@lockfile.path, 'rb+') do |lockf|
          unless nfslock(lockf, File::LOCK_EX | File::LOCK_NB)
            raise "lockfile collision"
          end
          begin
            ret = yield
          ensure
            nfslock lockf, File::LOCK_UN
          end
        end
      end
      ret
#}}}
    end
    def nfslock file, cmd
#{{{
      return self if @dont_use_nfslock
      begin
        icmd = ((cmd & File::LOCK_NB) == File::LOCK_NB) ? Fcntl::F_SETLK : Fcntl::F_SETLKW 
        type =
          case(cmd & ~File::LOCK_NB)
            when File::LOCK_SH
              Fcntl::F_RDLCK
            when File::LOCK_EX
              Fcntl::F_WRLCK
            when File::LOCK_UN
              Fcntl::F_UNLCK
            else
              raise ArgumentError, cmd.to_s
          end
        flock = [type, 0, 0, 0, 0].pack("ssqqi")
        file.fcntl icmd, flock 
      rescue => e
        raise "lockd problems - <#{ e.class }> <#{ e.message }>"
      end
#}}}
    end
    def getopt opt, hash
#{{{
      hash[opt] || hash[opt.to_str] || hash[opt.to_str.intern]
#}}}
    end
    def klass
#{{{
      self.class
#}}}
    end
    def uncache file 
#{{{
      f = nil

      begin
        path =
          case file
            when File
              file.path
            else
              file.to_str
          end

        attempts = 0
        begin
          f = open path, 'rb+'
        rescue Errno::ESTALE => e
          attempts += 1
          unless attempts >= @uncache_attempts
            retry
          else
            raise "unable to uncache <#{ path }>"
          end
        rescue Errno::ENOENT
          return self
        end

        dir, base = File.dirname(path), File.basename(path)
        refresh = tmpnam dir, base
      #
      # invalidate file
      #
        begin
          locked = nfslock(f,File::LOCK_EX|File::LOCK_NB)
          nfslock(f,File::LOCK_UN) if locked
          File.link path, refresh rescue nil
          stat = f.stat
          f.chmod stat.mode rescue nil
          File.utime stat.atime, stat.mtime, path rescue nil
        rescue
          nil
        end
      #
      # invalidate file's dir
      #
        begin
          open(dir) do |d| 
            path = d.path
            locked = nfslock(d,File::LOCK_SH|File::LOCK_NB)
            nfslock(d,File::LOCK_UN) if locked
            stat = d.stat
            d.chmod stat.mode rescue nil
            File.utime stat.atime, stat.mtime, path rescue nil
          end
        rescue
          nil
        end
        File.rm_f refresh
      ensure
        f.close if f
      end

      return self
#}}}
    end
    def tmpnam dir, seed = File.basename($0)
#{{{
      pid = Process.pid
      time = Time.now
      sec = time.to_i
      usec = time.usec
      "%s%s.%s_%d_%s_%d_%d_%d" %
        [dir, File::SEPARATOR, HOSTNAME, pid, seed, sec, usec, rand(sec)]
#}}}
    end
    def snapshot options = {} 
#{{{
      retries = getopt('retries', options) || 64
      sleeptime = getopt('sleep', options) || 0.02
      timeout = getopt('timeout', options) || 4

      content = nil
      snapshot = nil

      begin
        begin
          Timeout::timeout(timeout) do
            r = 0
            begin
              content = IO.read @filename
              snapshot = Marshal.load content
            rescue
              unless r > retries
                r += 1
                sleep sleeptime if sleeptime
                retry
              end
            end
          end
        rescue Timeout::Error
          #
          # do nothing
          #
        end

        unless snapshot
          attempts = 0
          tmp = nil
          begin
            tmp = Tempfile.new $0
            lock do 
              uncache @filename
              File::copy @filename, tmp.path
            end
            open(tmp.path, 'rb') do |t|
              content = t.read
              snapshot = (content == '' ? {} : Marshal.load(content))
            end
          rescue => e
            raise if attempts >= @load_attempts 
            attempts += 1
            warn "load failure <#{ attempts }>#{ errmsg e }" 
            sleep @load_timeout
            retry
          ensure
            tmp.close! if tmp
          end
        end
      ensure
        content = nil
      end

      snapshot
#}}}
    end
    def clear_locks options = {} 
#{{{
    #
    # a grace period to give other processes the chance to recover 
    #
      suspend = getopt('suspend', options) || 64
      begin
      #
      # kill lockfile
      #
        File.rm_f @lockfile.path
      #
      # clear fcntl based locks
      #
        dir, base = File.dirname(@filename), File.basename(@filename)
        tmp = tmpnam dir, base
        File.mv @filename, tmp
        File.mv tmp, @filename 
        sleep suspend if suspend
      rescue
      end
#}}}
    end
    def errmsg e
#{{{
      "%s (%s)\n%s\n" % [e.message, e.class, e.backtrace.join("\n")]
#}}}
    end
#}}}
  end # class NFSStore



if __FILE__ == $0
  db = NFSStore.new("/tmp/foo")
  db.transaction do
    p db.roots
    ary = db["root"] = [1,2,3,4]
    ary[1] = [1,1.5]
  end

  1000.times do
    db.transaction do
      db["root"][0] += 1
      p db["root"][0]
    end
  end

  db.transaction(true) do
    p db["root"]
  end
end
