require "socket"
require "timeout"

module Net
   class PingSimple
      VERSION = "0.3.0"
      attr_accessor :host, :port, :time, :data, :exception, :warning
      def initialize(host,port=nil,time=5)
         @host = host
         @port = port || Socket.getservbyname("echo")
         @time = time
         @data = "ping"
         @exception, @warning = nil
      end
   end

   ##########################################################################
   # With a TCP ping, simply try to open a connection.  If we are successful,
   # assume success.  In either case, close the connection to be polite.
   ##########################################################################
   class PingTCP < PingSimple
      @@econnrefused = false

      def self.econnrefused
         @@econnrefused
      end

      def self.ecr
         self.econnrefused
      end

      def self.econnrefused=(bool)
         unless bool.kind_of?(TrueClass) || bool.kind_of?(FalseClass)
            raise "Invalid type"
         end
         @@econnrefused = bool
      end

      def self.ecr=(bool)
         self.econnrefused = bool
      end

      def ping
         begin
            timeout(@time){
               begin
                  t = TCPSocket.new(@host,@port)
                  t.close
               rescue Errno::ECONNREFUSED => e
                  if @@econnrefused == true
                     return true
                  else
                     @exception = e
                     return false
                  end
               rescue
                  @exception = $!
                  return false
               else
                  return true
               end
            }
         rescue TimeoutError => t
            @exception = t
            return false
         end
      end

      alias ping? ping
   end
   
   ##########################################################################
   # With a UDP ping, send a simple text string and check the return string.
   # If they match, assume success.
   ##########################################################################
   class PingUDP < PingSimple

      def ping
         u = UDPSocket.open
         a = []
         begin
            timeout(@time){
               u.connect(@host,@port)
               u.send(@data,0)
               a = u.recvfrom(64)
            }
            if a[0] != @data
               return false
            end
         rescue TimeoutError => t
            @exception = t
            return false
         else
            return true
         end
      end
      alias_method(:ping?,:ping)
   end

   ##################################
   # Use your system's ping command
   ##################################
   class PingExternal < PingSimple

      def ping
         input, output, error = ""
         pstring = "ping "
         
         case PLATFORM
            when /linux|bsd/i
               pstring += "-c 1 #{@host}"
            when /solaris|sunos/i
               pstring += "#{@host} 1"
            when /hpux/i
               pstring += "#{@host} -n 1"
            when /win32|windows/i
               pstring += "-n 1 #{@host}"
            else
               pstring += "#{@host}"
         end
         
         if File::ALT_SEPARATOR
            require "win32_popen"
            timeout(@time){
               input,output,error = IO.win32_popen3(pstring)
            }
         else
            require "open3"            
            timeout(@time){
               input,output,error = Open3.popen3(pstring)
            }
         end

         e = error.gets # Can't chomp yet, might be nil

         input.close
         error.close
        
         unless e.nil?
            if e =~ /warning/i
               @warning = e.chomp
               return true
            else
               @exception = e.chomp
               return false
            end
         # The "no answer" response goes to stdout, not stderr, so check it
         else
            lines = output.readlines
            output.close
            if lines.nil? || lines.empty?
               return true
            else
               regexp = /no answer|host unreachable|could not find host/i
               lines.each{ |e|
                  if regexp.match(e)
                     @exception = e.chomp
                     return false
                  end
               }
               return true
            end
         end
      end
      alias_method(:ping?,:ping)
   end

end
