# The super class of dynamic element classes.
#
# Dynamic elements convert themselves to HTML.
# These are very important in development with CGIKit.
#
# CKElement returns HTML when to_s() is called.
# In components development, you bind elements to component's methods. 
# At runtime, the binding is executed by run(). 
# These two methods are expected to be overrided in the subclasses of CKElement.
#
# == Paths for searching elements and components
# CKElement objects, including components, are usually instantiated
# by CKElement.instance. This method loads elements/components and creates
# element objects. In this method, some paths are searched to require files.
#
# The searched paths:
# 1. CKApplicaton#component_path
# 2. ($LOAD_PATH)/cgikit/elements
# 3. ($LOAD_PATH)/cgikit/components
#
# The latter two paths are for extention elements or components.
# It is recommended that your own elements or components are installed
# in the 1st path.
#
# == Creation of custom elements
# If you create custom elements, the class of the custom elements must 
# inherit CKElement. Simple dynamic element can be created only 
# by overriding to_s(). to_s() is called when CGIKit
# converts the dynamic elements to HTML. If you add new attributes
# to your custom elements, run() must be overrided. 
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/elements.html]
class CKElement
	class UnknownElementError < CKError; end #:nodoc:
	class AttributeError      < CKError; end #:nodoc:

	attr_accessor :name
	attr_accessor :body
	attr_writer   :definition
	attr_accessor :parent
	attr_accessor :application

	# Array of repetitions.
	attr_accessor :repetitions

	# Index in the last repetition.
	attr_accessor :repetition_index

	class << self
		def instance( element, app, parent, name, body )
			body = '' unless body
			path = load_element_file(app.component_path, element)
			klass = app.class.const_get element
			klass.new(app, parent, name, body, path)
		end

		# Internal method.
		def load_element_file( component_path, element )
			component = "#{component_path}/#{element}/#{element}"

			# check paths for extension elements and components
			ext_element   = "cgikit/elements/#{element}"
			ext_component = "cgikit/components/#{element}/#{element}"
			paths         = [ ext_element, ext_component ]
			$LOAD_PATH.each do | load_path |
				paths.each do | path |
					ext_path = "#{load_path}/#{path}"
					if FileTest.exist? "#{ext_path}.rb".untaint
						require path.untaint
						return ext_path
					else
						next
					end
				end
			end

			# components
			if FileTest.exist? "#{component}.rb".untaint then
				require component.untaint
				return nil
			# starndard elements
			elsif Object.const_defined? element then
				return nil
			end

			raise UnknownElementError, "No such '#{element}' element or component."
		end

		# Internal method that will be overridden in subclasses.
		def bind_request( component, definition, value ); end

		# Return an array of names of the element's own attributes.
		# The value is used to generate optional HTML attributes.
		# Subclasses must override this method.
		def own_attributes
			[]
		end
	end

	# Returns the element ID.
	def element_id
		unless @element_id then
			@element_id             = CKElementID.new
			@element_id.component   = parent.class.to_s
			@element_id.element     = name

			unless repetitions.empty? then
				last_repetition = repetitions.pop
				last_repetition = [ last_repetition.first,repetition_index ]
				repetitions.push last_repetition
			end
			@element_id.repetitions = repetitions || []
		end
		@element_id
	end

	def initialize( app, parent, name, body, path = nil )
		@application      = app
		@parent           = parent
		@name             = name
		@body             = body
		@repetitions      = []
		@repetition_index = 0
	end

	# Fetches the result of the parent component's methods.
	# When is_binding is true, fetch() returns String without binding from . 
	def fetch( key, is_binding = true )
		data = definition[ key ]
		if is_binding == false
			data
		else
			parent.parse_ckd_value( data )
		end
	end

	# Return a definition hash of the element.
	def definition
		unless @definition then
			@definition = parent.definitions[name]
		end
		@definition
	end

	# Return a definition name of the element with the class name.
	def name_with_class
		"\"#{name}\" at #{self.class}"
	end

	# Internal method. This method will be overridden in subclasses.
	def run; end

	# Returns true if the element is a top level.
	def top_level?
		unless @parent then
			true
		else
			false
		end
	end

	# Return a string of values of optional attributes with
	# "other" attribute string excepting the element's own attributes.
	def other_attributes_string
		optional = ''
		each do |key, value|
			if (self.class.own_attributes.include?(key) == false) and \
				(value != nil) and (key != 'other') then
				value = parent.parse_ckd_value value
				optional << " #{key}=\"#{value}\""
			end
		end

		unless other = fetch('other') then
			other = ''
		else
			other = ' ' + other.to_s
		end			

		optional + other
	end

	# Calls block once for each attributes in definition
	# passing the key and value as parameters.
	def each
		definition.each do |key, value|
			unless (key == 'oid') or (key == 'element') then
				yield key, value
			end
		end
	end


	# ElementeAttribute module manages attributes of elements.
	# The module does that following:
	#
	# * add attributes for HTML common attributes to elements 
	# * add attributes for form to elements
	# * add "others" attributes to elements
	# * check attributes of elements
	#
	# == Added HTML common attributes
	# class, id, style, title, dir, lang, onclick, ondblcclick, onmousedown,
	# onmouseup, onmouseover, onmousemove, onmouseout, onkeypress,
	# onkeydown, onkeyup
	#
	# == Added form common attributes
	# enabled
	#
	# == "others" attributes
	#
	module ElementAttribute
		HTML_COMMON_ATTRIBUTES =
			[ 'class', 'id', 'style', 'title', 'dir', 'lang',
			  'onclick',     'ondblcclick', 'onmousedown', 'onmouseup',
			  'onmouseover', 'onmousemove', 'onmouseout',  'onkeypress',
			  'onkeydown',   'onkeyup' ]
		FORM_COMMON_ATTRIBUTES = ['enabled']
		OTHERS_ATTRIBUTE       = ['others']

		# Raises exception if the element has undefined attributes.
		def check_undefined_attributes( *attrs )
			attrs.flatten!
			attrs << 'oid'
			attrs << 'element'

			attr = name = nil
			definition.each_key do | attr |
				if ( attrs.include? attr ) == false then
					raise AttributeError, \
					"Attribute '#{attr}' is undefined attribute - #{name_with_class}"
				end
			end
		end

		# Raises an exception if the element doesn't have required attributes.
		def check_required_attributes( *patterns )
			requires = 0
			attrs    = []
			pattern  = attr = nil

			patterns.each do | pattern |
				pattern.each do | attr |
					requires += 1 if definition.include? attr
				end

				return true if requires == pattern.size
				requires = 0
				attrs << ( "'" + pattern.join(', ') + "'" )
			end

			message =  "#{self.class} requires #{attrs.join(' or ')} "
			message << "of attribute - #{name_with_class}"
			raise AttributeError, message
		end

		# Raises an exception if the element has attributes conflicting with 
		# each other.
		def check_conflicted_attributes( *attributes )
			combination = []
			attr = nil

			attributes.each do | attr |
				if definition.include? attr then
					combination << attr
				end
			end

			if combination.size >= 2 then
				_attr    = nil
				string   = ''
				pop1_str = combination.pop
				pop2_str = combination.pop
	
				combination.each do | _attr |
					string << "\"#{_attr}\", "
				end
				string << "\"#{pop2_str}\" or \"#{pop1_str}\""

				msg = "Exactly one of #{string} must be bound - "
				msg << name_with_class
				raise AttributeError, msg
			end
		end

		# Returns a string with HTML common attributes setted in the element.
		def string_for_html_common_attributes
			string = ''
			attr = value = nil
			HTML_COMMON_ATTRIBUTES.each do | attr |
				value = fetch attr
				if value then
					string << " #{attr}=\"#{value}\"" 
				end
			end
			string
		end

		# Returns a string with form common attributes setted in the element.
		def string_for_form_common_attributes
			string = ''
			attr = value = nil
			FORM_COMMON_ATTRIBUTES.each do | attr |
				value = fetch attr
				if ( attr == 'enabled' ) and ( value == false ) then
					string << " disabled" 
				elsif value and (attr != 'enabled') then
					string << " #{attr}=\"#{value}\"" 
				end
			end
			string
		end

		# Returns an array of names of HTML common attributes.
		def html_common_attributes
			HTML_COMMON_ATTRIBUTES
		end

		# Returns an array of names of form common attributes.
		def form_common_attributes
			FORM_COMMON_ATTRIBUTES
		end
	end


	# ElementState module manages states of elements.
	module ElementState
		class << self
			# Returns a hash parsed a string of state.
			def parse( component, string )
				hash     = {}
				splitted = []

				items = string.split ','
				items.each do | item |
					splitted = item.split ':'
					hash[splitted.first] = component.parse_ckd_value splitted.last
				end
				hash
			end
		end

		# Returns a hash of state for CKConditional.
		def state_for_conditional
			state( 'CKConditional', 'condition', 'negate' )
		end

		# Returns a hash of state.
		def state( element, *attributes )
			value  = nil
			values = []
			query  = {}

			parent.definitions.each do | key, definition |
				if definition['element'] == element then
					id           = CKElementID.new
					id.component = parent.class.to_s
					id.element   = definition['oid']

					values = []
					attributes.each do | attr |
						value = parent.parse_ckd_value( definition[attr] )
						values << "#{attr}:#{value.inspect}"
					end
					query[id.to_s] = values.join ','
				end
			end
			query
		end
	end

end



module CKValidatable
	def validate_with_format_key( key, format_key, pass_key )
		format = parse_ckd_value format_key
		validate(key, format, pass_key)
	end

	def validate( key, format, pass_key )
		qualifier = CKQualifier.format format

		if qualifier.eval? self then
			self[pass_key] = true
		else
			self[pass_key] = false
		end
	end
end


# When you define your component, its class must inherit CKComponent in code
# file. The component dynamically renders HTML with template and binding file. 
#
# A component has 3 files in a directory whose name is the same as the
# component's name. These three files composite Model-View-Controller
# architecture.
#
# Template(View)::      A HTML file includes CGIKit tags ( <cgikit> ). One
#                       CGIKit tag corresponds to one dynamic element.
#
# Binding(Controller):: A definition file for elements is used by a component.
#                       The suffix, ckd, is abbreviation of "CGIKit Definition".
#                       The definition file connects Template to Code.
#
# Code(Model)::         A Ruby source where a component's class is defined. 
#                       It must inherit CKComponent. The name of the component's
#                       class must be the same as the component name. 
#
# ex) An application with a MainPage component
#  /cgi-bin/cgikit-app
#    cgikit-app.cgi
#    /MainPage
#      MainPage.html
#      MainPage.ckd
#      MainPage.rb
#
# == Programming Topics
# * Architecture[www.spice-of-life.net/download/cgikit/en/userguide/architecture.html]
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKComponent < CKElement
	include CKKeyValueCoding, CKValidatable

	class FileNotFoundError < CKError; end #:nodoc:

	# Path for structual files of the component.
	attr_reader :path

	# Hash of definitions in a binding file of the component.	
	attr_accessor :definitions

	attr_accessor :is_top

	class << self
		# Enables accessing instance variables directly with value(), take_value().
		def access_instance_variables?
			true
		end
	end

	def initialize( app, parent, name, body, path = nil )
		super( app, parent, name, body, path )
		@path = path || "#{app.component_path}/#{_component_name}/#{_component_name}"
		@definitions = CKDefinition.parse_ckd_file definition_file
		@is_top = false
		init
	end

	# Hook method for initialization instead of initialize().
	# Form data is not setted when the method is called.
	def init; end

	# Returns a request object of the application.
	def request
		application.request
	end

	# Returns a response object of the application.
	def response
		application.response
	end

	# Returns the session object. If the session isn't existed, returns a new session.
	def session
		application.session
	end

	# Returns a resource manager object.
	def resource_manager
		application.resource_manager
	end

	# Internal method.
	def parse_ckd_value( string )
		object = nil
		if string =~ /\A[^"'](.*)\[(.*)\]/ then
			variable = retrieve_value $1
			key      = parse_ckd_value $2
			object   = variable[key]
		elsif string == 'true'   then object = true
		elsif string == 'false'  then object = false
		elsif string == 'nil'    then object = nil
		elsif string =~ /"(.*)"/ then object = $1
		elsif string =~ /'(.*)'/ then object = $1
		elsif ((not string =~ /[\D]/) and (string =~ /[\d]/)) then
			object = string.to_i
		else
			object = retrieve_value string if string
		end

		object
	end

	# Internal method.
	def variable?( string )
		if ( string == 'true' )   or ( string == 'false'  ) or \
		   ( string == 'nil'  )   or ( string =~ /"(.*)"/ ) or \
		   ( string =~ /'(.*)'/ ) or \
		   ( ( not string =~ /[\D]/ ) and ( string =~ /[\d]/ ) ) or \
			( self.respond_to?("#{string}=") == false ) then
			false
		else
			true
		end
	end

	# Returns name of the template file.
	def template_file
		if application.master_locale? then
			"#{path}.html"
		else
			file = "#{path}_#{application.locale}.html".untaint
			if FileTest.exist? file then
				return file
			end

			unless request.languages.empty? then
				request.languages.each do |lang|
					file = "#{path}_#{lang}.html".untaint
					if FileTest.exist? file then
						return file
					end
				end
			end

			"#{path}.html"
		end
	end

	# Returns body of the template file as string.
	def template_string
		begin
			f = File.new template_file.untaint
			@template_string = f.read
			f.close
		rescue
			msg = "Can't read a template file - #{template_file}"
			raise FileNotFoundError, msg
		end

		@template_string
	end

	# Returns name of the definition( binding ) file.
	def definition_file
		"#{path}.ckd".untaint
	end

	# Returns body of the definition( binding ) file as string.
	def definition_string
		unless @definition_string then
			begin
				f = File.new definition_file
				@definition_string = f.readlines.join
				f.close
			rescue
				msg = "Can't read a binding file - #{definition_file} #{self.class}"
				raise FileNotFoundError, msg
			end
		end

		@definition_string
	end

	# Creates a specified page component.
	def page( page )
		CKElement.instance( page, application, parent, nil, nil )
	end

	# Invokes the action after binding component's variables to attributes
	# for definitions, values from request to component's variables.
	def run
		_bind_attribute
		_bind_request

		if is_top then
			pre_action
			result = _invoke_action
			post_action
		end

		result
	end

	private

	# Binds attribute setted in parent component to self.
	# This method is for nested components.
	def _bind_attribute
		if parent and definition then
			defaults = { "name" => true, "oid" => true, "element" => true }
			definition.each do | key, value |
				unless defaults[key] then
					take_value( key, parent.parse_ckd_value(value) )
				end
			end
		end
	end

	def _bind_request
		id = key = value = klass = klass_name = list_name = item_name = nil

		application.request.form_values.sort.each do | key, value |
			id = CKElementID.new key.to_s
			next unless _component_name == id.component

			if definitions[id.element] then
				klass_name  = definitions[id.element]['element']
			else
				# for radio buttons of CKRadioButton and CKGenericElement
				def_name, attr_value = value.last.split( '.', 2 )
				klass_name           = definitions[def_name]['element']
				id.element           = def_name
				value                = attr_value
			end

			_bind_item_in_repetitions id
			value = convert_char_code value

			CKElement.load_element_file( application.component_path, klass_name )
			klass = Object.const_get klass_name
			klass.bind_request( self, definitions[id.element], value, id )
		end
	end

	def _bind_item_in_repetitions( id )
		rep_name = index = list_name = item_name = list = nil

		# check repetitions reflexively
		if id.repetitions? then
			id.each do | rep_name, index |
				if definitions.include? rep_name then
					list_name = definitions[rep_name]['list']
					item_name = definitions[rep_name]['item']
				end

				if list_name and item_name then
					list = retrieve_value list_name

					if list then
						list.each_with_index do |_item, _index|
							if index == _index then
								take_value(item_name, _item)
								break
							end
						end
					end
				end
			end
		end
	end

	public

	# Hook method to convert character code for the form values.
	def convert_char_code( values )
		if code = application.char_code then
			require 'kconv'

			case code.downcase
			when 'jis'  then kcode = Kconv::JIS
			when 'sjis' then kcode = Kconv::SJIS
			when 'euc'  then kcode = Kconv::EUC
			else
				return values
			end

			k_value = []
			values.each do |value|
				if String === value then
					k_value << Kconv.kconv(value, kcode)
				else
					k_value << value
				end
			end
			values = k_value
		end
		values
	end

	private

	def _invoke_action
		result = nil
		if id = application.element_id then
			_bind_item_in_repetitions id

			# get action
			def_action = nil
			if id.component == _component_name then
				definitions.each_value do | definition |
					if definition['oid'] == id.element then
						case definition['element']
						when 'CKGenericElement' then
							def_action = 'invoke_action'
						when 'CKFrame' then
							def_action = 'value'
						when 'CKImage' then
							def_action = 'data'
							mime = parse_ckd_value definition['mime']
							response.headers['Content-Type'] = mime
						else
							def_action = 'action'
						end

						@action = definition[def_action]
					end
				end

				result = retrieve_value @action if @action
			end
		end

		result
	end

	# Returns name of the component in name space package.
	def _component_name
		self.class.to_s.split('::').last
	end

	public

	# Hook method called after setting form data, before invoking action.
	def pre_action; end

	# Hook method called after run() that are setting form data and invoking action.
	def post_action; end

	# Converts the component to HTML.
	def to_s
		parser = CKHTMLParser.new( self, template_string )
		parser.parse
	end
end

