/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include <squirrel.h>

#include "../Globals.h"
#include "../Group.h"
#include "../RTS.h"
#include "../SettingsStruct.h"
#include "../Sound.h"
#include "../Stuff.h"
#include "../World.h"
#include "ScriptSquadron.h"

#include <iterator>
#include <sstream>
#include <string>
#include <stdexcept>
#include <stdarg.h>

using std::list;
using std::wostringstream;
using std::wstring;
using std::runtime_error;

namespace Scripting {

//print_func

void print_func(HSQUIRRELVM v, const SQChar *s, ...) {
	wostringstream ss;
	wchar_t buf[4096];

	va_list arglist;
	va_start(arglist, s);
	vswprintf(buf, 4096, s, arglist);
	ss << (const wchar_t*) buf << std::flush;
	va_end(arglist);

	//in case this is an error being printed by squirrel interpreter we also put the output into the global_error_string
	//for when we go back to main menu
	global_error_string = ss.str();
	create_info_string(global_error_string);
	write_log(global_error_string);
}

//ai functions

void throw_sq_error(HSQUIRRELVM v, std::exception& e) {
	wstring err = string_to_wstring(e.what());
	sq_throwerror(v, err.c_str());
}

SQInteger get_vm_my_side(HSQUIRRELVM v) {
	SQInteger ret;
	sq_pushstring(v, L"my_side", -1);
	sq_get(v, 1);
	if (SQ_FAILED(sq_getinteger(v, -1, &ret))) {
		sq_throwerror(v, _SC("Could not get my_side"));
		return SQ_ERROR;
	}

	return ret;
}

ScriptSquadron* get_squad_instance(HSQUIRRELVM v) {
	SQUserPointer data;
	if (SQ_FAILED(sq_getinstanceup(v, 1, &data, 0))) {
		sq_throwerror(v, _SC("Squad function called without instance"));
		return 0;
	}

	return reinterpret_cast<ScriptSquadron*> (data);
}

static SQInteger Squad_select_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		_this->select();
		return 0;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_highlight_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		_this->highlight();
		return 0;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_is_moving_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		sq_pushbool(v, _this->squad->is_moving());
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_ready_for_launch_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		sq_pushbool(v, _this->squad->ready_for_launch());
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_get_in_hangar_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		sq_pushbool(v, _this->squad->both_in_hangar());
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_get_type_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		sq_pushinteger(v, _this->squad->get_type());
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_get_pos_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	try {
		CoordsFloat pos = _this->squad->get_pos();
		sq_newarray(v, 0);
		sq_pushinteger(v, static_cast<int>(pos.x));
		sq_arrayappend(v, -2);
		sq_pushinteger(v, static_cast<int>(pos.y));
		sq_arrayappend(v, -2);
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_get_distance_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	SQUserPointer arg0;
	if (SQ_FAILED(sq_getinstanceup(v, 2, &arg0, 0))) {
		sq_throwerror(v, _SC("'Argument 1 is not an instance"));
		return SQ_ERROR;
	}
	ScriptSquadron* _arg = reinterpret_cast<ScriptSquadron*> (arg0);

	try {
		sq_pushinteger(v, _this->squad->find_distance_to(_arg->squad->my_side, _arg->squad->my_squad));
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger Squad_get_distance_to_point_wrapper(HSQUIRRELVM v) {
	ScriptSquadron* _this = get_squad_instance(v);

	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	try {
		sq_pushinteger(v, _this->squad->find_distance_to_point(arg0, arg1));
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger ai_suspend_wrapper(HSQUIRRELVM v) {
	world.script_manager.suspend_ai_script(v);
	return sq_suspendvm(v);
}

static SQInteger order_move_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	SQInteger my_side = get_vm_my_side(v);

	CoordsFloat center = sides[my_side].work_out_selected_center();

	for (list<CoordsInt>::iterator iter = sides[my_side].selected_squads.begin(); iter != sides[my_side].selected_squads.end(); ++iter) {
		CoordsFloat group_center = sides[my_side].squadrons[iter->y].get_center();
		float dx = group_center.x - center.x;
		float dy = group_center.y - center.y;
	
		sides[my_side].squadrons[iter->y].order_move(static_cast<float>(arg0) + dx, static_cast<float>(arg1) + dy);
	}

	sides[my_side].selected_squads.clear();

	return 0;
}

static SQInteger order_halt_wrapper(HSQUIRRELVM v) {
	SQInteger my_side = get_vm_my_side(v);

	for (list<CoordsInt>::iterator iter = sides[my_side].selected_squads.begin(); iter != sides[my_side].selected_squads.end(); ++iter)
		sides[my_side].squadrons[iter->y].order_halt();

	sides[my_side].selected_squads.clear();

	return 0;
}

static SQInteger add_waypoint_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	SQInteger my_side = get_vm_my_side(v);

	for (list<int>::iterator iter = sides[my_side].highlighted_squads.begin(); iter != sides[my_side].highlighted_squads.end(); ++iter)
		sides[my_side].squadrons[*iter].add_waypoint(CoordsFloat(static_cast<float>(arg0), static_cast<float>(arg1)));

	return 0;
}

static SQInteger launch_mission_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	SQInteger my_side = get_vm_my_side(v);

	sides[my_side].launch_mission(arg0);
	sides[my_side].highlighted_squads.clear();

	return 0;
}

static SQInteger order_return_wrapper(HSQUIRRELVM v) {
	SQInteger my_side = get_vm_my_side(v);

	sides[my_side].order_return();
	sides[my_side].selected_squads.clear();

	return 0;
}

static SQInteger order_recall_wrapper(HSQUIRRELVM v) {
	SQInteger my_side = get_vm_my_side(v);

	sides[my_side].order_recall();
	sides[my_side].selected_squads.clear();

	return 0;
}

static SQInteger world_frame_counter_wrapper(HSQUIRRELVM v) {
	try {
		sq_pushinteger(v, world.frame_counter);
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger wakeup_mission_script_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	WakeEventType which = WET_CUSTOM_AI_TASK_0;
	
	switch (arg0) {
		case 0:
			which = WET_CUSTOM_AI_TASK_0;
			break;

		case 1:
			which = WET_CUSTOM_AI_TASK_1;
			break;

		case 2:
			which = WET_CUSTOM_AI_TASK_2;
			break;

		case 3:
			which = WET_CUSTOM_AI_TASK_3;
			break;

		case 4:
			which = WET_CUSTOM_AI_TASK_4;
			break;

		default:
			sq_throwerror(v, _SC("There are only 5 custom ai tasks"));
			return SQ_ERROR;
	}

	world.script_manager.fire_wakeup_event(which);
	return 0;
}

static SQInteger is_cut_scene_on_wrapper(HSQUIRRELVM v) {
	try {
		sq_pushbool(v, world.cut_scene_on);
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

//mission functions

static SQInteger play_script_sound_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	try {
		sound.play_script_sound(arg0);
		return 0;
	} catch (std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger add_portrait_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}
	const SQChar* arg1;
	if (SQ_FAILED(sq_getstring(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not a string"));
		return SQ_ERROR;
	}
	SQInteger arg2;
	if (SQ_FAILED(sq_getinteger(v, 4, &arg2))) {
		sq_throwerror(v, _SC("Argument 3 not an integer"));
		return SQ_ERROR;
	}

	try {
		RTS::PortraitText::add_portrait(arg0, arg1, arg2);
		return 0;
	} catch (std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger set_portrait_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	SQBool arg1 = false;
	if (sq_gettop(v) == 3)
		sq_getbool(v, 3, &arg1);

	try {
		RTS::PortraitText::set_portrait(arg0, arg1);		
		return 0;
	} catch (std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger portrait_text_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	wstring sound_file = L"";

	if (sq_gettop(v) >= 3) {
		const SQChar* arg1;
		if (SQ_FAILED(sq_getstring(v, 3, &arg1))) {
			sq_throwerror(v, _SC("Argument 2 not a string"));
			return SQ_ERROR;
		}
		sound_file = arg1;
	}

	SQBool arg2 = false;
	if (sq_gettop(v) == 4) {
		if (SQ_FAILED(sq_getbool(v, 4, &arg2))) {
			sq_throwerror(v, _SC("Argument 3 not a bool"));
			return SQ_ERROR;
		}
	}

	try {
		RTS::PortraitText::play_comm(arg0, sound_file, arg2);
		return 0;
	} catch (std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger wakeup_reason_wrapper(HSQUIRRELVM v) {
	wstring str;

	switch(world.script_manager.wakeup_reason) {
	case WET_NONE:
		str = L"none";
		break;

	case WET_TIME:
		str = L"time";
		break;

	case WET_PORTRAIT_FINISHED:
		str = L"portrait_finished";
		break;

	case WET_ALL_BIG_SELECTED:
		str = L"all_big_selected";
		break;

	case WET_NUMBER_HOTKEYED:
		str = L"number_hotkeyed";
		break;

	case WET_SHIPS_MOVING_DOWN:
		str = L"ships_moving_down";
		break;

	case WET_MISSIONS_OPEN:
		str = L"missions_open";
		break;

	case WET_TWO_FIGHTERS_SELECTED:
		str = L"two_fighter_squadrons_selected";
		break;

	case WET_RECON_TARGET_OPEN:
		str = L"recon_target_open";
		break;

	case WET_RECON_LAUNCHED:
		str = L"recon_launched";
		break;

	case WET_FREIGHTER_FOUND:
		str = L"freighter_found";
		break;

	case WET_BOMBERS_COMING_IN:
		str = L"bombers_coming_in";
		break;

	case WET_ATTACK_BOMBER_LAUNCHED:
		str = L"attack_bomber_launched";
		break;

	case WET_ENEMY_UNDER_ATTACK:
		str = L"enemy_under_attack";
		break;

	case WET_ALL_BIG_SHIPS_DEAD:
		str = L"all_big_ships_dead";
		break;

	case WET_CUSTOM_AI_TASK_0:
		str = L"custom_ai_task_0";
		break;

	case WET_CUSTOM_AI_TASK_1:
		str = L"custom_ai_task_1";
		break;

	case WET_CUSTOM_AI_TASK_2:
		str = L"custom_ai_task_2";
		break;

	case WET_CUSTOM_AI_TASK_3:
		str = L"custom_ai_task_3";
		break;

	case WET_CUSTOM_AI_TASK_4:
		str = L"custom_ai_task_4";
		break;
	}	

	try {
		sq_pushstring(v, str.c_str(), -1);
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger wait_for_portrait_wrapper(HSQUIRRELVM v) {
	try {
		world.script_manager.set_wakeup_event(WET_PORTRAIT_FINISHED);
		return sq_suspendvm(v);
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger wait_for_task_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	wstring type_str = arg0;
	WakeEventType type;
	if (type_str == L"all_big_selected")
		type = WET_ALL_BIG_SELECTED;
	else if (type_str == L"number_hotkeyed")
		type = WET_NUMBER_HOTKEYED;
	else if (type_str == L"ships_moving_down")
		type = WET_SHIPS_MOVING_DOWN;
	else if (type_str == L"missions_open")
		type = WET_MISSIONS_OPEN;
	else if (type_str == L"two_fighter_squadrons_selected")
		type = WET_TWO_FIGHTERS_SELECTED;
	else if (type_str == L"recon_target_open")
		type = WET_RECON_TARGET_OPEN;
	else if (type_str == L"recon_launched")
		type = WET_RECON_LAUNCHED;
	else if (type_str == L"freighter_found")
		type = WET_FREIGHTER_FOUND;
	else if (type_str == L"bombers_coming_in")
		type = WET_BOMBERS_COMING_IN;
	else if (type_str == L"attack_bomber_launched")
		type = WET_ATTACK_BOMBER_LAUNCHED;
	else if (type_str == L"enemy_under_attack")
		type = WET_ENEMY_UNDER_ATTACK;
	else if (type_str == L"all_big_ships_dead")
		type = WET_ALL_BIG_SHIPS_DEAD;
	else if (type_str == L"custom_ai_task_0")
		type = WET_CUSTOM_AI_TASK_0;
	else if (type_str == L"custom_ai_task_1")
		type = WET_CUSTOM_AI_TASK_1;
	else if (type_str == L"custom_ai_task_2")
		type = WET_CUSTOM_AI_TASK_2;
	else if (type_str == L"custom_ai_task_3")
		type = WET_CUSTOM_AI_TASK_3;
	else if (type_str == L"custom_ai_task_4")
		type = WET_CUSTOM_AI_TASK_4;
	else {
		sq_throwerror(v, _SC("Task type not recognised"));
		return SQ_ERROR;
	}

	try {
		world.script_manager.set_wakeup_event(type, arg1);
		world.script_manager.wakeup_reason = WET_NONE;
		return sq_suspendvm(v);
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger wait_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	try {
		world.script_manager.set_wakeup_event(WET_TIME, arg0);
		return sq_suspendvm(v);
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger suspend_wrapper(HSQUIRRELVM v) {
	if (!global_settings.debug)
		sq_throwerror(v, L"Suspend used outside of debug mode");
	world.script_manager.set_wakeup_event(WET_TIME, -1);
	return sq_suspendvm(v);
}

static SQInteger mission_complete_wrapper(HSQUIRRELVM v) {
	world.mission_completed = true;
	if (global_settings.max_mission <= world.mission_number) {
		global_settings.max_mission = world.mission_number + 1;
		global_settings.save_settings();
	}
	return 0;
}

static SQInteger end_mission_wrapper(HSQUIRRELVM v) {
	//index starts at 1
	if (world.mission_number == num_missions)
		gs_to = GST_CREDITS;
	else
		gs_to = GST_SELECT_MISSION;
	return 0;
}

static SQInteger toggle_radar_sharing_wrapper(HSQUIRRELVM v) {
	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].force_fill_fog(false);
	}

	world.radar_sharing = !world.radar_sharing;

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].force_fill_fog(true);
	}

	return 0;
}

static SQInteger toggle_display_all_groups_wrapper(HSQUIRRELVM v) {
	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].force_fill_fog(false);
	}

	debug_display_all_groups = !debug_display_all_groups;

	for (int i = 0; i != sides.size(); ++i) {
		for (int j = 0; j != sides[i].groups.size(); ++j)
			sides[i].groups[j].force_fill_fog(true);
	}

	return 0;
}

static SQInteger change_side_flag_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	sides[arg0].my_flag = arg1;

	return 0;
}

static SQInteger player_failed_mission_wrapper(HSQUIRRELVM v) {
	const SQChar* arg0;
	if (SQ_FAILED(sq_getstring(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not a string"));
		return SQ_ERROR;
	}

	my_windows.push_back(GenWindow(RTS_MISSION_FAILED, arg0));

	return sq_suspendvm(v);
}

static SQInteger which_side_dead_wrapper(HSQUIRRELVM v) {
	try {
		sq_pushinteger(v, world.script_manager.which_side_dead);
		return 1;
	} catch(std::exception& e) {
		throw_sq_error(v, e);
		return SQ_ERROR;
	}
}

static SQInteger pause_world_wrapper(HSQUIRRELVM v) {
	world.pause(true);

	return 0;
}

static SQInteger unpause_world_wrapper(HSQUIRRELVM v) {
	world.unpause(true);

	return 0;
}

static SQInteger cut_scene_on_wrapper(HSQUIRRELVM v) {
	world.cut_scene_on = true;
	return 0;
}

static SQInteger cut_scene_off_wrapper(HSQUIRRELVM v) {
	world.cut_scene_on = false;

	return 0;
}

static SQInteger planetary_bombardment_anim_wrapper(HSQUIRRELVM v) {
	world.script_anim = ANIM_BOMBARDMENT;

	return 0;
}

static SQInteger planetary_invasion_anim_wrapper(HSQUIRRELVM v) {
	world.script_anim = ANIM_INVASION;

	return 0;
}

static SQInteger ion_pulse_anim_wrapper(HSQUIRRELVM v) {
	world.do_ion_pulse();

	return 0;
}

static SQInteger end_anim_wrapper(HSQUIRRELVM v) {
	world.script_anim = ANIM_NONE;

	return 0;
}

static SQInteger set_viewpoint_wrapper(HSQUIRRELVM v) {
	SQInteger arg0;
	if (SQ_FAILED(sq_getinteger(v, 2, &arg0))) {
		sq_throwerror(v, _SC("Argument 1 not an integer"));
		return SQ_ERROR;
	}

	SQInteger arg1;
	if (SQ_FAILED(sq_getinteger(v, 3, &arg1))) {
		sq_throwerror(v, _SC("Argument 2 not an integer"));
		return SQ_ERROR;
	}

	world.viewx = arg0;
	world.viewy = arg1;

	return 0;
}

//registration functions

void register_function(HSQUIRRELVM v, const wstring& name, SQInteger (*p_func)(HSQUIRRELVM)) {
	sq_pushstring(v, name.c_str(), -1);
	sq_newclosure(v, p_func, 0);
	sq_createslot(v, -3);
}

void register_mission_functions(HSQUIRRELVM v) {
	sq_setprintfunc(v, print_func);

	sq_pushroottable(v);

	register_function(v, L"play_script_sound", &play_script_sound_wrapper);
	register_function(v, L"set_portrait", &set_portrait_wrapper);
	register_function(v, L"portrait_text", &portrait_text_wrapper);
	register_function(v, L"add_portrait", &add_portrait_wrapper);
	register_function(v, L"wakeup_reason", &wakeup_reason_wrapper);
	register_function(v, L"wait", &wait_wrapper);
	register_function(v, L"wait_for_portrait", &wait_for_portrait_wrapper);
	register_function(v, L"wait_for_task", &wait_for_task_wrapper);
	register_function(v, L"suspend", &suspend_wrapper);
	register_function(v, L"mission_complete", &mission_complete_wrapper);
	register_function(v, L"end_mission", &end_mission_wrapper);
	register_function(v, L"toggle_radar_sharing", &toggle_radar_sharing_wrapper);
	register_function(v, L"toggle_display_all_groups", &toggle_display_all_groups_wrapper);
	register_function(v, L"change_side_flag", &change_side_flag_wrapper);
	register_function(v, L"player_failed_mission", &player_failed_mission_wrapper);
	register_function(v, L"which_side_dead", &which_side_dead_wrapper);
	register_function(v, L"pause_world", &pause_world_wrapper);
	register_function(v, L"unpause_world", &unpause_world_wrapper);
	register_function(v, L"cut_scene_on", &cut_scene_on_wrapper);
	register_function(v, L"cut_scene_off", &cut_scene_off_wrapper);
	register_function(v, L"planetary_bombardment_anim", &planetary_bombardment_anim_wrapper);
	register_function(v, L"planetary_invasion_anim", &planetary_invasion_anim_wrapper);
	register_function(v, L"ion_pulse_anim", &ion_pulse_anim_wrapper);
	register_function(v, L"end_anim", &end_anim_wrapper);
	register_function(v, L"set_viewpoint", &set_viewpoint_wrapper);

	//pop root table
	sq_pop(v,1);
}

void register_ai_functions(HSQUIRRELVM v) {
	sq_setprintfunc(v, print_func);

	sq_pushroottable(v);

	//register class Squad
	sq_pushstring(v, L"Squad", -1);
	if (sq_newclass(v, SQFalse) < 0) {
		wostringstream msg;
		msg << L"Couldn't create new class 'Squad'";
		throw runtime_error(wstring_to_string(msg.str()));
	}

	register_function(v, L"select", &Squad_select_wrapper);
	register_function(v, L"highlight", &Squad_highlight_wrapper);
	register_function(v, L"is_moving", &Squad_is_moving_wrapper);
	register_function(v, L"ready_for_launch", &Squad_ready_for_launch_wrapper);
	register_function(v, L"get_in_hangar", &Squad_get_in_hangar_wrapper);
	register_function(v, L"get_type", &Squad_get_type_wrapper);
	register_function(v, L"get_pos", &Squad_get_pos_wrapper);
	register_function(v, L"get_distance", &Squad_get_distance_wrapper);
	register_function(v, L"get_distance_to_point", &Squad_get_distance_to_point_wrapper);

	if(SQ_FAILED(sq_createslot(v, -3)))
		throw runtime_error("Couldn't register class 'Squad'");
	
	register_function(v, L"order_move", &order_move_wrapper);
	register_function(v, L"order_halt", &order_halt_wrapper);
	register_function(v, L"add_waypoint", &add_waypoint_wrapper);
	register_function(v, L"launch_mission", &launch_mission_wrapper);
	register_function(v, L"order_recall", &order_recall_wrapper);
	register_function(v, L"order_return", &order_return_wrapper);
	register_function(v, L"ai_suspend", &ai_suspend_wrapper);
	register_function(v, L"world_frame_counter", &world_frame_counter_wrapper);
	register_function(v, L"wakeup_mission_script", &wakeup_mission_script_wrapper);
	register_function(v, L"is_cut_scene_on", &is_cut_scene_on_wrapper);

	//pop root table
	sq_pop(v,1);
}

}

