/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include <squirrel.h>

#include "../Globals.h"
#include "../Squadron.h"
#include "../Inlines.h"
#include "ScriptSquadron.h"

#include <stdexcept>
#include <sstream>
#include <string>

using std::wostringstream;
using std::runtime_error;
using std::wstring;

namespace Scripting {

void print_squirrel_stack(HSQUIRRELVM v) {
	wchar_t output[64];
    swprintf(output, 64, L"--------------------------------------------------------------");
	write_text(output, true);
    int count = sq_gettop(v);
    for(int i = 1; i <= count; ++i) {
        swprintf(output, 64, L"%d: ",i);
        switch(sq_gettype(v, i))
        {
            case OT_NULL:
                swprintf(output, 64, L"null");
				write_text(output, true);
                break;
            case OT_INTEGER: {
                SQInteger val;
                sq_getinteger(v, i, &val);
				swprintf(output, 64, L"integer (%d)", val);
				write_text(output, true);
                break;
            }
            case OT_FLOAT: {
                float val;
                sq_getfloat(v, i, &val);
				swprintf(output, 64, L"float (%f)", val);
				write_text(output, true);
                break;
            }
            case OT_STRING: {
                const wchar_t* val;
                sq_getstring(v, i, &val);
				swprintf(output, 64, L"string (%ls)", val);
				write_text(output, true);
                break;    
            }
            case OT_TABLE:
				swprintf(output, 64, L"table");
				write_text(output, true);
                break;
            case OT_ARRAY:
				swprintf(output, 64, L"array");
				write_text(output, true);
                break;
            case OT_USERDATA:
				swprintf(output, 64, L"userdata");
				write_text(output, true);
                break;
            case OT_CLOSURE:        
				swprintf(output, 64, L"closure(function)");
				write_text(output, true);
                break;
            case OT_NATIVECLOSURE:
				swprintf(output, 64, L"native closure(C function)");
				write_text(output, true);
                break;
            case OT_GENERATOR:
				swprintf(output, 64, L"generator");
				write_text(output, true);
                break;
            case OT_USERPOINTER:
				swprintf(output, 64, L"userpointer");
				write_text(output, true);
                break;
            case OT_THREAD:
				swprintf(output, 64, L"thread");
				write_text(output, true);
                break;
            case OT_CLASS:
				swprintf(output, 64, L"class");
				write_text(output, true);
                break;
            case OT_INSTANCE:
				swprintf(output, 64, L"instance");
				write_text(output, true);
                break;
            default:
				swprintf(output, 64, L"unknown?!?");
				write_text(output, true);
                break;
        }
    }
	swprintf(output, 64, L"--------------------------------------------------------------");
	write_text(output, true);
}

void create_squirrel_instance(HSQUIRRELVM v, ScriptSquadron* object) {
	sq_pushroottable(v);
	sq_pushstring(v, L"Squad", -1);

	if (SQ_FAILED(sq_get(v, -2))) {
		wostringstream msg;
		msg << L"Couldn't resolve squirrel type 'Squad'";
		throw runtime_error(wstring_to_string(msg.str()));
	}

	if (SQ_FAILED(sq_createinstance(v, -1)) || SQ_FAILED(sq_setinstanceup(v, -1, object))) {
		wostringstream msg;
		msg << L"Couldn't setup squirrel instance for object of type 'Squad'";
		throw runtime_error(wstring_to_string(msg.str()));
	}

	sq_remove(v, -2); // remove object name
	sq_remove(v, -2); // remove root table
}

void expose_squads(HSQUIRRELVM v, int side, const wstring& array_name) {
	sq_pushroottable(v);

	sq_pushstring(v, array_name.c_str(), -1);
	sq_newarray(v, 0);
	for (int i = 0; i != sides[side].squadrons.size(); ++i) {
		ScriptSquadron* interface = new ScriptSquadron(&(sides[side].squadrons[i]));
		create_squirrel_instance(v, interface);
		sq_arrayappend(v, -2);
	}

	// register array in root table
	if (SQ_FAILED(sq_createslot(v, -3))) {
		wostringstream msg;
		msg << L"Couldn't register " << array_name << " array in squirrel table";
		throw runtime_error(wstring_to_string(msg.str()));
	}

	//pop root table
	sq_pop(v,1);
}

void delete_squad(HSQUIRRELVM v, int side, int squad, const wstring& array_name) {
	sq_pushroottable(v);
	sq_pushstring(v, array_name.c_str(), -1);

	if (SQ_FAILED(sq_get(v, -2))) {
		wostringstream msg;
		msg << L"Couldn't resolve " << array_name;
		throw runtime_error(wstring_to_string(msg.str()));
	}

	//function called separately for each squad which dies, so as soon as one is found to be dead, stop iteration
	SQUnsignedInteger current_index = 0;
	bool dead_found = false;

	//iterate through array
	sq_pushnull(v);  //null iterator
	while(!dead_found && SQ_SUCCEEDED(sq_next(v,-2))) {
		//here -1 is the value and -2 is the key
		SQUserPointer data;
		if (SQ_FAILED(sq_getinstanceup(v, -1, &data, 0))) {
			wostringstream msg;
			msg << L"Couldn't get Squad instance";
			throw runtime_error(wstring_to_string(msg.str()));
		}

		ScriptSquadron* _this = reinterpret_cast<ScriptSquadron*> (data);
		if (_this->squad->my_squad == squad) {
			//delete actual object		
			delete _this;
			
			dead_found = true;
			sq_pop(v,2); //pops key and val
			break;
		}

		sq_pop(v,2); //pops key and val before the next iteration
		++current_index;
	}
	sq_pop(v,1); //pops the null iterator

	//this function was added to the squirrel sources by me, not a standard function
	sq_arrayremove(v, -1, current_index);

	sq_pop(v,1); //pop root table
}

void delete_script_squads(HSQUIRRELVM v, const wstring& array_name) {
	sq_pushroottable(v);
	sq_pushstring(v, array_name.c_str(), -1);

	if (SQ_FAILED(sq_get(v, -2))) {
		wostringstream msg;
		msg << L"Couldn't resolve " << array_name;
		throw runtime_error(wstring_to_string(msg.str()));
	}

	//iterate through array
	sq_pushnull(v);  //null iterator

	while(SQ_SUCCEEDED(sq_next(v,-2))) {
		//here -1 is the value and -2 is the key
		SQUserPointer data;
		if (SQ_FAILED(sq_getinstanceup(v, -1, &data, 0))) {
			wostringstream msg;
			msg << L"Couldn't get Squad instance";
			throw runtime_error(wstring_to_string(msg.str()));
		}

		ScriptSquadron* _this = reinterpret_cast<ScriptSquadron*> (data);
		delete _this;
		sq_pop(v,2); //pops key and val before the next iteration
	}

	sq_pop(v,1); //pops the null iterator
	sq_pop(v,1); //pop root table
}

}
