/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "GenWindow.h"
#include "Globals.h"
#include "Group.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Stuff.h"
#include "World.h"

using std::runtime_error;
using std::string;
using std::wstring;

/*
ultimately we want x and y where z is 1
first we work out z using dx and dy given
dz^2 = dx^2 + dy ^2
then we find out what x and y are if z is reduced to 1
propx = dx / dz
propy = dy / dz
*/
int get_move_props(float& propx, float& propy, float dx, float dy) {
	propx = 0;
	propy = 0;
	float adx = fabs(dx);
	float ady = fabs(dy);

	float dz = dx*dx + dy*dy;
	dz = std::sqrt(dz);

	//don't divide by 0!
	if (dz) {
		if (dx < 0)
			propx = -(adx / dz);
		else
			propx = adx / dz;

		if (dy < 0)
			propy = -(ady / dz);
		else
			propy = ady / dz;

		return 1;
	}
	else
		return 0;
}

int test_overlap(const Rect32& rect_one, const Rect32& rect_two) {
	int oneRight = rect_one.x + rect_one.w - 1;
	int oneBottom = rect_one.y + rect_one.h - 1;

	int twoRight = rect_two.x + rect_two.w - 1;
	int twoBottom = rect_two.y + rect_two.h - 1;

	//one inside two
	if (rect_two.x <= rect_one.x && rect_one.x <= twoRight
	        && rect_two.y <= rect_one.y && rect_one.y <= twoBottom)
		return 1;

	if (rect_two.x <= oneRight && oneRight <= twoRight
	        && rect_two.y <= rect_one.y && rect_one.y <= twoBottom)
		return 1;

	if (rect_two.x <= rect_one.x && rect_one.x <= twoRight
	        && rect_two.y <= oneBottom && oneBottom <= twoBottom)
		return 1;

	if (rect_two.x <= oneRight && oneRight <= twoRight
	        && rect_two.y <= oneBottom && oneBottom <= twoBottom)
		return 1;

	//two inside one
	if (rect_one.x <= rect_two.x && rect_two.x <= oneRight
	        && rect_one.y <= rect_two.y && rect_two.y <= oneBottom)
		return 1;

	if (rect_one.x <= twoRight && twoRight <= oneRight
	        && rect_one.y <= rect_two.y && rect_two.y <= oneBottom)
		return 1;

	if (rect_one.x <= rect_two.x && rect_two.x <= oneRight
	        && rect_one.y <= twoBottom && twoBottom <= oneBottom)
		return 1;

	if (rect_one.x <= twoRight && twoRight <= oneRight
	        && rect_one.y <= twoBottom && twoBottom <= oneBottom)
		return 1;

	//rect one left side runs across rect two (think of a cross)
	if (rect_two.x <= rect_one.x && rect_one.x <= twoRight
	        && rect_two.y >= rect_one.y && oneBottom >= twoBottom)
		return 1;

	//rect two left side runs across rect one (think of a cross)
	if (rect_one.x <= rect_two.x && rect_two.x <= oneRight
	        && rect_one.y >= rect_two.y && twoBottom >= oneBottom)
		return 1;

	return 0;
}

//FIXME these values found through trial and error. I think maybe the way OpenGL uses a different coordinate system
//to SDL causes confusion? Or anyway for some reason the 3 "1"s are somewhat random.
void draw_rect_border(SDL_Rect& the_rect, const OpenGLColor& color, float width) {
	float top = static_cast<float>(the_rect.y);
	float left = static_cast<float>(the_rect.x - 1);
	float bottom = static_cast<float>(the_rect.y + the_rect.h + 1);
	float right = static_cast<float>(the_rect.x + the_rect.w);

	//top+bottom
	display.draw_line(left, top, right, top, width, color);
	display.draw_line(left, bottom, right, bottom, width, color);
	//left+right
	display.draw_line(left, top, left, bottom, width, color);
	display.draw_line(right, top - 1, right, bottom, width, color);
}

void error_back_to_mm(const wstring& error) {
	write_log(error);
	global_error_string = error;
	gs_to = GST_MAIN_MENU;
}

const string make_pic_dir_string(int n_side, const string& end_of_dir, bool in_mission_folder) {
	string ret;
	if (in_mission_folder)
		ret = world.mission_folder + "unitpictures/";
	else
		ret = "unitpictures/";

	if (sides[n_side].color.sdl_color == standard_colors.side_red.sdl_color)
		ret += "red/";
	else if (sides[n_side].color.sdl_color == standard_colors.side_green.sdl_color)
		ret += "green/";
	else if (sides[n_side].color.sdl_color == standard_colors.side_blue.sdl_color)
		ret += "blue/";
	else if (sides[n_side].color.sdl_color == standard_colors.side_yellow.sdl_color)
		ret += "yellow/";

	ret += end_of_dir;

	return ret;
}

void follow_view_center() {
	//scrolling via following a group
	//must be done at the end of each frame,
	//because we don't want it to constantly flicker
	//due to being a frame out of date
	if (world.view_side != -1) {
		CoordsFloat center = sides[world.view_side].squadrons[world.view_squad].get_center();

		world.viewx = static_cast<int>(center.x) - (global_settings.screen_width / 2);
		world.viewy = static_cast<int>(center.y) - (global_settings.screen_height / 2);
	}
}

const ustring str_to_ustr(const string& str) {
	ustring ret;
	string::const_iterator l_iter = str.begin();
	string::const_iterator l_end =  str.end();
	
	while (l_iter != l_end) {
		ret.push_back(*l_iter);
		++l_iter;
	}
	
	return ret;
}

const string ustr_to_str(const ustring& ustr) {
	string ret;
	ustring::const_iterator l_iter = ustr.begin();
	ustring::const_iterator l_end =  ustr.end();
	
	while (l_iter != l_end) {
		ret.push_back(*l_iter);
		++l_iter;
	}
	
	return ret;
}

//freeing pictures must be seperate from constructor for when being creating and destructed in the process of being added to a vector
void clear_sides(){
	for (int i = 0; i != sides.size(); ++i)
		sides[i].free_pictures();

	sides.clear();
}

int sdl_key_to_number(int key) {
	switch(key) {
	case SDLK_0:
		return 0;
	case SDLK_1:
		return 1;
	case SDLK_2:
		return 2;
	case SDLK_3:
		return 3;
	case SDLK_4:
		return 4;
	case SDLK_5:
		return 5;
	case SDLK_6:
		return 6;
	case SDLK_7:
		return 7;
	case SDLK_8:
		return 8;
	case SDLK_9:
		return 9;
	default:
		throw runtime_error("Invalid group hot key");
	}
}

void create_info_string(const wstring& i_the_string, bool force) {
	if (!an_info_string || force)
		my_windows.push_back(GenWindow(i_the_string));
}


bool distance_within_range(CoordsFloat& point_one, CoordsFloat& point_two, float range) {
	//comparison done on squared distances so
	//we don't have to do square roots
	//because they get squared it doesn't matter if these
	//answers are negative or positive
	float dx = point_one.x - point_two.x;
	float dy = point_one.y - point_two.y;

	if (range * range >= dx*dx + dy*dy)
		return true;
	return false;
}


