/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Globals.h"
#include "Group.h"
#include "SettingsStruct.h"
#include "Squadron.h"

using std::vector;
using std::wstring;

Squadron::Squadron(int i_my_side, int i_my_squad, int i_group_one, int i_group_two):
my_side(i_my_side), my_squad(i_my_squad), my_group_one(i_group_one), my_group_two(i_group_two),
group_one(0), group_two(0), selected(false), hot_key(0), exposed_to_sq(true) {
}

void Squadron::keyboard(SDL_keysym& keysym) {
	if (!get_alive())
		return;
	
	if (keysym.sym >=SDLK_0 && keysym.sym <= SDLK_9 && keysym.mod & KMOD_CTRL && selected)
		hot_key = keysym.sym;
	else if (keysym.sym == hot_key && keysym.mod & KMOD_CTRL && !selected)
		hot_key = 0;
	else if (keysym.sym == hot_key) {
		sides[0].select_squad(my_side, my_squad);
		if (SDL_GetModState() & KMOD_ALT) {
			CoordsFloat center = get_center();
			world.viewx = static_cast<int>(center.x) - (global_settings.screen_width / 2);
			world.viewy = static_cast<int>(center.y) - (global_settings.screen_height / 2);
		}
	} else if (selected && keysym.sym >=SDLK_0 && keysym.sym <= SDLK_9) {
		sides[0].deselect_squad(my_side, my_squad);
	} else if (global_settings.debug && selected && keysym.sym == SDLK_F4) {
		group_one->destroy();
		if (my_group_two != -1)
			group_two->destroy();
	}
}

void Squadron::drag_select(SDL_Rect& drag_box) {
	if (group_one->alive && group_one->drag_select(drag_box)
	|| (my_group_two != -1 && group_two->alive && group_two->drag_select(drag_box)))
		sides[0].select_squad(my_side, my_squad);
	else if (!(SDL_GetModState() & KMOD_SHIFT))
		sides[0].deselect_squad(my_side, my_squad);
}

void Squadron::init_pointer(Group_Base* pointer, int which_group) {
	if (which_group == my_group_one)
		group_one = pointer;
	else
		group_two = pointer;
}

void Squadron::select() {
	selected = true;
	if (!is_small())
		return;

	world.view_side = my_side;
	world.view_squad = my_squad;
}

void Squadron::unselect() {
	selected = false;
}

void Squadron::add_waypoint(CoordsFloat waypoint) {
	group_one->add_waypoint(waypoint);
	group_two->add_waypoint(waypoint);
}

void Squadron::add_current_pos_as_waypoint() {
	group_one->add_waypoint(group_one->get_center());
	group_two->add_waypoint(group_one->get_center());
}

void Squadron::launch(const std::wstring& i_ai_filename, int mission_slot) {
	int parent = group_one->my_parent_cap;
	int launch_slot = sides[my_side].groups[parent].get_launch_slot();
	group_one->launch(i_ai_filename, mission_slot, launch_slot);
	group_two->launch(i_ai_filename, mission_slot, launch_slot);
}

void Squadron::change_ai_script(const std::wstring& i_ai_filename) {
	group_one->change_ai_script(i_ai_filename);
	group_two->change_ai_script(i_ai_filename);
}

void Squadron::order_move(float x, float y) {
	if (group_one->alive) group_one->order_move(x, y);
}

void Squadron::order_halt() {
	if (group_one->alive) group_one->order_halt();
}

UnitType Squadron::get_type() const {return group_one->get_type();}

const wstring Squadron::get_unit_name() const {return group_one->data_filename;}

bool Squadron::is_small() const {return group_one->is_small();}

bool Squadron::is_big_ship() const {return group_one->is_big_ship();}

int Squadron::get_parent_cap() const {return group_one->my_parent_cap;}

bool Squadron::both_in_hangar() const {
	if ((group_one->get_in_hangar() || !group_one->alive)
	&& (group_two->get_in_hangar() || !group_two->alive))
		return true;
	else
		return false;
}

bool Squadron::either_in_hangar() const {
	if (group_one->get_in_hangar() || group_two->get_in_hangar())
		return true;
	else
		return false;
}

bool Squadron::get_alive() const {
	if (group_one->alive || (my_group_two != -1 && group_one->alive))
		return true;
	return false;
}

bool Squadron::is_first(int group) const {
	if (group == my_group_one)
		return true;
	else
		return false;
}

int Squadron::get_other_wing (int you) const {
	if (you == my_group_one)
		return my_group_two;
	else
		return my_group_one;
}

CoordsInt Squadron::get_starting_coords() const {
	Rect32 tmp_rect;
	group_one->get_rect(tmp_rect);

	return CoordsInt(static_cast<int>(tmp_rect.x), static_cast<int>(tmp_rect.y));
}

bool Squadron::get_pic_flip() const {
	return group_one->units[0].get_pic_flip();
}

//we only start to refuel when both groups are home, so both always have the same fuel, unless dead
int Squadron::get_fuel() const {
	int fuel_current = 0;
	if (group_one->alive)
		fuel_current = group_one->fuel_current;
	else if (group_two->alive)
		fuel_current = group_two->fuel_current;
	return fuel_current;
}

//we only start to refuel when both groups are home, so both always have the same fuel, unless dead
int Squadron::get_fuel_percent() const {
	int fuel_current = get_fuel();
	
	int fuel_max = equip_lookup[L"Fuel"].max;

	return fuel_current * 100 / fuel_max;
}

int Squadron::get_big_ammo() const {
	//group checks for alive units
	return group_one->get_big_ammo()+ group_two->get_big_ammo();
}

int Squadron::get_ammo_percent() const {
	int ammo_max = group_one->units[0].get_big_ammo_max() * (group_one->units_left + group_two->units_left);
	int ammo_current = get_big_ammo();

	//don't divide by 0
	if (ammo_max == 0)
		return 0;	

	return ammo_current * 100 / ammo_max;
}

int Squadron::get_health() const {
	int ret = group_one->get_health();
	if (my_group_two != -1)
		ret += group_two->get_health();
	return ret;
}

int Squadron::get_shield() const {
	int ret = group_one->get_shield();
	if (my_group_two != -1)
		ret += group_two->get_shield();
	return ret;
}

int Squadron::get_armour() const {
	int ret = group_one->get_armour();
	if (my_group_two != -1)
		ret += group_two->get_armour();
	return ret;
}

int Squadron::get_shield_max() const {
	int ret = group_one->get_shield_max();
	if (my_group_two != -1)
		ret += group_two->get_shield_max();
	return ret;
}

int Squadron::get_armour_max() const {
	int ret = group_one->get_armour_max();
	if (my_group_two != -1)
		ret += group_two->get_armour_max();
	return ret;
}

const wstring Squadron::get_current_mission() const {
	if (group_one->alive && !group_one->get_in_hangar())
		return group_one->get_current_mission();
	else if (my_group_two != -1 && group_two->alive && !group_two->get_in_hangar())
		return group_two->get_current_mission();
	else return
		L"Dead or in hangar";
}

void Squadron::upkeep() {
	if (!exposed_to_sq)
		return;

	//out of scanning range? If so, deselect
	if (!sides[0].scanned_groups[my_side][my_group_one]
	&& (my_group_two == -1 || !sides[0].scanned_groups[my_side][my_group_two])
	&& get_type() != UT_PLANET)
		sides[0].deselect_squad(my_side, my_squad);

	//if dead, unexpose from squirrel and deselect
	if (!get_alive()) {
		world.script_manager.delete_squad(my_side, my_squad);

		//remove player selection
		if (selected)	
			sides[0].deselect_squad(my_side, my_squad);

		//remove ai selection
		if (my_side != 0)
			sides[my_side].deselect_squad(my_side, my_squad);

		//remove highlight
		sides[my_side].unhighlight_squadron(my_squad);

		exposed_to_sq = false;
	}
}

//for squirrel, only called if squad is alive
bool Squadron::is_moving() const {
	if ((group_one->the_commands.move_command != MC_NO_MOVE)
	|| (my_group_two != -1 && group_two->the_commands.move_command != MC_NO_MOVE))
		return true;
	return false;
}
bool Squadron::ready_for_launch() const {
	if (get_fuel_percent() == 100 && both_in_hangar())
		return true;
	return false;
}

CoordsFloat Squadron::get_pos() const {
	if (group_one->alive)
		return group_one->get_center();
	else
		return group_two->get_center();
}

int Squadron::find_distance_to(int n_side, int n_squad) const {
	CoordsFloat my_pos = get_pos();
	CoordsFloat other_pos = sides[n_side].squadrons[n_squad].get_pos();
	CoordsFloat d(my_pos.x - other_pos.x, my_pos.y - other_pos.y);
	return static_cast<int>(slow_dist(d.x, d.y));
}

int Squadron::find_distance_to_point(int x, int y) const {
	CoordsFloat my_pos = get_pos();
	CoordsFloat d(my_pos.x - x, my_pos.y - y);
	return static_cast<int>(slow_dist(d.x, d.y));
}

CoordsFloat Squadron::get_center() const {
	if (my_group_two == -1 || group_one->alive)
		return group_one->get_center();
	else
		return group_two->get_center();
}

const std::wstring Squadron::get_engine_name() const {return group_one->units[0].get_engine_name();}
const std::wstring Squadron::get_shield_name() const {return group_one->units[0].get_shield_name();}
const std::wstring Squadron::get_armour_name() const {return group_one->units[0].get_armour_name();}
WeaponType Squadron::get_big_type() const {return group_one->units[0].get_big_type();}
WeaponType Squadron::get_small_type() const {return group_one->units[0].get_small_type();}
const std::wstring Squadron::get_blurb() const {return group_one->units[0].get_blurb();}

int Squadron::get_units_left() const {
	int ret = group_one->units_left;
	if (my_group_two != -1)
		ret += group_two->units_left;
	return ret;
}

const AICommands* Squadron::get_the_commands() const {return &(group_one->the_commands);}
int Squadron::get_script_var(int which) const {return group_one->ai_interpreter.script_vars[which];}

// assumes only called for squads with one unit
void Squadron::get_rect(Rect32& give_rect) const {return group_one->get_rect(give_rect);}
// assumes only called for squads with one unit
int Squadron::get_small_number() const {return group_one->units[0].get_small_number();}

