/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "GenWindow.h"
#include "Group.h"
#include "FileUtils.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Squadron.h"

#include <fstream>
#include <sstream>
#include <iterator>
#include <stdexcept>
#include <algorithm>
#include <map>

using std::endl;
using std::list;
using std::map;
using std::ofstream;
using std::runtime_error;
using std::string;
using std::wstring;
using std::vector;

Side::Side(const wstring& i_name, int i_flag):
name(i_name), my_flag(i_flag), already_ai_error(false) {
	my_side = sides.size();
}

void Side::free_pictures() {
	for (map<wstring, ScaledPic>::iterator iter = unit_pictures.begin(); iter != unit_pictures.end(); ++iter)
		glSDL_FreeSurface(iter->second.pic);
}

void Side::load_data() {
	string concat_string = world.mission_folder + wstring_to_string(name) + ".dat";

	FileReader file_reader(concat_string);

	string color_string = file_reader.get_string_after_colon();
	set_colors(color_string);

	int i = 0;
	int current_parent = -1;
	
	while (!file_reader.at_end()) {
		wstring type_string = string_to_wstring(file_reader.get_string_after_colon());
		wstring unit_name = string_to_wstring(file_reader.get_string_after_colon());
		CoordsInt starting_coords;
		starting_coords.x = file_reader.get_int_after_colon();
		starting_coords.y = file_reader.get_int();
		bool face_left = static_cast<bool>(file_reader.get_int());

		int squad = squadrons.size();

		if (type_string != L"Fighter" && type_string != L"Bomber") {
			squadrons.push_back(Squadron(my_side, squad, i, -1));
			groups.push_back(Group(my_side, squad, i, unit_name, starting_coords, face_left));
			current_parent = i;
			++i;
		} else {
			squadrons.push_back(Squadron(my_side, squad, i, i + 1));
			groups.push_back(Group(my_side, squad, i, unit_name, current_parent, starting_coords));
			groups.push_back(Group(my_side, squad, i + 1, unit_name, current_parent, starting_coords));
			i += 2;
		}
	}
}

void Side::init() {
	for (int i = 0; i != sides.size(); ++i) {
		//ai sides don't do scanning, they just treat everyone as always scanned
		if (sides[i].my_flag == my_flag || my_side != 0)	 
			scanned_groups.push_back(vector<bool>(sides[i].groups.size(), true));	 
		else	 
			scanned_groups.push_back(vector<bool>(sides[i].groups.size(), false));	 
	}

	missions.resize(max_simultaneous_missions);
	start_big_ships = get_total_big_ships();
}

//only run by player side
void Side::blank_scanned_groups() {
	for (int i = 0; i != scanned_groups.size(); ++i) {	 
		if (sides[i].my_side == my_side || (sides[i].my_flag == my_flag && world.radar_sharing))	 
			//if radar_sharing was turned off and then back on again this would need to set them back to true again
			//and not just continue
			continue;	 

		for (int j = 0; j != scanned_groups[i].size(); ++j)	 
			scanned_groups[i][j] = false;	 
	}
}

void Side::check_if_scanned() {
	for (int i  = 0; i != groups.size(); ++i)
		groups[i].check_if_scanned();
}

void Side::run_ai() {
	//check if any save groups have gone out of scanning range - if so, forget them
	//the rest start off invalid
	for (vector<Mission>::iterator mission = missions.begin(); mission != missions.end(); ++mission) {
		for (int i = 0; i != n_ai_vars; ++i) {
			if (mission->save_groups[i].x != -1 && !scanned_groups[mission->save_groups[i].x][mission->save_groups[i].y]) {
				mission->save_groups[i].x = -1;
				mission->save_groups[i].y = -1;
			}
		}
	}
	
	for (int i  = 0; i != groups.size(); ++i)
		groups[i].run_group_ai();
}

void Side::save_to_file() {
	string concat_string = world.mission_folder + wstring_to_string(name) + ".dat";

	ofstream output(concat_string.c_str(), std::ios::trunc | std::ios::out);

	output << "Color: " << wstring_to_string(color_to_string()) << endl << endl;

	for (int i = 0; i != squadrons.size(); ++i) {
		UnitType type = squadrons[i].get_type();
		output << "Type: " << wstring_to_string(unit_type_to_string[type]) << endl;

		output << "Data: " << wstring_to_string(squadrons[i].get_unit_name()) << endl;

		CoordsInt start_coords = squadrons[i].get_starting_coords();
		output << "Starting coords: " << start_coords.x << ", " << start_coords.y << ", " << squadrons[i].get_pic_flip() << endl;

		output << endl;
	}

	//an extra blank line on the end for luck
	output << endl;
}

int Side::get_total_units() {
	int total = 0;

	for (int i = 0; i != squadrons.size(); ++i)
		total += squadrons[i].get_units_left();

	return total;
}

int Side::get_total_health() {
	int total = 0;

	for (int i = 0; i != squadrons.size(); ++i)
		total += squadrons[i].get_health();

	return total;
}

int Side::get_total_big_ships() {
	int total = 0;

	for (int i = 0; i != squadrons.size(); ++i) {
		if (squadrons[i].is_big_ship())
			total += squadrons[i].get_units_left();
	}

	return total;
}

int Side::get_mission_slot() {
	for (int i = 0; i != missions.size(); ++i) {
		bool slot_free = true;

		for (int j = 0; j != groups.size(); ++j) {
			if (groups[j].get_alive() && groups[j].get_mission_slot() == i) {
				slot_free = false;
				break;
			}
		}

		if (!slot_free)
			continue;

		missions[i].reset();
		return i;
	}

	return -1;
}

void Side::set_colors(const string& color_string) {
	if (color_string == "blue") {
		color = standard_colors.side_blue;
		radar_color = standard_colors.radar_blue;
		//green looks better
		laser_color = standard_colors.laser_green;
	} else if (color_string == "red") {
		color = standard_colors.side_red;
		radar_color = standard_colors.radar_red;
		laser_color = standard_colors.laser_red;
	} else if (color_string == "green") {
		color = standard_colors.side_green;
		radar_color = standard_colors.radar_green;
		laser_color = standard_colors.laser_green;
	} else if (color_string == "yellow") {
		color = standard_colors.side_yellow;
		radar_color = standard_colors.radar_yellow;
		laser_color = standard_colors.laser_yellow;
	}
}

const wstring Side::color_to_string() {
	if (color.sdl_color == standard_colors.side_blue.sdl_color)
		return L"blue";
	else if (color.sdl_color == standard_colors.side_red.sdl_color)
		return L"red";
	else if (color.sdl_color == standard_colors.side_green.sdl_color)
		return L"green";
	else if (color.sdl_color == standard_colors.side_yellow.sdl_color)
		return L"yellow";
	else
		return L"";
}

void Side::select_squad(int side, int squad) {
	CoordsInt this_squad(side, squad);

	//don't select squadrons from multiple sides at once
	if (selected_squads.size() && selected_squads.front().x != side) {
		selected_squads.clear();
		selected_squads.push_back(this_squad);
		return;
	}

	if (find(selected_squads.begin(), selected_squads.end(), this_squad) == selected_squads.end()) {
		selected_squads.push_back(this_squad);
	}
}

void Side::deselect_squad(int side, int squad) {
	CoordsInt this_squad(side, squad);
	list<CoordsInt>::iterator iter = find(selected_squads.begin(), selected_squads.end(), this_squad);
	if (iter != selected_squads.end())
		selected_squads.erase(iter);
}

void Side::toggle_highlight_squadron(int squad) {
	list<int>::iterator iter = find(highlighted_squads.begin(), highlighted_squads.end(), squad);
	if (iter != highlighted_squads.end())
		highlighted_squads.erase(iter);
	else
		highlighted_squads.push_back(squad);

	message_windows(WC_NO_SQUAD_HIGHLIGHT, 0, 0, window_id_all, window_id_none);

	//if noone highlighted, then no icons
	if (highlighted_squads.size() == 0)
		return;

	//if any big ships or ships not at home selected, then return
	iter = highlighted_squads.begin();
	for (; iter != highlighted_squads.end(); ++iter) {
		if (!squadrons[*iter].is_small() || !squadrons[*iter].both_in_hangar()) {
			message_windows(WC_SQUAD_AWAY_HIGHLIGHT, 0, 0, window_id_all, window_id_none);
			return;
		}
	}

	message_windows(WC_SQUAD_HOME_HIGHLIGHT, 0, 0, window_id_all, window_id_none);
}

CoordsFloat Side::work_out_selected_center() {
	int most_left = world.width;
	int most_right = 0;
	int most_top = world.height;
	int most_bottom = 0;
	Rect32 tmp_rect;

	for (list<CoordsInt>::iterator iter = selected_squads.begin(); iter != selected_squads.end(); ++iter) {
		if (!sides[iter->x].squadrons[iter->y].is_small()) {
			sides[iter->x].squadrons[iter->y].get_rect(tmp_rect);
			if (tmp_rect.x < most_left)
				most_left = tmp_rect.x;
			if (tmp_rect.x + tmp_rect.w > most_right)
				most_right = tmp_rect.x + tmp_rect.w;
			if (tmp_rect.y < most_top)
				most_top = tmp_rect.y;
			if (tmp_rect.y + tmp_rect.h > most_bottom)
				most_bottom = tmp_rect.y + tmp_rect.h;
		}
	}
	
	float centerx = static_cast<float>(most_left + ((most_right - most_left) / 2));
	float centery = static_cast<float>(most_top + ((most_bottom - most_top) / 2));
	return CoordsFloat(centerx, centery);
}

void Side::launch_mission(const wstring& ai_script) {
	int mission_slot = get_mission_slot();

	if (mission_slot == -1) {
		create_info_string(L"Maximum simultaneous missions reached, please try again in a moment");
		return;
	}

	for (list<int>::iterator iter = highlighted_squads.begin(); iter != highlighted_squads.end(); ++iter) {
		squadrons[*iter].launch(ai_script, mission_slot);
		if (my_side == 0)
			select_squad(0, *iter);

		if (my_side == 0 && ai_script == L"attack" && squadrons[*iter].get_type() == UT_BOMBER)
			world.script_manager.fire_wakeup_event(WET_ATTACK_BOMBER_LAUNCHED);
	}

	if (my_side == 0 && ai_script != L"recall")
		sound.play_sound(SE_MISSION_UNDER_WAY);

	if (my_side == 0 && ai_script == L"recon")
		world.script_manager.fire_wakeup_event(WET_RECON_LAUNCHED);
}

void Side::order_return() {
	for (list<CoordsInt>::iterator iter = selected_squads.begin(); iter != selected_squads.end(); ++iter)
		squadrons[iter->y].change_ai_script(L"return");

	if (my_side == 0)
		sound.play_sound(SE_HEADING_FOR_HOME);
}

void Side::order_recall() {
	highlighted_squads.clear();

	for (list<CoordsInt>::iterator iter = selected_squads.begin(); iter != selected_squads.end(); ++iter) {
		if (squadrons[iter->y].both_in_hangar()) {		
			highlighted_squads.push_back(iter->y);
			squadrons[iter->y].add_current_pos_as_waypoint();
		//don't give orders to half a squad only
		} else if (squadrons[iter->y].either_in_hangar())
			return;
		//both out of hangar
		else
			squadrons[iter->y].change_ai_script(L"recall");
	}

	if (highlighted_squads.size())
		launch_mission(L"recall");

	if (my_side == 0)
		sound.play_sound(SE_BE_THERE);
}

void Side::select_highlighted_squadrons() {
	for (list<int>::iterator iter = highlighted_squads.begin(); iter != highlighted_squads.end(); ++iter)
		select_squad(0, *iter);
}

void Side::unhighlight_squadron(int squad) {
	list<int>::iterator iter = find(highlighted_squads.begin(), highlighted_squads.end(), squad);
	if (iter != highlighted_squads.end())
		toggle_highlight_squadron(squad);
}

void Side::display_missions_fuel() {
	for (list<int>::iterator iter = highlighted_squads.begin(); iter != highlighted_squads.end(); ++iter) {
		int parent = squadrons[*iter].get_parent_cap();
		groups[parent].display_missions_fuel();
	}
}

void Side::hide_missions_fuel() {
	for (int i = 0; i != groups.size(); ++i)
		groups[i].hide_missions_fuel();
}

///

Mission::Mission() {
	reset();
}

void Mission::reset() {
	for (int i = 0; i != n_ai_vars; ++i)
		script_vars[i] = 0;

	//start off invalid
	for (int i = 0; i != n_ai_vars; ++i) {
		save_groups[i].x = -1;
		save_groups[i].y = -1;
	}
}

