/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "FileUtils.h"
#include "Globals.h"
#include "Group.h"
#include "Inlines.h"
#include "Projectile.h"
#include "RTSUnit_Base.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "World.h"

#include <sstream>
#include <iterator>

using std::string;
using std::wstring;

RTSUnit_Base::RTSUnit_Base(const wstring& i_name, int i_my_side, int i_my_group):
name(i_name), my_side(i_my_side), my_group(i_my_group),
pic(0), b_flip(false),
extra_move_frames(0), extra_move_x(0), extra_move_y(0), alive(1), shield_timer(0) {
	big_stage = W_READY;
	big_target.x = -1;
	big_target.y = -1;
}

void RTSUnit_Base::load_data() {
	string filename = "unitdata/" + wstring_to_string(name) + ".dat";
	if (!does_file_exist(filename))
		filename = world.mission_folder + "unitdata/" + wstring_to_string(name) + ".dat";
	FileReader the_file(filename);
	
	//type
	string temp_str = the_file.get_string_after_colon();
	my_type = string_to_unit_type[string_to_wstring(temp_str)];

	//picture
	pic_name = string_to_wstring(the_file.get_string_after_colon());

	//blurb - planets only
	if (my_type == UT_PLANET)
		blurb = string_to_wstring(the_file.get_string_after_colon());

	//engine
	engine_name = string_to_wstring(the_file.get_string_after_colon());

	//shield
	shield_name = string_to_wstring(the_file.get_string_after_colon());
	shield_current = shield_max = equip_lookup[shield_name].max;
	shield_recharge = equip_lookup[shield_name].recharge;

	//armour
	armour_name = string_to_wstring(the_file.get_string_after_colon());
	armour_current = armour_max = equip_lookup[armour_name].max;

	//small_type
	temp_str = the_file.get_string_after_colon();
	small_type = weapon_load_lookup[string_to_wstring(temp_str)];
	
	//big type
	temp_str = the_file.get_string_after_colon();	
	big_type = weapon_load_lookup[string_to_wstring(temp_str)];
	big_ammo = big_ammo_max = weapon_lookup[big_type].start_ammo;

	load_picture();
	type_dep_stats(the_file);

	init_small();
	big_stage = W_READY;
}

void RTSUnit_Base::init_small() {
	small_targets.resize(small_number, CoordsInt(-1, -1));
	small_timer.resize(small_number);
	small_aiming.resize(small_number);

	small_stage.resize(small_number, W_READY);
}

void RTSUnit_Base::load_picture() {
	if (sides[my_side].unit_pictures.find(pic_name) == sides[my_side].unit_pictures.end()) {
		string dir_name = make_pic_dir_string(my_side, wstring_to_string(pic_name), true);
		if (!does_file_exist(dir_name))
			dir_name = make_pic_dir_string(my_side, wstring_to_string(pic_name), false);

		sides[my_side].unit_pictures[pic_name] = ScaledPic();
		sides[my_side].unit_pictures[pic_name].load(dir_name);
	}

	pic = sides[my_side].unit_pictures[pic_name].pic;
	width = sides[my_side].unit_pictures[pic_name].w;
	height = sides[my_side].unit_pictures[pic_name].h;
}


void RTSUnit_Base::set_pos(float ix, float iy) {
	myx = ix;
	myy = iy;
}

void RTSUnit_Base::move(float distx, float disty) {
	myx += distx;
	myy += disty;

	//FIXME planets never flip because flipped blitting for huge images is not supported
	if (my_type == UT_PLANET)
		return;

	if (distx < 0)
		b_flip = true;
	else if (distx > 0)
		b_flip = false;
}

void RTSUnit_Base::add_extra_move_frames() {
	//we may be outside because of previous in-group movement,
	//or the bounding rect may have got smaller.

	Rect32 group_rect;
	sides[my_side].groups[my_group].get_rect(group_rect);

	if (myx < group_rect.x) {
		extra_move_frames = 1;
		extra_move_x = 0.5;
		extra_move_y = 0;
	}
	if (myy < group_rect.y) {
		extra_move_frames = 1;
		extra_move_x = 0;
		extra_move_y = 0.5;
	}
	if (myx + width > group_rect.x + group_rect.w) {
		extra_move_frames = 1;
		extra_move_x = -0.5;
		extra_move_y = 0;
	}
	if (myy + height > group_rect.y + group_rect.h) {
		extra_move_frames = 1;
		extra_move_x = 0;
		extra_move_y = -0.5;
	}

	//if we're inside the rect, add on a bit of random in group movement
	//for next time
	if (!extra_move_frames) {
		extra_move_x = static_cast<float>(rand() % 3 - 1) / 2;
		extra_move_y = static_cast<float>(rand() % 3 - 1) / 2;
		extra_move_frames = rand() % 30;
	}
}

void RTSUnit_Base::setup_small_for_firing(int which_small) {
	small_stage[which_small] = W_AIMING;
	small_timer[which_small] = world.frame_counter;
	small_aiming[which_small] = rand() % max_aiming;
}

void RTSUnit_Base::setup_big_for_firing() {
	big_stage = W_AIMING;	
	big_timer = world.frame_counter;
	big_aiming = rand() % max_aiming;
}

void RTSUnit_Base::been_hit(int power) {
	shield_current -= power;
	shield_timer = world.frame_counter;

	if (shield_current < 0) {
		armour_current -= abs(shield_current);
		shield_current = 0;
	}

	if (armour_current < 0)
		armour_current = 0;
}

void RTSUnit_Base::upkeep() {
	//shields
	if (shield_current < shield_max && world.frame_counter - shield_timer > shield_recharge) {
		++shield_current;
		shield_timer = world.frame_counter;
	}

	//check for big reloading
	if (big_stage == W_RELOADING && world.frame_counter - big_timer > weapon_lookup[big_type].reload)
		big_stage = W_READY;

	//check for big aiming (time decided when we choose a target)
	if (big_stage == W_AIMING && world.frame_counter - big_timer > big_aiming) {
		if (big_type == WT_LARGE) {
			world.play_sound(myx, myy, SE_LARGE_LASER);

			TargetDesc target_info;

			sides[big_target.x].groups[big_target.y].get_unit_target_info(target_info);

			big_target_unit = target_info.which_unit;
			target_weak_spot = target_info.weak_spot;

			big_timer = world.frame_counter;

			we_hit_with_big = Projectile_Base::check_to_hit(weapon_lookup[big_type].accuracy);

			if (!we_hit_with_big) {
				target_weak_spot.x += rand() % 21 - 10;
				target_weak_spot.y += rand() % 21 - 10;
			} else if (sides[big_target.x].groups[big_target.y].is_big_ship() && sides[big_target.x].groups[big_target.y].get_alive())
				world.projectiles.push_back(Projectile(target_info.currentx + target_weak_spot.x, target_info.currenty + target_weak_spot.y));
		}

		big_stage = W_FIRING;
	}
}

//only called if group is on the screen
void RTSUnit_Base::draw_self() {
	screen_rect.x = static_cast<int>(myx) - world.viewx;
	screen_rect.y = static_cast<int>(myy) - world.viewy;
	screen_rect.w = width;
	screen_rect.h = height;

	if (alive) {
		if (b_flip)
			display.blt(pic, screen_rect, GL_EFFECT_HFLIPPED | GL_EFFECT_SCALED);
		else
			display.blt(pic, screen_rect, GL_EFFECT_SCALED);
	} else if (!finished_anim)
		explode();
}

//relative to centre of unit
CoordsInt RTSUnit_Base::get_weak_spot() const {
	CoordsInt ret(0, 0);
    //relative to centre of unit
	return ret;
}

