/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#ifndef GUARD_RTS
#define GUARD_RTS

#include "DragWindow.h"
#include "GameState.h"
#include "Globals.h"
#include "Menu_Base.h"
#include "OptWindow.h"
#include "RTSStructs.h"
#include "Sound.h"

#include <map>

namespace RTS {

class RTS_State: public GameState {
public:
	RTS_State();
	~RTS_State();
	void main();
	
	virtual void mouse_d(Uint8 button, Uint16 x, Uint16 y);
	virtual void mouse_u(Uint8 button, Uint16 x, Uint16 y);
	virtual void mouse_m(Uint8 state, Uint16 x, Uint16 y);
	virtual void keyboard(SDL_keysym& keysym);

	static void play_alert(SoundEffect which_sound);

	static SDL_Rect radar_rect;
	static SDL_Rect bottom_left_box_rect;

private:
	void change_info_window();
	void draw_radar();

	bool radar_dragging;	
};

const std::wstring move_com_to_string(const AICommands& the_commands);
const std::wstring fire_com_to_string(const AICommands& the_commands);

class Portrait {
public:
	Portrait(int i_side, SDL_Surface* i_portrait): side(i_side), portrait(i_portrait) {}
	Portrait() {}

	SDL_Surface* portrait;
	int side;
};

class CommMessage {
public:
	CommMessage(const std::wstring i_title, const std::wstring& i_msg, const std::wstring& i_sound_file, bool i_align_bottom):
	title(i_title), msg(i_msg), sound_file(i_sound_file), align_bottom(i_align_bottom) {} 

	std::wstring title;
	std::wstring msg;
	std::wstring sound_file;
	bool align_bottom;
};

class PortraitText: public WrappedString {
public:
	PortraitText();
	~PortraitText();

	static void init();
	static void shutdown();
	static void add_portrait(const std::wstring& title, const std::wstring& filename, int side);
	static void set_portrait(const std::wstring& title, bool align_bottom);
	static void play_comm(const std::wstring& msg, const std::wstring& sound_file, bool i_unskippable);
	static void replay_last_comm();
	static void restart_current_comm();
	static void end_current_comm(bool skip = false);
	static void run();
	void update();
	//don't interfere with mouse clicks or cursors
	bool mouse_d(Uint8 button, Uint16 x, Uint16 y) {return false;}
	bool mouse_m(Uint8 state, Uint16 x, Uint16 y) {return false;}
	void draw_self();
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);

protected:
	CommMessage current_msg;
	SDL_Rect portrait_rect;
	int end_time;
	bool wait_for_sound;

	static std::map<std::wstring, Portrait> portraits;
	static std::list<CommMessage> messages;
	static std::list<CommMessage>::iterator next_msg;
	static std::wstring new_title;
	static bool new_align_bottom;

	static bool currently_open;
	static bool unskippable;

	static const int default_portrait_duration_per_word = 500;
};

class ControlIcon: public GenWindow_Base {
public:
	ControlIcon(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const std::wstring& i_desc, bool minus_width, bool i_active);
	ControlIcon(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const std::wstring& i_desc, int i_text_offset_x, bool i_active);

	void update();
	bool mouse_d(Uint8 button, Uint16 x, Uint16 y);
	bool mouse_m(Uint8 state, Uint16 x, Uint16 y);
	bool keyboard(SDL_keysym& keysym);
	void draw_self();
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);

	static const int icon_dim = 32;
	static const int icon_spacing = 8;

private:
	void init_vars(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const std::wstring& i_desc, bool i_active);

	WindowChoice type;
	SDL_Surface* pic;
	std::wstring desc;
	int text_offset_x;
	int text_offset_y;
	bool active;
	bool hover;
};

class Missions: public Menu_Base {
public:
	Missions();

	void update();
	void switch_on_choice(Uint16 x, Uint16 y);
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);
	void draw_self();

private:
	void new_column();

	int group_item_spacing;
	int group_item_width;
};

class TargetInput: public GenWindow_Base {
public:
	TargetInput(GenPicture i_cursor_type);
	virtual ~TargetInput();

	bool mouse_d(Uint8 button, Uint16 x, Uint16 y);	
	bool mouse_m(Uint8 state, Uint16 x, Uint16 y);
	bool keyboard(SDL_keysym& keysym);
	virtual void take_action(Uint16 x, Uint16 y) = 0;

private:
	GenPicture cursor_type;
	std::vector<int> parent_list;
};

class TargetMove: public TargetInput {
public:
	TargetMove(bool patrol);

	void take_action(Uint16 x, Uint16 y);	
	bool key_u(SDL_keysym& keysym);
	void draw_self();

private:
	bool b_patrol;
	bool got_waypoint;
};

class TargetAttack: public TargetInput {
public:
	TargetAttack();

	void take_action(Uint16 x, Uint16 y);
	void draw_self();
};

class SquadsWindow: public DragWindow {
public:
	SquadsWindow(int ix, int iy, int i_my_side, int i_my_squad);
	bool mouse_d(Uint8 button, Uint16 x, Uint16 y);
	virtual void draw_self() = 0;
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);

protected:
	void add_side_title();

	int my_side;
	int my_squad;
};

class SquadList: public Menu_Base {
public:
	SquadList(int ix, int iy, int i_my_side, int parent, int flags);

	void init_rects();
	void switch_on_choice(Uint16 x, Uint16 y);
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);

protected:
	int my_side;
};

class SquadStatsInfo: public SquadsWindow {
public:
	SquadStatsInfo(int ix, int iy, int i_my_side, int i_my_squad);

	void draw_self();

private:
	void draw_planet_info();
};

class SquadStatusReport: public SquadsWindow {
public:
	SquadStatusReport(int ix, int iy, int i_my_side, int i_my_squad);

	void draw_self();
};

class SquadVarsInfo: public SquadsWindow {
public:
	SquadVarsInfo(int ix, int iy, int i_my_side, int i_my_squad);

	void draw_self();
};

class DebugWindow: public DragWindow {
public:
	DebugWindow();
	void draw_self();
	void win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id);
};

class RestartQ: public Menu_Base {
public:
	RestartQ();
	void switch_on_choice(Uint16 x, Uint16 y);
};

class Briefing: public WrappedString {
public:
	Briefing();
	~Briefing();
};

class MissionFailed: public Menu_Base {
public:
	MissionFailed(const std::wstring& failure_msg);
	void switch_on_choice(Uint16 x, Uint16 y);
};

class Options: public Menu_Base {
public:
	Options();
	~Options();
	void switch_on_choice(Uint16 x, Uint16 y);
};

class GameSpeedSlider: public SliderWithUnits {
public:
	GameSpeedSlider(int ix, int iy, int flags);

	bool mouse_m(Uint8 state, Uint16 x, Uint16 y);
	void draw_self();
};

class ScrollSpeedSlider: public SliderWithUnits {
public:
	ScrollSpeedSlider(int ix, int iy, int flags);

	bool mouse_m(Uint8 state, Uint16 x, Uint16 y);
};

} //end namespace

#endif

