/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "Font.h"
#include "GenWindow_Base.h"
#include "Group.h"
#include "LoadGameData.h"
#include "RTS.h"
#include "SelectMission.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Sound.h"
#include "World.h"

#include <SDL_opengl.h>

#include <vector>
#include <list>

using std::find;
using std::list;
using std::map;
using std::runtime_error;
using std::string;
using std::wstring;
using std::vector;

namespace RTS {

//static members
SDL_Rect RTS_State::radar_rect;
SDL_Rect RTS_State::bottom_left_box_rect;

map<wstring, Portrait> PortraitText::portraits;
list<CommMessage> PortraitText::messages;
list<CommMessage>::iterator PortraitText::next_msg;
wstring PortraitText::new_title;
bool PortraitText::new_align_bottom;
bool PortraitText::currently_open = false;
bool PortraitText::unskippable = false;

RTS_State::RTS_State() {
	bottom_left_box_rect.w = 200;
	bottom_left_box_rect.h = 200;
	bottom_left_box_rect.x = radar_indent;
	bottom_left_box_rect.y = global_settings.screen_height - bottom_left_box_rect.h - radar_indent;

	SDL_Rect tmp_rect;
	tmp_rect.x = bottom_left_box_rect.x + bottom_left_box_rect.w;
	tmp_rect.y = bottom_left_box_rect.y + ControlIcon::icon_spacing;
	tmp_rect.w = ControlIcon::icon_dim;
	tmp_rect.h = ControlIcon::icon_dim;

	my_windows.push_back(GenWindow(RTS_MISSIONS, tmp_rect, gen_pictures[GENPIC_MISSIONS_ICON], L"Missions", false, true));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_COMM, tmp_rect, gen_pictures[GENPIC_COMM_ICON], L"Repeat last comm message", false, true));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_BRIEFING, tmp_rect, gen_pictures[GENPIC_BRIEFING_ICON], L"Mission briefing", false, true));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_RANGE, tmp_rect, gen_pictures[GENPIC_RANGE_ICON], L"Display weapon ranges", false, true));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_OPTIONS, tmp_rect, gen_pictures[GENPIC_OPTIONS_ICON], L"Options", false, true));

	tmp_rect.x = global_settings.screen_width - info_window_width - small_border_size * 2 - ControlIcon::icon_dim;
	tmp_rect.y = global_settings.screen_height - info_window_height + ControlIcon::icon_spacing;	

	my_windows.push_back(GenWindow(RTS_SQUAD_STATS, tmp_rect, gen_pictures[GENPIC_STATS_ICON], L"Ship information", true, false));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_SQUAD_STATUS, tmp_rect, gen_pictures[GENPIC_STATUS_ICON], L"Ship status report", true, false));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_RETURN, tmp_rect, gen_pictures[GENPIC_RETURN_ICON], L"Return home", true, false));
	tmp_rect.y += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_RECALL, tmp_rect, gen_pictures[GENPIC_RECALL_ICON], L"Defend parent capital ship", true, false));

	tmp_rect.x =  global_settings.screen_width / 2 - (((ControlIcon::icon_dim + ControlIcon::icon_spacing) * 5) / 2);
	tmp_rect.y = global_settings.screen_height - (info_window_height / 2);
	int text_x = tmp_rect.x;

	my_windows.push_back(GenWindow(RTS_RECON, tmp_rect, gen_pictures[GENPIC_RECON_ICON], L"Launch reconnaissance mission", text_x, false));
	tmp_rect.x += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_PATROL, tmp_rect, gen_pictures[GENPIC_PATROL_ICON], L"Launch patrol mission", text_x, false));
	tmp_rect.x += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_ATTACK, tmp_rect, gen_pictures[GENPIC_ATTACK_ICON], L"Launch attack mission", text_x, false));
	tmp_rect.x += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_SELECT, tmp_rect, gen_pictures[GENPIC_SELECT_ICON], L"Select these ships", text_x, false));
	tmp_rect.x += ControlIcon::icon_dim + ControlIcon::icon_spacing;
	my_windows.push_back(GenWindow(RTS_CANCEL, tmp_rect, gen_pictures[GENPIC_CANCEL_ICON], L"Cancel", text_x, false));

	PortraitText::init();

	radar_rect.x = bottom_left_box_rect.x + small_border_size;
	radar_rect.y = bottom_left_box_rect.y + small_border_size;
	radar_rect.w = bottom_left_box_rect.w - (small_border_size * 2);
	radar_rect.h = bottom_left_box_rect.h - (small_border_size * 2);

	radar_dragging = false;

	//need to reload images if an exception is thrown and we never run the destructor
	//must init world after PortraitText for PortraitText::init not to break dialog
	try {
		world.init();
	} catch (runtime_error e) {
		load_title_images();
		throw e;
	}
}

RTS_State::~RTS_State() {
	world.shutdown();

	kill_all_windows();
	PortraitText::shutdown();

	current_cursor_type = GENPIC_CURSOR;

	sound.end_sound();
	sound.end_music();

	debug_display_all_groups = false;

	if (gs_to != GST_THE_OS) {
		load_title_images();
		load_title_music();
		if (global_settings.music_volume >= 90)
			sound.set_music_volume(128);
		sound.start_music();
	}
}

void RTS_State::main() {
	if (world.update_squad_selection())
		change_info_window();

	world.run();
	PortraitText::run();
	
	if (skip_display_frame)
		return;
		
	world.draw();
	draw_radar();
	draw_all_windows();
}

void RTS_State::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	//radar before groups
	if (point_in_rect(x, y, radar_rect)) {
		if (button == SDL_BUTTON_LEFT) {
			radar_dragging = true;
			mouse_m(0, x, y);
		}
	} else
		world.mouse_d(button, x, y);	
}	

void RTS_State::mouse_u(Uint8 button, Uint16 x, Uint16 y) {
	if (button == SDL_BUTTON_LEFT)
		radar_dragging = false;

	world.mouse_u(button, x, y);
}

void RTS_State::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	//radar before groups
	if (point_in_rect(x, y, radar_rect)) {
		mouse_over_ui = true;
		current_cursor_type = GENPIC_CURSOR;

		if (radar_dragging) {
			int temp_center_x = (x - radar_rect.x) * world.width / radar_rect.w;
			int temp_center_y = (y - radar_rect.y) * world.height / radar_rect.h;

			world.viewx = temp_center_x - (global_settings.screen_width / 2);
			world.viewy = temp_center_y - (global_settings.screen_height / 2);

			world.view_side = -1;
		}
	} else
		world.mouse_m(state, x, y);
}

void RTS_State::keyboard(SDL_keysym& keysym) {
	switch(keysym.sym) {
	case SDLK_RETURN:
		PortraitText::end_current_comm(true);
		break;

	case SDLK_COMMA:
		sides[0].selected_squads.clear();
		for (int i = 0; i != sides[0].squadrons.size(); ++i) {
			if (sides[0].squadrons[i].is_big_ship() && sides[0].squadrons[i].get_alive())
				sides[0].select_squad(0, i);
		}
		break;

	case SDLK_PERIOD:
		sides[0].selected_squads.clear();
		for (int i = 0; i != sides[0].squadrons.size(); ++i) {
			if (sides[0].squadrons[i].get_type() == UT_FRIGATE && sides[0].squadrons[i].get_alive())
				sides[0].select_squad(0, i);
		}
		break;

	case SDLK_SLASH:
		sides[0].selected_squads.clear();
		for (int i = 0; i != sides[0].squadrons.size(); ++i) {
			if (sides[0].squadrons[i].get_type() == UT_CAPITAL && sides[0].squadrons[i].get_alive())
				sides[0].select_squad(0, i);
		}
		break;

	case SDLK_SPACE: {
		if (!global_settings.debug)
			break;

		static bool speed_slider = false;
		static int slider_win_id = 0;		
		if (speed_slider) {
			try {
				kill_window(slider_win_id);
				speed_slider = false;
				break;
			} catch (runtime_error e) {
			}
		}
		//if we fell through due to a runtime error because window was already closed, create a new one
		my_windows.push_back(GenWindow(bottom_left_box_rect.x, bottom_left_box_rect.y + bottom_left_box_rect.h, RTS_GAME_SPEED, 0, 0, 0));
		slider_win_id = window_ids;
		speed_slider = true;
		}
		break;

	case SDLK_F7:
		if (global_settings.debug) {
			debug_display_all_groups = !debug_display_all_groups;
		}
		break;

	case SDLK_F8:
		if (global_settings.debug || gs_current == GST_MISSION_EDITOR) {
			message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);
			if (sides[0].selected_squads.size()) {
				CoordsInt& first_squad = *(sides[0].selected_squads.begin());
				my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_VARS, first_squad.x, first_squad.y, 0));
			} else
				my_windows.push_back(GenWindow(0, 0, RTS_DEBUG_WIN, 0, 0, 0));
		}
		break;

	case SDLK_F11:
		if (global_settings.debug)
			display.take_screenshot();
		break;

	default:
		world.keyboard(keysym);
		break;
	}
}

void RTS_State::draw_radar() {
	display.blt_fill(bottom_left_box_rect, standard_colors.light_blue, 0.5);
	display.blt_fill(radar_rect, standard_colors.black, 0.5);
	
	glLoadIdentity();
	glTranslatef(radar_rect.x, radar_rect.y, 0);
	glPointSize(4);
	glBegin(GL_POINTS);

	for (int i  = 0; i != sides.size(); ++i) {
		glColor3f(sides[i].radar_color.r, sides[i].radar_color.g, sides[i].radar_color.b);
		for (int j = 0; j != sides[i].groups.size(); ++j) {
			if (sides[i].groups[j].get_alive()
			&& (sides[0].scanned_groups[i][j] || sides[i].groups[j].get_type() == UT_PLANET || debug_display_all_groups)
			&& !sides[i].groups[j].get_in_hangar()) {
				CoordsFloat center = sides[i].groups[j].get_center();

				int radarx = static_cast<int>(center.x) * radar_rect.w / world.width;
				int radary = static_cast<int>(center.y) * radar_rect.h / world.height;
				
				if (radarx >= 0 && radarx < radar_rect.w && radary >= 0 && radary < radar_rect.h)
					glVertex2f(static_cast<float>(radarx), static_cast<float>(radary));
			}
		}
	}
	glEnd();

	//outline current view area on radar
	SDL_Rect view_area;
	view_area.x = (world.viewx * radar_rect.w / world.width) + radar_rect.x;
	view_area.y = (world.viewy * radar_rect.h / world.height) + radar_rect.y;
	view_area.w = radar_rect.w * global_settings.screen_width / world.width;
	view_area.h = radar_rect.h * global_settings.screen_height / world.height;
	
	draw_rect_border(view_area, standard_colors.light_blue, 1);
}

void RTS_State::change_info_window() {
	message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);

	switch (sides[0].selected_squads.size()) {
	case 0:
		break;

	case 1: {
		CoordsInt& first_squad = *(sides[0].selected_squads.begin());
		my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_STATS, first_squad.x, first_squad.y, 0));
		}
		break;

	default: {
		CoordsInt& first_squad = *(sides[0].selected_squads.begin());
		my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_LIST, first_squad.x, window_ids, 0));
		}
		break;
	}
}

void RTS_State::play_alert(SoundEffect which_sound) {
	sound.play_sound(which_sound);

	if (!global_settings.disable_sound)
		world.set_update_interval(standard_interval);
}

////

void PortraitText::init() {	
	next_msg = messages.end();
}

void PortraitText::shutdown() {
	for (map<wstring, Portrait>::iterator iter = portraits.begin(); iter != portraits.end(); ++iter)
		glSDL_FreeSurface(iter->second.portrait);
	portraits.clear();

	messages.clear();
}

PortraitText::PortraitText():
WrappedString(global_settings.screen_width / 2 - 500 / 2, 0, 400, 200,
next_msg->title + L"\n\n" + next_msg->msg, true, WFLAG_STATIC | WFLAG_CANTCLOSE | WFLAG_TRANSPARENT), current_msg(*next_msg) {
	if (current_msg.align_bottom)
		rect.y = global_settings.screen_height;
	else
		rect.y = radar_indent;

	init_rects();

	Portrait& current_portrait = portraits[current_msg.title];
	portrait_rect.x = rect.x + 400 + big_border_size;
	portrait_rect.y = rect.y;
	portrait_rect.w = current_portrait.portrait->w;
	portrait_rect.h = current_portrait.portrait->h;
	
	if (!world.paused_by_script)
		world.set_update_interval(standard_interval);

	int num_words = 1;
	for (int i = 0; i != current_msg.msg.size(); ++i) {
		if (current_msg.msg[i] == ' ')
			++num_words;
	}

	end_time = now + (default_portrait_duration_per_word * num_words);

	//argument is optional in squirrel
	/*if (current_msg.sound_file != "") {
		sound.play_script_sound(current_msg.sound_file);
		wait_for_sound = true;
	} else*/
		wait_for_sound = false;

	currently_open = true;
}

PortraitText::~PortraitText() {
	if (next_msg == messages.end())
		world.script_manager.fire_wakeup_event(WET_PORTRAIT_FINISHED);
	currently_open = false;
}

void PortraitText::add_portrait(const std::wstring& title, const std::wstring& filename, int side) {
	string full_name = world.mission_folder + "images/" + wstring_to_string(filename);
	if (!does_file_exist(full_name))
		full_name = "images/" + wstring_to_string(filename);
	SDL_Surface* tmp = display.file_to_surface(full_name);

	portraits[title] = Portrait(side, tmp);
}

void PortraitText::set_portrait(const wstring& title, bool align_bottom) {
	if (portraits.find(title) == portraits.end()) {
		wstring error = L"Error: Portrait \"" + title + L"\" not loaded";
		write_log(error);
		throw runtime_error(wstring_to_string(error));
	}

	new_title = title;
	new_align_bottom = align_bottom;
}

void PortraitText::play_comm(const std::wstring& msg, const std::wstring& sound_file, bool i_unskippable) {	
	unskippable = i_unskippable;
	messages.push_back(CommMessage(new_title, msg, sound_file, new_align_bottom));
	if (messages.size() > 2)
		messages.erase(messages.begin());

	if (next_msg == messages.end())
		--next_msg;
}

void PortraitText::replay_last_comm() {
	if (messages.size() == 0)
		return;

	int end;
	if (currently_open)
		end = 2;
	else
		end = 1;

	for (int i = 0; i != end; ++i) {
		if (next_msg != messages.begin())
			--next_msg;
	}
	
	message_windows(WC_CLOSE_PORTRAIT_TEXT, 0, 0, window_id_all, window_id_none);
}

void PortraitText::restart_current_comm() {
	if (!currently_open)
		return;

	if (messages.size() != 0 && next_msg != messages.begin())
		--next_msg;
	message_windows(WC_CLOSE_PORTRAIT_TEXT, 0, 0, window_id_all, window_id_none);
}

void PortraitText::end_current_comm(bool skip) {
	if (skip && unskippable && next_msg == messages.end())
		return;

	sound.stop_script_sound();
	message_windows(WC_CLOSE_PORTRAIT_TEXT, 0, 0, window_id_all, window_id_none);
}

void PortraitText::run()  {
	if (!currently_open && next_msg != messages.end()) {
		my_windows.push_back(GenWindow(0, 0, RTS_PORTRAIT_TEXT, 0, 0, 0));
		++next_msg;
	}
}

void PortraitText::update()  {
	if (!world.paused_by_script && world.paused)
		return;

	if (global_settings.disable_sound || !wait_for_sound) {
		if (now > end_time)
			closed = true;
	} else if (!sound.script_sound_playing())
		closed = true;
}

void PortraitText::draw_self() {
	WrappedString::draw_self();
	Portrait& current_portrait = portraits[current_msg.title];
	display.blt(current_portrait.portrait, portrait_rect);
	draw_rect_border(portrait_rect, sides[current_portrait.side].color, 1);
}

void PortraitText::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (the_msg == WC_CLOSE_PORTRAIT_TEXT)
		closed = true;
}

///

ControlIcon::ControlIcon(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const wstring& i_desc, bool minus_width, bool i_active):
GenWindow_Base(0, 0, window_id_none, 0) {
	init_vars(i_type, i_rect, i_pic, i_desc, i_active);
	if (minus_width)
		text_offset_x = rect.x - bold_font.get_width(desc) - icon_spacing;
	else
		text_offset_x = rect.x + + rect.w + icon_spacing;

	text_offset_y = rect.y + rect.h / 2 - normal_font.get_height() / 2;
}

ControlIcon::ControlIcon(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const wstring& i_desc, int i_text_offset_x, bool i_active):
GenWindow_Base(0, 0, window_id_none, 0) {
	init_vars(i_type, i_rect, i_pic, i_desc, i_active);
	text_offset_x = i_text_offset_x;
	text_offset_y = rect.y + rect.h * 2;
}

void ControlIcon::init_vars(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const wstring& i_desc, bool i_active) {
	type = i_type;
	pic = i_pic;
	desc = i_desc;
	active = i_active;

	rect = i_rect;
	hover = false;
}

//cannot use mouse_m because if user moves mouse quickly we might never receive the mouse_m message that
//tells us to deactivate (only the top most window gets input messages)
//plus we need to update to be false when icon is no longer active due to change in current active icons
void ControlIcon::update() {
	if (!active)
		return;

	int x, y;
	SDL_GetMouseState(&x, &y);

	if (point_in_rect(x, y, rect))
		hover = true;
	else
		hover = false;
}

//currently this code is largely copied in keyboard() for keyboard shortcuts so any edits here must be copied across
bool ControlIcon::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	if (!active)
		return false;

	bool ret = GenWindow_Base::mouse_d(button, x, y);

	if (ret && button == SDL_BUTTON_LEFT) {
		switch(type) {
		case RTS_MISSIONS:
			my_windows.push_back(GenWindow(0, 0, RTS_MISSIONS, 0, 0, 0));
			break;

		case RTS_COMM:
			PortraitText::replay_last_comm();
			break;

		case RTS_BRIEFING:
			my_windows.push_back(GenWindow(0, 0, RTS_BRIEFING, 0, 0, 0));
			break;

		case RTS_RANGE:
			for (int i = 0; i != sides.size(); ++i) {
				for (int j = 0; j != sides[i].groups.size(); ++j)
					sides[i].groups[j].toggle_draw_weapon_range();
			}
			break;

		case RTS_OPTIONS:
			my_windows.push_back(GenWindow(0, 0, RTS_OPTIONS, 0, 0, 0));
			break;

		case RTS_SQUAD_STATS: {
			message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);
			CoordsInt& first_squad = *(sides[0].selected_squads.begin());
			my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_STATS, first_squad.x, first_squad.y, 0));
			}
			break;

		case RTS_SQUAD_STATUS: {
			message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);
			CoordsInt& first_squad = *(sides[0].selected_squads.begin());
			my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_STATUS, first_squad.x, first_squad.y, 0));
			}
			break;

		case RTS_RETURN:
			sides[0].order_return();
			break;

		case RTS_RECALL:
			sides[0].order_recall();
			break;

		case RTS_PATROL:
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_PATROL, 0, 0, 0));
			break;
		case RTS_ATTACK:
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_ATTACK, 0, 0, 0));
			break;
		case RTS_RECON:
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_RECON, 0, 0, 0));
			break;
		case RTS_SELECT:
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			sides[0].select_highlighted_squadrons();
			sides[0].highlighted_squads.clear();
			break;
		case RTS_CANCEL:
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			sides[0].highlighted_squads.clear();
			break;
		default:
			break;
		}

		sound.play_sound(SE_MENU_CLICK);
	}

	return ret;
}

bool ControlIcon::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	if (!active)
		return false;

	return GenWindow_Base::mouse_m(state, x, y);
}

bool ControlIcon::keyboard(SDL_keysym& keysym) {
	if (!active)
		return false;

	bool ret = false;

	switch(type) {
	case RTS_OPTIONS:
		if (keysym.sym == SDLK_ESCAPE) {
			if (global_settings.debug)
				gs_to = GST_THE_OS;
			else
				my_windows.push_back(GenWindow(0, 0, RTS_OPTIONS, 0, 0, 0));

			ret = true;
		}
		break;

	case RTS_MISSIONS:
		if (keysym.sym == SDLK_w) {
			my_windows.push_back(GenWindow(0, 0, RTS_MISSIONS, 0, 0, 0));
			ret = true;
		}
		break;

	case RTS_COMM:
		if (keysym.sym == SDLK_q) {
			PortraitText::replay_last_comm();
			ret = true;
		}
		break;

	case RTS_BRIEFING:
		if (keysym.sym == SDLK_e) {
			my_windows.push_back(GenWindow(0, 0, RTS_BRIEFING, 0, 0, 0));
			ret = true;
		}
		break;

	case RTS_RANGE:
		if (keysym.sym == SDLK_r) {
			for (int i = 0; i != sides.size(); ++i) {
				for (int j = 0; j != sides[i].groups.size(); ++j)
					sides[i].groups[j].toggle_draw_weapon_range();
			}
			ret = true;
		}
		break;

	case RTS_SQUAD_STATS:
		if (keysym.sym == SDLK_z) {
			message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);
			CoordsInt& first_squad = *(sides[0].selected_squads.begin());
			my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_STATS, first_squad.x, first_squad.y, 0));
			ret = true;
		}
		break;

	case RTS_SQUAD_STATUS:
		if (keysym.sym == SDLK_x) {
			message_windows(WC_CLOSE_GROUP_WIN, 0, 0, window_id_all, window_id_none);
			CoordsInt& first_squad = *(sides[0].selected_squads.begin());
			my_windows.push_back(GenWindow(0, 0, RTS_SQUAD_STATUS, first_squad.x, first_squad.y, 0));
			ret = true;
		}
		break;

	case RTS_RETURN:
		if (keysym.sym == SDLK_c) {
			sides[0].order_return();
			ret = true;
		}
		break;

	case RTS_RECALL:
		if (keysym.sym == SDLK_v) {
			sides[0].order_recall();
			ret = true;
		}
		break;

	case RTS_RECON:
		if (keysym.sym == SDLK_a) {
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_RECON, 0, 0, 0));
			ret = true;
		}
		break;

	case RTS_PATROL:
		if (keysym.sym == SDLK_s) {
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_PATROL, 0, 0, 0));
			ret = true;
		}
		break;
	case RTS_ATTACK:
		if (keysym.sym == SDLK_d) {
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			message_windows(WC_TARGET_WIN_OPEN, 0, 0, window_id_all, window_id_none);
			my_windows.push_back(GenWindow(0, 0, RTS_TARGET_ATTACK, 0, 0, 0));
			ret = true;
		}
		break;
	case RTS_SELECT:
		if (keysym.sym == SDLK_f) {
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			sides[0].select_highlighted_squadrons();
			sides[0].highlighted_squads.clear();
			ret = true;
		}
		break;
	case RTS_CANCEL:
		if (keysym.sym == SDLK_ESCAPE || keysym.sym == SDLK_w) {
			message_windows(WC_CLOSE_MISSION_WIN, 0, 0, window_id_all, window_id_none);
			sides[0].highlighted_squads.clear();
			ret = true;
		}
		break;
	default:
		break;
	}

	if (ret)
		sound.play_sound(SE_MENU_CLICK);

	return ret;
}

void ControlIcon::draw_self() {
	if (!active)
		return;

	display.blt(pic, rect);

	if (hover)
		bold_font.render(text_offset_x, text_offset_y, desc);
}

void ControlIcon::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	switch(the_msg) {
	case WC_OPTIONS_WIN_OPEN:	
		if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = false;
		break;

	case WC_MISSION_WIN_OPEN:
		if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = false;
		else if (type == RTS_CANCEL)
			active = true;
		break;

	case WC_SQUAD_LIST_WIN_SS:
		if (type == RTS_RETURN || type == RTS_RECALL)
			active = true;
		break;

	case WC_SQUAD_LIST_WIN_NO_SS:
		if (type == RTS_RETURN || type == RTS_RECALL)
			active = false;
		break;

	case WC_BS_WIN_OPEN:
		if (type == RTS_SQUAD_STATS || type == RTS_SQUAD_STATUS)
			active = true;
		break;

	case WC_SS_WIN_OPEN:
		if (type == RTS_SQUAD_STATS
		|| type == RTS_SQUAD_STATUS
		|| type == RTS_RETURN
		|| type == RTS_RECALL)
			active = true;
		break;

	case WC_TARGET_WIN_OPEN:
		if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = false;
		break;

	case WC_OPTIONS_WIN_CLOSED:
		if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = true;
		break;

	case WC_CLOSE_MISSION_WIN:
		if (type == RTS_PATROL
		|| type == RTS_ATTACK
		|| type == RTS_RECON
		|| type == RTS_SELECT
		|| type == RTS_CANCEL)
			active = false;
		else if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = true;
		break;

	case WC_CLOSE_GROUP_WIN:
		if (type == RTS_SQUAD_STATS
		|| type == RTS_SQUAD_STATUS
		|| type == RTS_RETURN
		|| type == RTS_RECALL)
			active = false;
		break;

	case WC_TARGET_WIN_CLOSED:
		if (type == RTS_MISSIONS || type == RTS_COMM || type == RTS_BRIEFING || type == RTS_RANGE || type == RTS_OPTIONS)
			active = true;

	case WC_NO_SQUAD_HIGHLIGHT:
		if (type == RTS_PATROL
		|| type == RTS_ATTACK
		|| type == RTS_RECON
		|| type == RTS_SELECT)
			active = false;
		break;

	case WC_SQUAD_HOME_HIGHLIGHT:
		if (type == RTS_PATROL
		|| type == RTS_ATTACK
		|| type == RTS_RECON
		|| type == RTS_SELECT)
			active = true;
		break;

	case WC_SQUAD_AWAY_HIGHLIGHT:
		if (type == RTS_SELECT)
			active = true;
		break;
	}
}

////

Missions::Missions():
Menu_Base(0, 0, window_id_none, WFLAG_TRANSPARENT) {
	message_windows(WC_MISSION_WIN_OPEN, 0, 0, window_id_all, window_id_none);

	b_hover_highlight = false;
	rect = viewable_screen_rect;
	if (global_settings.screen_width > 800)
		rect.h = global_settings.screen_height - RTS_State::bottom_left_box_rect.h - radar_indent;
	else
		rect.h = global_settings.screen_height - (info_window_height / 2) - ControlIcon::icon_dim;

	group_item_spacing = normal_font.get_height() * 3;
	group_item_width = 240;

	sides[0].selected_squads.clear();

	world.script_manager.fire_wakeup_event(WET_MISSIONS_OPEN);

	update();
}

void Missions::update() {
	clear_items();

	MenuItem temp_item;
	new_item_x = group_item_spacing;
	new_item_y = big_border_size;
	temp_item.rect.w = group_item_width;
	temp_item.rect.h = group_item_spacing;

	temp_item.desc = L"Missions";
	temp_item.choice = WC_NOTHING;
	temp_item.flags = MFLAG_BOLD;
	add_item(temp_item);

	temp_item.choice = RTS_GROUP;

	UnitType last_type = UT_CAPITAL;

	for (int i = 0; i != sides[0].squadrons.size(); ++i) {
		if (!sides[0].squadrons[i].get_alive())
			continue;

		UnitType type = sides[0].squadrons[i].get_type();
		wstring type_name = unit_type_to_string[type];
		
		switch (type) {
		case UT_BOMBER:
			//pad bomber name to fighter name length
			type_name += L" ";
			//fall through
		case UT_FIGHTER: {
			type_name += L" ";
			if (sides[0].squadrons[i].both_in_hangar()) {
				wchar_t output[48];

				int fuel_per = sides[0].squadrons[i].get_fuel_percent();

				if (type == UT_BOMBER) {
					int ammo_per = sides[0].squadrons[i].get_ammo_percent();					
					swprintf(output, 48, L"Fuel: %d%% Ammo %d%%", fuel_per, ammo_per);
				} else
					swprintf(output, 48, L"Fuel: %d%%", fuel_per);

				type_name += output;
			} else {
				type_name += L"- " + sides[0].squadrons[i].get_current_mission();
			}
			}
			break;
		case UT_FRIGATE:
		case UT_FREIGHTER:
		case UT_DEFENCE_NODE:
		case UT_CAPITAL:
		case UT_PLANET:
			if (i != 0 && last_type != UT_FRIGATE && last_type != UT_FREIGHTER && last_type != UT_DEFENCE_NODE)
				new_column();
			break;
		}
			
		wstring name = sides[0].squadrons[i].get_unit_name();
		temp_item.desc = type_name + L'\n' + name;
		temp_item.parem = i;

		if (find(sides[0].highlighted_squads.begin(), sides[0].highlighted_squads.end(), i) != sides[0].highlighted_squads.end())
			temp_item.flags = MFLAG_HIGHLIGHT;
		else
			temp_item.flags = 0;

		add_item(temp_item);

		last_type = type;
	}
}

void Missions::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR)
		sides[0].toggle_highlight_squadron(current_selection.parem);
}

void Missions::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (the_msg == WC_CLOSE_MISSION_WIN)
		closed = true;
}

void Missions::draw_self() {
	//start at 1, don't include title
	for (int i = 1; i != items.size(); ++i) {
		if (sides[0].squadrons[items[i].parem].get_type() != UT_PLANET) {
			SDL_Rect health_bar;
			if (sides[0].squadrons[items[i].parem].get_shield_max() > 0) {
				health_bar.x = items[i].rect.x + small_border_size;
				health_bar.y = items[i].rect.y + normal_font.get_height() * 2;
				health_bar.w = sides[0].squadrons[items[i].parem].get_shield() * items[i].rect.w * 3 / 4 / sides[0].squadrons[items[i].parem].get_shield_max();
				health_bar.h = health_bar_height;

				display.blt_fill(health_bar, standard_colors.shield_green);
			}

			//SDL clips draw rects, though in that mission menu isn't clipped it doesn't really matter here    
			health_bar.x = items[i].rect.x + small_border_size;
			health_bar.y = health_bar_height + items[i].rect.y + normal_font.get_height() * 2;
			health_bar.w = sides[0].squadrons[items[i].parem].get_armour() * items[i].rect.w *3 / 4 / sides[0].squadrons[items[i].parem].get_armour_max();
			health_bar.h = health_bar_height;

			display.blt_fill(health_bar, standard_colors.armour_blue);
		}
	}

	Menu_Base::draw_self();
}

void Missions::new_column() {
	new_item_y = big_border_size + group_item_spacing;
	new_item_x += group_item_width;
}

TargetInput::TargetInput(GenPicture i_cursor_type):
GenWindow_Base(0, 0, window_id_none, WFLAG_INVISIBLE), cursor_type(i_cursor_type) {
	rect = viewable_screen_rect;
	current_cursor_type = cursor_type;
	sides[0].display_missions_fuel();

	//list of the parent capital ships of the highlighted squadrons
	for (list<int>::iterator iter = sides[0].highlighted_squads.begin(); iter != sides[0].highlighted_squads.end(); ++iter) {
		int parent = sides[0].squadrons[*iter].get_parent_cap();
		if (find(parent_list.begin(), parent_list.end(), parent) == parent_list.end())
			parent_list.push_back(parent);
	}
}

TargetInput::~TargetInput() {
	message_windows(WC_TARGET_WIN_CLOSED, 0, 0, window_id_all, window_id_none);
	current_cursor_type = GENPIC_CURSOR;
	sides[0].hide_missions_fuel();
	sides[0].highlighted_squads.clear();
}

bool TargetInput::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	bool ret = GenWindow_Base::mouse_m(state, x, y);

	current_cursor_type = cursor_type;

	for (vector<int>::iterator iter = parent_list.begin(); iter != parent_list.end(); ++iter) {
		if (sides[0].groups[*iter].find_point_distance_from_center(static_cast<float>(x + world.viewx), static_cast<float>(y + world.viewy)) > (equip_lookup[L"Fuel"].max - fuel_leeway) / 2) {
			current_cursor_type = GENPIC_CURSOR;
			break;
		}
	}
	
	return ret;
}

bool TargetInput::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	if (point_in_rect(x, y, RTS_State::radar_rect))
		return false;

	GenWindow_Base::mouse_d(button, x, y);

	if (button == SDL_BUTTON_LEFT)
		closed = true;
	else if (button == SDL_BUTTON_RIGHT && current_cursor_type != GENPIC_CURSOR)
		take_action(x, y);

	return true;
}

bool TargetInput::keyboard(SDL_keysym& keysym) {
	if (keysym.sym == SDLK_ESCAPE) {
		closed = true;
		return true;
	}

	return false;
}

TargetMove::TargetMove(bool patrol):
TargetInput(GENPIC_CURSORMOVE), b_patrol(patrol), got_waypoint(false) {
	if (!b_patrol)
		world.script_manager.fire_wakeup_event(WET_RECON_TARGET_OPEN);
}

void TargetMove::take_action(Uint16 x, Uint16 y) {
	for (list<int>::iterator iter = sides[0].highlighted_squads.begin(); iter != sides[0].highlighted_squads.end(); ++iter)
		sides[0].squadrons[*iter].add_waypoint(CoordsFloat(static_cast<float>(x + world.viewx), static_cast<float>(y + world.viewy)));

	got_waypoint = true;
	if (! (SDL_GetModState() & KMOD_SHIFT)) {
		closed = true;
		if (b_patrol)
			sides[0].launch_mission(L"patrol");
		else
			sides[0].launch_mission(L"recon");
	}
}

bool TargetMove::key_u(SDL_keysym& keysym) {
	if (keysym.sym == SDLK_LSHIFT || keysym.sym == SDLK_RSHIFT) {
		closed = true;
		
		if (got_waypoint) {
			if (b_patrol)
				sides[0].launch_mission(L"patrol");
			else
				sides[0].launch_mission(L"recon");
		}

		return true;
	}

	return false;
}

void TargetMove::draw_self() {
	wstring str;
	if (b_patrol)
		str = L"Hold down the shift key and right click to place patrol waypoints.";
	else
		str = L"Hold down the shift key and right click to place reconnaissance waypoints.";

	bold_font.render(big_border_size, big_border_size, str);
	bold_font.render(big_border_size, big_border_size + bold_font.get_height(), L"Release shift key to launch mission. Left click to cancel.");
}

TargetAttack::TargetAttack():
TargetInput(GENPIC_CURSORATTACK) {}

void TargetAttack::take_action(Uint16 x, Uint16 y) {	
	for (list<int>::iterator iter = sides[0].highlighted_squads.begin(); iter != sides[0].highlighted_squads.end(); ++iter)
		sides[0].squadrons[*iter].add_waypoint(CoordsFloat(static_cast<float>(x + world.viewx), static_cast<float>(y + world.viewy)));
	sides[0].launch_mission(L"attack");
	closed = true;
}

void TargetAttack::draw_self() {
	wstring str = L"Right click the target area to attack. Left click to cancel.";	
	bold_font.render(big_border_size, big_border_size, str);
}

SquadsWindow::SquadsWindow(int ix, int iy, int i_my_side, int i_my_squad):
DragWindow(ix, iy, window_id_none, WFLAG_STATIC | WFLAG_CANTCLOSE | WFLAG_TILED), my_side(i_my_side), my_squad(i_my_squad) {
	border_color = sides[my_side].color;
	rect.w = info_window_width;
	rect.h = info_window_height;

	init_rects();
}

bool SquadsWindow::mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	bool ret = DragWindow::mouse_d(button, x, y);

	if (ret && button == SDL_BUTTON_LEFT) {
		world.view_side = my_side;
		world.view_squad = my_squad;
	}
		
	return ret;
}

void SquadsWindow::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (the_msg == WC_CLOSE_GROUP_WIN)
		closed = true;
}

void SquadsWindow::add_side_title() {
	wstring str = sides[my_side].name + L":";	
	the_text.push_back(WindowText(str, true));
	the_text.push_back(WindowText(L""));
}

SquadList::SquadList(int ix, int iy, int i_my_side, int parent, int flags):
Menu_Base(ix, iy, parent, flags | WFLAG_STATIC | WFLAG_CANTCLOSE | WFLAG_TILED), my_side(i_my_side) {
	border_color = sides[my_side].color;
	rect.w = info_window_width;
	rect.h = info_window_height;
	tile();

	add_item(L"Currently selected squadrons:", WC_NOTHING);
	add_blank_item();
	
	bool make_command_icons = true;	
	for (list<CoordsInt>::iterator iter = sides[0].selected_squads.begin(); iter != sides[0].selected_squads.end(); ++iter) {
		if (!sides[my_side].squadrons[iter->y].is_small())
			make_command_icons = false;
		wstring desc = sides[my_side].squadrons[iter->y].get_unit_name();
		add_item(desc, RTS_SQUAD_STATS, iter->y);
	}

	if (make_command_icons)
		message_windows(WC_SQUAD_LIST_WIN_SS, 0, 0, window_id_all, window_id_none);
	else
		message_windows(WC_SQUAD_LIST_WIN_NO_SS, 0, 0, window_id_all, window_id_none);		

	init_rects();
}

void SquadList::init_rects() {
	for (int i = 0; i != items.size(); ++i)
		items[i].rect.w = rect.w;
	init_border();
}

void SquadList::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR && current_selection.choice == RTS_SQUAD_STATS) {
		sides[0].selected_squads.clear();
		sides[0].selected_squads.push_back(CoordsInt(my_side, current_selection.parem));
	}
}

void SquadList::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (the_msg == WC_CLOSE_GROUP_WIN)
		closed = true;
}

SquadStatsInfo::SquadStatsInfo(int ix, int iy, int i_my_side, int i_my_squad):
SquadsWindow(ix, iy, i_my_side, i_my_squad) {
	if (my_side == 0 && sides[my_side].squadrons[my_squad].is_small())
		message_windows(WC_SS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
	else
		message_windows(WC_BS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
}

void SquadStatsInfo::draw_self() {
	the_text.clear();

	wchar_t output[48];

	add_side_title();

	if (sides[my_side].squadrons[my_squad].get_type() == UT_PLANET) {
		draw_planet_info();
		DragWindow::draw_self();
		return;
	}

	wstring output_str = sides[my_side].squadrons[my_squad].get_unit_name();
	if (sides[my_side].squadrons[my_squad].is_small())
		output_str += L" squadron";

	the_text.push_back(WindowText(output_str));

	int number_left = sides[my_side].squadrons[my_squad].get_units_left();
	wstring asterisks;
	if (number_left)
		asterisks.insert(asterisks.end(), number_left, '*');
	else
		asterisks = L"Destroyed";

	int health = sides[my_side].squadrons[my_squad].get_health();
	if (health > 0) {
		swprintf(output, 48, L"Total health: %d", health);
		the_text.push_back(WindowText(output));
	}

	output_str = L"Engine: " + sides[my_side].squadrons[my_squad].get_engine_name();
	the_text.push_back(WindowText(output_str));

	output_str = L"Shield: " + sides[my_side].squadrons[my_squad].get_shield_name();
	the_text.push_back(WindowText(output_str));

	output_str = L"Armour: " + sides[my_side].squadrons[my_squad].get_armour_name();
	the_text.push_back(WindowText(output_str));

	if (!sides[my_side].squadrons[my_squad].is_small()) {
		swprintf(output, 48, L"Small weapon: %ls * %d", weapon_lookup[sides[my_side].squadrons[my_squad].get_small_type()].name.c_str(), sides[my_side].squadrons[my_squad].get_small_number());
		the_text.push_back(WindowText(output));
	} else {
		output_str = L"Small weapon: " + weapon_lookup[sides[my_side].squadrons[my_squad].get_small_type()].name;
		the_text.push_back(WindowText(output_str));
	}

	if (sides[my_side].squadrons[my_squad].is_small()) {
		swprintf(output, 48, L"Big weapon: %ls * %d", weapon_lookup[sides[my_side].squadrons[my_squad].get_big_type()].name.c_str(), sides[my_side].squadrons[my_squad].get_big_ammo());
		the_text.push_back(WindowText(output));
	} else {
		output_str = L"Big weapon: " + weapon_lookup[sides[my_side].squadrons[my_squad].get_big_type()].name;
		the_text.push_back(WindowText(output_str));
	}

	DragWindow::draw_self();
}

void SquadStatsInfo::draw_planet_info() {
	the_text.push_back(sides[my_side].squadrons[my_squad].get_blurb());
}

SquadStatusReport::SquadStatusReport(int ix, int iy, int i_my_side, int i_my_squad):
SquadsWindow(ix, iy, i_my_side, i_my_squad) {
	if (my_side == 0 && sides[my_side].squadrons[my_squad].is_small())
		message_windows(WC_SS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
	else
		message_windows(WC_BS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
}

void SquadStatusReport::draw_self() {
	the_text.clear();
	const AICommands* p_commands = sides[my_side].squadrons[my_squad].get_the_commands();
	wchar_t output[48];

	add_side_title();

	if (sides[my_side].squadrons[my_squad].is_small()) {
		wstring tmp_str = L"Current mission: " + sides[my_side].squadrons[my_squad].get_current_mission();
		the_text.push_back(WindowText(tmp_str));
		swprintf(output, 48, L"Remaining fuel: %d", sides[my_side].squadrons[my_squad].get_fuel());
		the_text.push_back(WindowText(output));
		the_text.push_back(WindowText(L""));
	}

	the_text.push_back(WindowText(move_com_to_string(*p_commands)));

	if (p_commands->move_command != MC_NO_MOVE) {
		swprintf(output, 48, L"Distance: %d", p_commands->move_target_dist);
		the_text.push_back(WindowText(output));
	} else
		the_text.push_back(WindowText(L"Distance: N/A"));

	the_text.push_back(WindowText(L""));

	the_text.push_back(WindowText(fire_com_to_string(*p_commands)));

	if (p_commands->ordered_to_fire) {
		swprintf(output, 48, L"Distance: %d", p_commands->fire_target_dist);
		the_text.push_back(WindowText(output));
	} else
		the_text.push_back(WindowText(L"Distance: N/A"));

	DragWindow::draw_self();
}

SquadVarsInfo::SquadVarsInfo(int ix, int iy, int i_my_side, int i_my_squad):
SquadsWindow(ix, iy, i_my_side, i_my_squad) {
}

void SquadVarsInfo::draw_self() {
	the_text.clear();
	add_side_title();

	wchar_t output[32];
	for (int i = 0; i != n_ai_vars; ++i) {
		swprintf(output, 32, L"$%d:    %d", i, sides[my_side].squadrons[my_squad].get_script_var(i));
		the_text.push_back(WindowText(output));
	}

	DragWindow::draw_self();
}

DebugWindow::DebugWindow():
DragWindow(0, 0, window_id_none, WFLAG_STATIC | WFLAG_CANTCLOSE | WFLAG_TILED) {
	rect.w = info_window_width;
	rect.h = info_window_height;

	init_rects();
}

void DebugWindow::draw_self() {
	the_text.clear();
	wchar_t output[64];
	swprintf(output, 64, L"Current view position: %d %d", world.viewx, world.viewy);
	the_text.push_back(WindowText(output));
	swprintf(output, 64, L"World frame counter: %d", world.frame_counter);
	the_text.push_back(WindowText(output));

	DragWindow::draw_self();
}

void DebugWindow::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (the_msg == WC_CLOSE_GROUP_WIN)
		closed = true;
}

///

Options::Options():
Menu_Base(0, 0, window_id_none, 0) {
	message_windows(WC_OPTIONS_WIN_OPEN, 0, 0, window_id_all, window_id_none);

	world.pause(false);

	add_item(L"Return to game", RTS_RETURN_TO_GAME);

	add_blank_item();

	add_item(L"Scroll speed", RTS_SCROLL_SPEED);

	add_blank_item();

	add_item(L"Sound volume", OPT_SOUND_VOLUME);
	add_item(L"Music volume", OPT_MUSIC_VOLUME);
	
	add_blank_item();

	add_item(L"Restart mission", RTS_RESTART);
	add_item(L"Exit to main menu", WC_QUIT);

	init_rects();
	center_window();
}

Options::~Options() {
	message_windows(WC_OPTIONS_WIN_CLOSED, 0, 0, window_id_all, window_id_none);	
	world.unpause(false);
}

void Options::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR) {
		switch (current_selection.choice) {
		case RTS_SCROLL_SPEED:
			my_windows.push_back(GenWindow(rect.x, rect.y, RTS_SCROLL_SPEED, 0, 0, 0));
			closed = true;
			break;

		case OPT_SOUND_VOLUME:
			my_windows.push_back(GenWindow(rect.x, rect.y, OPT_SOUND_VOLUME, 0, 0, 0));
			break;

		case OPT_MUSIC_VOLUME:
			my_windows.push_back(GenWindow(rect.x, rect.y, OPT_MUSIC_VOLUME, 0, 0, 0));
			break;

		case RTS_RESTART:
			my_windows.push_back(GenWindow(0, 0, RTS_RESTART, 0, 0, 0));
			break;

		case WC_QUIT:
			gs_to = GST_MAIN_MENU;
			break;

		case RTS_RETURN_TO_GAME:
			closed = true;
			break;
		}
	}
}

Briefing::Briefing():
WrappedString(0, 0, SelectMission::briefing_width, SelectMission::briefing_height, world.mission_name + L"\n\n" + world.mission_briefing, true, WFLAG_CENTER) {
	message_windows(WC_OPTIONS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
	world.pause(false);
}

Briefing::~Briefing() {
	message_windows(WC_OPTIONS_WIN_CLOSED, 0, 0, window_id_all, window_id_none);	
	world.unpause(false);
	PortraitText::restart_current_comm();
}

MissionFailed::MissionFailed(const wstring& failure_msg):
Menu_Base(0, 0, window_id_none, 0) {
	message_windows(WC_OPTIONS_WIN_OPEN, 0, 0, window_id_all, window_id_none);
	world.pause(false);

	add_item(L"Mission Failed: " + failure_msg, WC_NOTHING, 0);
	add_blank_item();

	for (int i = 0; i != failure_msg.size(); ++i) {
		if (failure_msg[i] == '\n')
			add_blank_item();
	}

	add_item(L"Restart mission", WC_YES);
	add_item(L"Return to main menu", WC_NO);

	init_rects();
	center_window();
}

void MissionFailed::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR) {
		switch(current_selection.choice) {
		case WC_YES:
			gs_to = GST_RELOAD;
			closed = true;
			break;

		case WC_NO:
			gs_to = GST_MAIN_MENU;
			break;
		}
	}
}

RestartQ::RestartQ():
Menu_Base(0, 0, window_id_none, 0) {
	add_item(L"Are you sure you want to restart the battle?", WC_NOTHING, 0, normal_font.get_height() * 2);
	add_item(L"Yes", WC_YES);
	add_item(L"No", WC_NO);

	init_rects();
	center_window();
}

void RestartQ::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR) {
		switch(current_selection.choice) {
		case WC_YES:
			gs_to = GST_RELOAD;
			closed = true;
			break;

		case WC_NO:
			closed = true;
			break;
		}
	}
}

GameSpeedSlider::GameSpeedSlider(int ix, int iy, int flags):
SliderWithUnits(ix, iy, max_world_update_interval - world.update_interval, 0, max_world_update_interval, L"Game speed", L"", window_id_none, flags) {}

bool GameSpeedSlider::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	bool ret = Slider::mouse_m(state, x, y);

	if (b_slider_drag)
		world.set_update_interval(max_world_update_interval - slider_var);

	return ret;
}

void GameSpeedSlider::draw_self() {
	slider_var = max_world_update_interval - world.update_interval;
	if (world.paused || world.update_interval == standard_interval) {
		Slider::draw_self();
		int x = rect.x + small_border_size;
		int y = rect.y + small_border_size;
		wchar_t output[48];
		if (world.paused)
			swprintf(output, 48, L"%ls: Paused", var_name.c_str());
		else
			swprintf(output, 48, L"%ls: %d - Normal", var_name.c_str(), *var_pointer);
		normal_font.render(x, y, output);
	} else
		SliderWithUnits::draw_self();
}

ScrollSpeedSlider::ScrollSpeedSlider(int ix, int iy, int flags):
SliderWithUnits(ix, iy, max_scroll_interval - world.scroll_interval, 0, max_scroll_interval, L"Scroll speed", L"", window_id_none, flags) {}

bool ScrollSpeedSlider::mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	bool ret = Slider::mouse_m(state, x, y);
	
	world.scroll_interval = max_scroll_interval - slider_var;
	
	if (world.scroll_interval > standard_interval - 10 && world.scroll_interval < standard_interval + 10) {
		world.scroll_interval = standard_interval;
		slider_var = max_scroll_interval - world.scroll_interval;
	}
	
	return ret;
}

const wstring move_com_to_string(const AICommands& the_commands) {
	wstring ret;
	wchar_t output[32];

	if (!the_commands.b_inverse)
		ret = L"Moving towards ";
	else
		ret = L"Moving away from ";

	switch (the_commands.move_command) {
	case MC_NO_MOVE:
		ret = L"Not moving";
		break;

	case MC_MOVE_COMPASS:
		switch (the_commands.compass_target) {
		case 0:
			ret += L"North";
			break;

		case 1:
			ret += L"North-East";
			break;

		case 2:
			ret += L"East";
			break;

		case 3:
			ret += L"South-East";
			break;

		case 4:
			ret += L"South";
			break;

		case 5:
			ret += L"South-West";
			break;

		case 6:
			ret += L"West";
			break;

		case 7:
			ret += L"North-West";
			break;
		}
		break;

	case MC_MOVE_GROUP:
		//+1 so side 1/group is 1
		swprintf(output, 32, L"%ls Group %d", sides[the_commands.move_target.x].name.c_str(), the_commands.move_target.y + 1);

		ret += output;
		break;

	case MC_PATROL:
		ret = L"Patrolling ";
		swprintf(output, 32, L"%ls Group %d", sides[the_commands.move_target.x].name.c_str(), the_commands.move_target.y + 1);
		ret += output;
		break;

	case MC_MOVE_POINT:
		swprintf(output, 32, L"(%.0f, %.0f)", the_commands.point_target.x, the_commands.point_target.y);
		ret += output;
		break;
	}

	return ret;
}

const wstring fire_com_to_string(const AICommands& the_commands) {
	wstring ret;
	wchar_t output[32];

	ret = L"Fire target: ";

	if (the_commands.ordered_to_fire) {
		//+1 so side 1/group is 1
		swprintf(output, 32, L"%ls Group %d", sides[the_commands.fire_target.x].name.c_str(), the_commands.fire_target.y + 1);
		ret += output;
	} else
		ret += L"No target";

	return ret;
}

} //end namespace
