/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Display.h"
#include "Font.h"
#include "GenWindow.h"
#include "Globals.h"
#include "LoadGameData.h"
#include "MainMenu.h"
#include "Menu_Base.h"
#include "SettingsStruct.h"
#include "Sound.h"
#include "Stuff.h"

using std::wstring;

namespace MainMenu {

MainMenu_State::MainMenu_State() {
	my_windows.push_back(GenWindow(0, 0, MM_MAIN_SM, 0, 0, 0));

	if (global_error_string != L"No error right now") {
		wstring error = L'\n' + global_error_string;
		create_info_string(error);
		global_error_string = L"No error right now";
	}
}

MainMenu_State::~MainMenu_State() {
	kill_all_windows();
}

void MainMenu_State::main() {
	display.blt(gen_pictures[GENPIC_INTRO_NEBULA], viewable_screen_rect, GL_EFFECT_SCALED);

	int y_pos = global_settings.screen_height / 2 - gen_pictures[GENPIC_TITLE]->h - ((global_settings.screen_height / 2 - gen_pictures[GENPIC_TITLE]->h) / 2);
	SDL_Rect temp_rect = {global_settings.screen_width / 2 - gen_pictures[GENPIC_TITLE]->w / 2, y_pos, gen_pictures[GENPIC_TITLE]->w, gen_pictures[GENPIC_TITLE]->h};
	if (global_settings.screen_width <= 800) {
		temp_rect.x = global_settings.screen_width / 2 - ((gen_pictures[GENPIC_TITLE]->w /1.5) / 2);
		temp_rect.w = temp_rect.w / 1.5;
		temp_rect.h = temp_rect.h / 1.5;
	}

	display.blt_fill(temp_rect, standard_colors.black, 0.3f);
	display.blt(gen_pictures[GENPIC_TITLE], temp_rect);
	wstring output = L"Version " + version_string;
	normal_font.render(global_settings.screen_width - 120, global_settings.screen_height - 20, output);

	draw_all_windows();
}

MainSM::MainSM():
Menu_Base(0, 0, window_id_none, 0) {
	add_item(L"Campaign", MM_SELECT_MISSION);
	add_item(L"Options", MM_OPTIONS);
	add_item(L"Help", WC_INFO_STRING);
	add_blank_item();
	add_item(L"Quit", WC_QUIT);

	init_rects();

	center_window();
}

void MainSM::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR) {
		switch (current_selection.choice) {
		case MM_SELECT_MISSION:
			gs_to = GST_SELECT_MISSION;
			break;

		case MM_OPTIONS:
			my_windows.push_back(GenWindow(rect.x, rect.y, MM_OPTIONS, 0, 0, WFLAG_STATIC | WFLAG_CANTCLOSE));
			closed = true;
			break;

		case WC_INFO_STRING:
			create_help_window();
			break;

		case WC_QUIT:
			gs_to = GST_THE_OS;
			break;
		}
	}
}

void MainSM::create_help_window() {
	wstring tmp_str =
	L"Welcome to Really Rather Good Battles In Space: Galactic Conflict 4320 - 4359 AD.\n\n"

	L"The single player campaign teaches you how to play as you "
	L"go along, so feel free to close this window by clicking the "
	L"cross in the top right hand corner and then select "
	L"\"Campaign\" from the main menu.\n\n"

	L"If you do want to read some documentation, for instance to "
	L"see a hot key reference, or for technical troubleshooting "
	L"information, open the folder you installed the game to and "
	L"then open \"readme.txt\"\n\n"

	L"The game also has a website, complete with a discussion "
	L"forum. You can visit it at:\n\n"

	L"http://rrgbis.sourceforge.net\n\n";

	my_windows.push_back(GenWindow(0, 0, WC_WRAPPED_STRING, 400, -1, tmp_str, false, WFLAG_CENTER));
}

bool MainSM::keyboard(SDL_keysym& keysym) {
	bool ret = false;

	if (keysym.sym == SDLK_ESCAPE) {
		gs_to = GST_THE_OS;
		ret = true;
	}

	return ret;
}

Options::Options():
Menu_Base(0, 0, window_id_none, 0) {
	refresh();
}

void Options::refresh() {
	clear_items();

	add_item(L"Changing display options may require\nyou to restart the game", WC_NOTHING, 0, normal_font.get_height() * 2);
	add_blank_item();

	add_item(L"Screen resolution", OPT_SET_RESOLUTION);

	if (global_settings.full_screen)
		add_item(L"Fullscreen: On", OPT_FULLSCREEN_TOGGLE);
	else
		add_item(L"Fullscreen: Off", OPT_FULLSCREEN_TOGGLE);
	
	switch (global_settings.texture_size) {
	case 0:
		add_item(L"Texture size: Small", OPT_HIGH_RES_TOGGLE);
		break;
	case 1:
		add_item(L"Texture size: Medium", OPT_HIGH_RES_TOGGLE);
		break;
	case 2:
		add_item(L"Texture size: Large", OPT_HIGH_RES_TOGGLE);
		break;
	}

	add_blank_item();

	add_item(L"Sound volume", OPT_SOUND_VOLUME);
	add_item(L"Music volume", OPT_MUSIC_VOLUME);

	add_blank_item();
	add_item(L"Reset all settings to defaults", OPT_DEFAULTS);
	add_item(L"Return to main menu", WC_QUIT);
	
	init_rects();
	center_window();
}
	
void Options::switch_on_choice(Uint16 x, Uint16 y) {
	if (current_selection.choice_type == MCT_LEFTCURSOR) {
		switch (current_selection.choice) {
		case OPT_FULLSCREEN_TOGGLE:
			global_settings.full_screen = !global_settings.full_screen;
			reset_graphics();
			gs_to = GST_RELOAD;
			break;

		case OPT_HIGH_RES_TOGGLE:
			++global_settings.texture_size;
			if (global_settings.texture_size == num_texture_size_levels)
				global_settings.texture_size = 0;
			refresh();
			break;

		case OPT_SET_RESOLUTION:
			my_windows.push_back(GenWindow(rect.x, rect.y, OPT_SET_RESOLUTION, 0, 0, 0));
			break;

		case OPT_SOUND_VOLUME:
			my_windows.push_back(GenWindow(rect.x, rect.y, OPT_SOUND_VOLUME, 0, 0, 0));
			break;

		case OPT_MUSIC_VOLUME:
			my_windows.push_back(GenWindow(rect.x, rect.y, OPT_MUSIC_VOLUME, 0, 0, 0));
			break;

		case OPT_DEFAULTS:
			global_settings.full_screen = true;
			global_settings.screen_width = 1024;
			global_settings.screen_height = 768;
			global_settings.bpp = 32;
			reset_graphics();
			global_settings.texture_size = 1;
			global_settings.sound_volume = 128;
			global_settings.music_volume = 90;
			sound.set_sound_volume(global_settings.sound_volume);
			sound.set_music_volume(global_settings.music_volume);
			refresh();
			break;

		case WC_QUIT:
			closed = true;
			my_windows.push_back(GenWindow(0, 0, MM_MAIN_SM, 0, 0, 0));
			break;
		}
	}
}

bool Options::keyboard(SDL_keysym& keysym) {
	bool ret = false;

	if (keysym.sym == SDLK_ESCAPE) {
		gs_to = GST_RELOAD;
		ret = true;
	}

	return ret;
}

} //end namespace

