/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Credits.h"
#include "Display.h"
#include "Globals.h"
#include "Group.h"
#include "LoadGameData.h"
#include "Font.h"
#include "Intro.h"
#include "MainMenu.h"
#include "MissionEditor.h"
#include "RTS.h"
#include "SelectMission.h"
#include "SettingsStruct.h"
#include "Side.h"
#include "Sound.h"
#include "Stuff.h"

#ifndef WIN32
//for chdir
#include <unistd.h>
#endif

#include <iostream>
#include <stdexcept>
#include <iterator>

using std::cout;
using std::endl;
using std::runtime_error;
using std::terminate;
using std::list;
using std::string;
using std::wstring;

//Global just to this file:
GameState* game = 0;
bool fps_state = 0;
int last_switch_time; //set in init

// Function declarations //////////////////////////////////
void find_home_path();
bool deal_with_args(int argc, char* argv[], string& mission_str, bool& edit_mode);
void game_init();
bool game_main();
void poll_input();
void deal_with_event(SDL_Event& event);
void fps_counter();
void game_shutdown(bool do_save = true);
void exception_shutdown(runtime_error e);

int main(int argc, char* argv[]) {
	try {
		string mission_str;
		bool edit_mode;
		if (!deal_with_args(argc, argv, mission_str, edit_mode))
			return 0;
		game_init();

		if (mission_str != "") {
			//loading mission requires we set gs_to so load_music knows to look in the mission folder
			if (edit_mode)
				gs_to = GST_MISSION_EDITOR;
			else
				gs_to = GST_BATTLE;
			world.load_mission(mission_str);
		}
	} catch(runtime_error e) {
		const char* error = e.what();
		write_log(string_to_wstring(error));
		game_shutdown(false);
		terminate();
	}
	
	try {
		while (game_main());
	} catch(runtime_error e) {
		exception_shutdown(e);
	}

	game_shutdown();

	return 0;
}

void find_home_path() {
#ifndef WIN32
	char *home = 0;

	home = getenv("HOME");

	if (home) {
		global_settings.user_home_path = home;
		global_settings.user_home_path += "/.rrgbis/";
	}
#endif
}

bool deal_with_args(int argc, char* argv[], string& mission_str, bool& edit_mode) {
	if (argc == 1)
		return true;

	edit_mode = false;
	bool valid_args = true;

	for (int i = 1; i != argc; ++i) {
		string the_arg = argv[i];

		if ((the_arg == "-mission" || the_arg == "-edit") && argc > i + 1) {
			mission_str = argv[i + 1];
			if (the_arg == "-edit")
				edit_mode = true;
			++i;		
		} else
			valid_args = false;
	}

	if (!valid_args) {
		cout << "Usage: ./rrgbis" << endl << endl;
		cout << "Really Rather Good Battles in Space: Galactic Conflict 4320 - 4359 AD. A real time strategy game with fleets of spaceships" << endl << endl;
		cout << "To change game settings, either use the in-game options menu or otherwise edit ~/.rrgbis/settings.dat" << endl << endl;
		return false;
	}

	return true;
}

void game_init() {
	find_home_path();
	global_settings.load_settings();
	
	srand(SDL_GetTicks());
	sound.init();
	display.init();
	setup_lookup_tables();
	standard_colors.init();
	load_standard_graphics();
	load_ai_scripts();

	last_switch_time = SDL_GetTicks();
}


bool game_main() {
	now = SDL_GetTicks();

	if (world.update_interval <= time_acceleration_interval) {
		//1000ms/20fps = 50
		if (now - last_switch_time > 50) {
			skip_display_frame = false;
			last_switch_time = now;
		} else
			skip_display_frame = true;
	} else {
		//1000ms/60fps = 16 ish, though this will get rounded to 20
		//try not to hose CPU if not neccessary
		int diff = 16 - (now - last_switch_time);
		if (diff > 10) {
			SDL_Delay(diff - 2);
			now = SDL_GetTicks();
		}
		skip_display_frame = false;
		last_switch_time = now;
	}

	poll_input();
	update_windows();
	sound.update();

	if (gs_to == gs_current)
		game->main();

	else try {
			safe_delete(game);

			if (gs_to == GST_RELOAD) {
				if (gs_current == GST_BATTLE)
					world.load_mission(world.mission_folder_name);
				gs_to = gs_current;
			}

			switch (gs_to) {
			case GST_INTRO:
				game = new Intro::Intro_State;
				break;

			case GST_CREDITS:
				game = new Credits::Credits_State;
				break;

			case GST_MAIN_MENU:
				game = new MainMenu::MainMenu_State;
				break;

			case GST_SELECT_MISSION:
				game = new SelectMission::SelectMission_State;
				break;

			case GST_MISSION_EDITOR:
				game = new MissionEditor::MissionEditor_State;
				break;

			case GST_BATTLE:
				game = new RTS::RTS_State;
				break;

			case GST_THE_OS:
				return false;
				break;
			}
			
			//make sure menus have a starting event to force them to update screen before user
			//input in spite of SDL_WaitEvent
			SDL_Event event;
			event.type = SDL_USEREVENT;
			SDL_PushEvent(&event);

			//make sure we don't get stuck in poll event loop before display is updated on fast computers
			last_switch_time -= 1000;
			
			gs_current = gs_to;
		} catch(runtime_error e) {
			if (gs_to != GST_MAIN_MENU) {
				clear_sides();
				kill_all_windows();
				wstring error = string_to_wstring(e.what());
				write_log(error);
				global_error_string = error;
				
				if (gs_current == GST_MAIN_MENU)
					gs_current = GST_THE_OS;
				gs_to = GST_MAIN_MENU;
			} else
				throw runtime_error(e);
		}

	if (fps_state == true)
		fps_counter();
	
	static SDL_Rect mouse_rect = {0, 0, gen_pictures[current_cursor_type]->w, gen_pictures[current_cursor_type]->h};
	
	int mx, my;
	SDL_GetMouseState(&mx, &my);

	switch (current_cursor_type) {
	//hide
	case GENPIC_CLOSEBOX:
		break;

	case GENPIC_CURSOR:
		mouse_rect.x = mx;
		mouse_rect.y = my;
		display.blt(gen_pictures[current_cursor_type], mouse_rect);
		break;

	//targeting cursors must be drawn centred on mouse point
	default:
		mouse_rect.x = mx - gen_pictures[current_cursor_type]->w / 2;
		mouse_rect.y = my - gen_pictures[current_cursor_type]->h / 2;
		display.blt(gen_pictures[current_cursor_type], mouse_rect);
		break;
	}

	//this does two things:

	//1. (outdated) in windowed mode, it prevents it going so fast
	//it actually slows down to a crawl, I think maybe
	//because blits start having to wait for other ones to finish
	//though this is no longer an issue now we do SDL_Delay in various places

	//2. In full screen mode, it means if you have the game
	//set to go more than 60fps it won't have a limit of the
	//monitor refresh rate

	if (!skip_display_frame)
		display.flip();

	return true;
}

void poll_input() {
	SDL_Event event;
	
	if (gs_current == GST_MAIN_MENU || gs_current == GST_SELECT_MISSION) {
		int result = SDL_WaitEvent(&event);
		if (!result)
			throw runtime_error("Error when waiting for event");
		deal_with_event(event);
		return;
	}
	
	while (SDL_PollEvent(&event))
		deal_with_event(event);
}

void deal_with_event(SDL_Event& event) {
	switch (event.type) {		
	case SDL_MOUSEBUTTONDOWN:
		if (game && !win_mouse_d(event.button.button, event.button.x, event.button.y))
			game->mouse_d(event.button.button, event.button.x, event.button.y);
		break;

	case SDL_MOUSEBUTTONUP:
		if (game)
			game->mouse_u(event.button.button, event.button.x, event.button.y);
		break;
		
	case SDL_MOUSEMOTION:
		mouse_over_ui = false;
		win_mouse_m(event.motion.state, event.motion.x, event.motion.y);
		if (game)
			game->mouse_m(event.motion.state, event.motion.x, event.motion.y);
		break;
		
	case SDL_KEYDOWN:
		if (win_key_d(event.key.keysym))
			break;
			
		switch(event.key.keysym.sym) {
		case SDLK_F5:
			if (global_settings.debug) {
				if (fps_state == false)
					fps_state = true;
				else
					fps_state = false;
			}
			break;
		}
		
		if (game)
			game->keyboard(event.key.keysym);			
		break;

	case SDL_KEYUP:
		win_key_u(event.key.keysym);
		break;
	
	/* FIXME this doesn't work at all
	case SDL_ACTIVEEVENT:
		if (event.active.state & SDL_APPACTIVE && event.active.gain == 0) {
			while (1) {
				if (SDL_PollEvent(&event)) {		
					if (event.active.state == SDL_APPACTIVE && event.active.gain == 1)
						break;
				}
				SDL_Delay(500);
			}
		}
		break;
	*/

	case SDL_QUIT:
		gs_to = GST_THE_OS;
		break;
		
	default:
		break;
	}
}

void fps_counter() {
	static int fps_counter = 0;
	static int fps_save = 0;
	static unsigned int fps_timer = SDL_GetTicks();

	if (now - fps_timer > 1000) {
		fps_save = fps_counter;
		fps_counter = 0;
		fps_timer = now;
	} else
		++fps_counter;

	wchar_t output[60];
	swprintf(output, 60, L"FPS: %d", fps_save);
	normal_font.render(50, global_settings.screen_height - 20, output);
}

void game_shutdown(bool do_save) {
	if (do_save)
		global_settings.save_settings();

	clear_standard_graphics();

	//should already be deleted unless an exception has been thrown
	//delete before shutting down sound, or otherwise deleting game might cause a sound function to be called
	safe_delete(game);

	display.shutdown();
	sound.shutdown();

	if (global_settings.bdp != "pwd") {
		#ifndef WIN32
		chdir(global_settings.rem_dir.c_str());
		#endif
	}
}

void exception_shutdown(runtime_error e) {
	const wstring except_str = string_to_wstring(e.what());

	write_log(L"Exception shutdown due to: " + except_str);
	
	const wstring display_str = L"Major problem:\n" + except_str + L"\n\nClick the left mouse button to exit";
	create_info_string(display_str, true);
	list<GenWindow>::reverse_iterator iter = my_windows.rbegin();
	iter->draw_self();

	display.flip();

	while (1) {
		SDL_Event event;
		while (SDL_PollEvent(&event)) {		
			if (event.type == SDL_MOUSEBUTTONDOWN || event.type == SDL_KEYDOWN) {
				game_shutdown();
				terminate();
			}
		}
	}
}

