/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#ifndef GUARD_Inlines
#define GUARD_Inlines

#include "GLSDL.h"

#include "GlobalConstants.h"

#include <cstdlib>
#include <cctype>
#include <string>
#include <cmath>
#include <algorithm>

//the passed argument must be a reference if this is to work with handles
//as well as pointers
template <class PointerClass>
inline void safe_delete(PointerClass& p) {
	if (p) {
		delete p;
		p = 0;
	}
}

inline int not_space(char c) {
	return !std::isspace(c);
}

template <class Input_Iter>
inline int iter_to_int(Input_Iter& iter, const Input_Iter& lineEnd) {
	//make a max_script_digits digit c-style string
	char temp_array[max_script_digits] = {0};

	for (int i = 0; iter != lineEnd && (isdigit(static_cast<int>(*iter)) || *iter == '-') && i != max_script_digits; ++i) {
		temp_array[i] = *iter;
		++iter;
	}

	return atoi(temp_array);
}

//function never gets used
template <class Input_Iter>
inline int iter_to_large_int(Input_Iter& iter, const Input_Iter& lineEnd) {
	//make a 20 digit c-style string
	char temp_array[20] = {0};

	for (int i = 0; isdigit(static_cast<unsigned char>(*iter)) && iter != lineEnd && i != 20; ++i) {
		temp_array[i] = *iter;
		++iter;
	}

	return atoi(temp_array);
}

template <class Input_Iter>
inline int find_not_comment(Input_Iter& iter, const Input_Iter& lineEnd) {
	iter = std::find_if(iter, lineEnd, not_space);

	//ignore empty lines and comments and labels
	if (iter == lineEnd || *iter == '#')
		return 0;
	else
		return 1;
}

//used by projectiles because for them working out distance is only for visual effect
//and so accuracy is not so important as speed of execution
inline float fast_dist(float xf, float yf) {
	int x = abs(static_cast<int>(xf));
	int y = abs(static_cast<int>(yf));

	int min = (std::min)(x, y);

	//FIXME is this even what I supposed to put?
	return static_cast<float>((x + y - (min / 2) - (min / 4) + (min / 16)));
}

inline float slow_dist(float x, float y) {
	float z = x*x + y*y;
	return std::sqrt(z);
}

inline bool point_in_rect(int x, int y, const SDL_Rect& rect) {
	if (x >= rect.x
	&& x < rect.x + rect.w
	&& y >= rect.y
	&& y < rect.y + rect.h)
		return true;
	return false;
}

inline const std::string wstring_to_string(const std::wstring& str) {
	std::string tmp(str.begin(), str.end());
	return tmp;
}

inline const std::wstring string_to_wstring(const std::string& str) {
	std::wstring tmp(str.begin(), str.end());
	return tmp;
}

#endif


