/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.

   See the COPYING file for more details.
*/

#ifndef GUARD_Group
#define GUARD_Group

#include "GenericHandle.h"
#include "Enums.h"
#include "Group_Base.h"

#include <string>

class AIInterpreter;
class Squadron;

class Group {
public:
	Group(int i_my_side, int i_my_squad, int i_my_group, const std::wstring& i_name, CoordsInt i_starting_coords, bool face_left);
	Group(int i_my_side, int i_my_squad, int i_my_group, const std::wstring& i_name, int i_parent_cap, CoordsInt i_starting_coords);

	bool set_pos(float ix, float iy) {return h_group->set_pos(ix, iy);}
	bool go_to_start_coords() {return h_group->go_to_start_coords();}
	void init(int i_ai_stagger) {h_group->init(i_ai_stagger);}

	void mouse_d(Uint16 x, Uint16 y, CoordsFloat center) {if (h_group->alive) h_group->mouse_d(x, y, center);}
	//for mission editor
	void mouse_d_set_pos(Uint16 x, Uint16 y, CoordsFloat center) {h_group->mouse_d_set_pos(x, y, center);}

	bool check_for_cursor(Uint16 x, Uint16 y) const {if (h_group->alive) return h_group->check_for_cursor(x, y); else return false;}

	void add_waypoint(CoordsFloat waypoint) {if (h_group->alive) h_group->add_waypoint(waypoint);}
	void launch(const std::wstring& i_ai_filename, int mission_slot, int launch_slot) {if (h_group->alive) h_group->launch(i_ai_filename, mission_slot, launch_slot);}
	void change_ai_script(const std::wstring& i_ai_filename) {if (h_group->alive) h_group->change_ai_script(i_ai_filename);}
	void run_group_ai() {if (h_group->alive) h_group->run_group_ai();}
	void move() {if (h_group->alive) h_group->move();}
	void run_fire_commands() {if (h_group->alive) h_group->run_fire_commands();}
	void check_if_scanned() {h_group->check_if_scanned();}
	void upkeep() {if (h_group->alive) h_group->upkeep();}
	void dead_upkeep() {h_group->dead_upkeep();}

	void get_unit_target_info(TargetDesc& target_info) {h_group->get_unit_target_info(target_info);}
	void been_hit(int which_unit, int power) {if (h_group->alive) h_group->units[which_unit].been_hit(power);}
	void destroy() {if (h_group->alive) h_group->destroy();}
	void force_fill_fog(bool make_visible) {if (h_group->alive) h_group->force_fill_fog(make_visible);}

	//don't check alive, we might want to draw explosions - therefore leave checking for alive to individual units
	//do, however, check for if they are in scanning range
	void set_screen_rect() {h_group->set_screen_rect();}

	//planets don't need scanning to be shown
	void draw_self_back_back() {h_group->draw_self_back_back();}
	void draw_self_back() {if (sides[0].scanned_groups[h_group->my_side][h_group->my_group] || debug_display_all_groups) h_group->draw_self_back();}
	void draw_self_middle() {if (sides[0].scanned_groups[h_group->my_side][h_group->my_group] || debug_display_all_groups) h_group->draw_self_middle();}
	void draw_self_front() {if (sides[0].scanned_groups[h_group->my_side][h_group->my_group] || debug_display_all_groups) h_group->draw_self_front();}
	void draw_bound() {if ((sides[0].scanned_groups[h_group->my_side][h_group->my_group] || debug_display_all_groups) && h_group->alive) h_group->draw_bound();}

	//always draw laser fire even if not in scanning range. And don't draw range circles if dead.
	void draw_big_laser() {if (h_group->alive) h_group->draw_big_laser();}

	void toggle_draw_bound() {h_group->toggle_draw_bound();}
	void toggle_draw_number() {h_group->toggle_draw_number();}
	void toggle_draw_weapon_range() {h_group->toggle_draw_weapon_range();}
	void display_missions_fuel() {h_group->display_missions_fuel();}
	void hide_missions_fuel() {h_group->hide_missions_fuel();}

	//Get
	CoordsFloat get_center() const {return h_group->get_center();}
	/*
	ish - ignores both: 
	a) fact that we test from your centre, not your nearest side
	b) only returns mid point of closest side, not the actual closest point
	*/
	CoordsFloat get_closest_point(const CoordsFloat you) const {return h_group->get_closest_point(you);}
	CoordsFloat get_unit_center(int which_unit) const {return h_group->units[which_unit].get_center();}
	int find_distance_to(int n_side, int n_group) const {return h_group->find_distance_to(n_side, n_group);}
	void get_dimensions(int& giveWidth, int& giveHeight) const {giveWidth = h_group->width; giveHeight = h_group->height;}
	void get_rect(Rect32& give_rect) const {return h_group->get_rect(give_rect);}
    int get_on_screen() const {return h_group->on_screen;}

	UnitType get_type() const {return h_group->get_type();}
	bool is_small() const {return h_group->is_small();}
	bool is_big_ship() const {return h_group->is_big_ship();}
	int get_launch_slot() {return h_group->get_launch_slot();}
	CoordsFloat get_hangar_bay_coords(bool alpha_wing) const {return h_group->get_hangar_bay_coords(alpha_wing);}

	const std::wstring get_unit_name() const {return h_group->data_filename;}
	int get_parent_cap() const {return h_group->my_parent_cap;}
	int get_in_hangar() const {return h_group->get_in_hangar();}
	int get_mission_slot() const {return h_group->my_mission;}
	bool is_alpha_wing() const {return sides[h_group->my_side].squadrons[h_group->my_squad].is_first(h_group->my_group);}
	
	int get_health() const {return h_group->get_health();}
	int get_shield() const {return h_group->get_shield();}
	int get_armour() const {return h_group->get_armour();}
	int get_health_max() const {return h_group->get_health_max();}
	int get_shield_max() const {return h_group->get_shield_max();}
	int get_armour_max() const {return h_group->get_armour_max();}
	int get_unit_shield_max() const {return h_group->units[0].get_shield_max();}
	int get_unit_armour_max() const {return h_group->units[0].get_armour_max();}
	
	int get_speed_max() const {return static_cast<int>(h_group->speed_max);}
	int get_fuel() const {return static_cast<int>(h_group->fuel_current);}
	WeaponType get_small_type() const {return h_group->units[0].get_small_type();}
	WeaponType get_big_type() const {return h_group->units[0].get_big_type();}
	int get_big_ammo() const {return h_group->get_big_ammo();}
	int get_big_ammo_max() const {return h_group->get_big_ammo_max();}
	const std::wstring get_current_mission() const {return h_group->get_current_mission();}

	int get_units_left() const {return h_group->units_left;}

	int get_capacity() const {return h_group->units[0].get_capacity();}
	int get_how_full() const {return h_group->get_how_full();}

	bool get_alive() const {return h_group->alive;}
	int get_unit_alive(int which_unit) const {return h_group->units[which_unit].get_alive();}

	bool get_speed_plane_matches_target() const {return h_group->speed_plane_matches_target;}
	const AICommands* get_the_commands() const {return &(h_group->the_commands);}
	
	bool check_for_collision(int n_side, int n_group) const {return h_group->check_for_collision(n_side, n_group);}
	int find_point_distance_from_center(float x, float y) const {return h_group->find_point_distance_from_center(x, y);}

	friend class Squadron;

private:
	GenericHandle<Group_Base> h_group;
};

#endif
