/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "Credits.h"
#include "GenWindow.h"
#include "Globals.h"
#include "Intro.h"
#include "MainMenu.h"
#include "RTS.h"
#include "SelectMission.h"

#include <string>
#include <list>
#include <stdexcept>

using std::wstring;
using std::list;
using std::runtime_error;

/*IMPORTANT NOTE:
win_mouse_d() and win_key_d() rely upon windows returning true if they are going to do anything. The fact that a window can create another window in the same list as itself can otherwise cause issues. It would seem adding a window mid loop moves the reverse iterator back one somehow, meaning a window can accidently get input twice.
 
For mouse input this is automatically handled by GenWindow_Base::mouse_d(), which will always return true if the mouse is over a given window.

Another way of doing this would be to make a copy of the my_windows list each frame, make it so windows only push back new windows on to this copy, and then at the end of each frame have the copied (and maybe altered) list assigned back to the real list. The fact that GenWindows are a handle class means this wouldn't actually l_iterally copy the entire contents of the window list twice each frame, but it would seem rather inefficient to me, nonetheless.
*/

/*ANOTHER NOTE:
Any pushing back of keyboard-reading windows must come after win_key_d(), or the new window will get the keyboard input too
*/

void update_windows() {
	//altering list whilst iterating through it, dodgy, see note above, some hackery here
	list<GenWindow>::iterator endIter = my_windows.end();

	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != endIter; ) {
		if (iter->TakeActivate()) {
			my_windows.splice(my_windows.end(), my_windows, iter);
			iter = my_windows.begin();
			endIter = my_windows.end();
		}
		else
			++iter;
	}

	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end();) {
		if (iter->GetClosed())
			iter = my_windows.erase(iter);
		else
			++iter;
	}
	
	for (list<GenWindow>::reverse_iterator iter = my_windows.rbegin(); iter != my_windows.rend(); ++iter)
		iter->update();
}

bool win_mouse_d(Uint8 button, Uint16 x, Uint16 y) {
	//means no windowed input
	bool ret = false;

	//break as soon as the front most window returns true
	for (list<GenWindow>::reverse_iterator iter = my_windows.rbegin(); iter != my_windows.rend(); ++iter) {
		if (iter->mouse_d(button, x, y)) {
			ret = true;
			break;
		}
	}

	return ret;
}

void win_mouse_m(Uint8 state, Uint16 x, Uint16 y) {
	//break as soon as the front most window deals with motion
	for (list<GenWindow>::reverse_iterator iter = my_windows.rbegin(); iter != my_windows.rend(); ++iter) {
		if (iter->mouse_m(state, x, y))
			return;
	}
}

bool win_key_d(SDL_keysym& keysym) {
	bool ret = false;
	
	//break as soon as the front most window has some keyboard input
	for (list<GenWindow>::reverse_iterator iter = my_windows.rbegin(); iter != my_windows.rend(); ++iter) {
		if (iter->keyboard(keysym)) {
			ret = true;
			break;
		}
	}

	return ret;
}

bool win_key_u(SDL_keysym& keysym) {
	bool ret = false;
	
	//break as soon as the front most window has some keyboard input
	for (list<GenWindow>::reverse_iterator iter = my_windows.rbegin(); iter != my_windows.rend(); ++iter) {
		if (iter->key_u(keysym)) {
			ret = true;
			break;
		}
	}

	return ret;
}


void message_windows(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end(); ++iter)
		iter->win_message(the_msg, parem_one, parem_two, target_id, source_id);
}

void draw_all_windows() {
	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end(); ++iter)
		iter->draw_self();
}

void kill_all_windows() {
	//we can't use clear because with clear the size of the list
	//isn't set to 0 until the clearing has finished, yet
	//the base window destructor has to find the size
	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end();)
		iter = my_windows.erase(iter);
}

void kill_window(int id) {
	for (list<GenWindow>::iterator iter = my_windows.begin(); iter != my_windows.end(); ++iter) {
		GenWindow_Base* win_pointer = iter->get_pointer();
		if (win_pointer->my_id == id) {
			win_pointer->win_message(WC_YOU_CLOSE, 0, 0, id, window_id_none);
			return;
		}
	}

	char output[80];
	sprintf(output, "Unable to find window with ID %d", id);
	throw runtime_error(output);

	return;
}

GenWindow::GenWindow(int ix, int iy, WindowChoice i_type, int parem_one, int parem_two, int flags) {
	switch (i_type) {
	case MM_MAIN_SM:
		h_window = new MainMenu::MainSM();
		break;
		
	case MM_OPTIONS:
		h_window = new MainMenu::Options();
		break;

	case OPT_SET_RESOLUTION:
		h_window = new ChooseResolution();
		break;

	case OPT_SOUND_VOLUME:
		h_window = new SoundVolume(ix, iy, flags);
		break;

	case OPT_MUSIC_VOLUME:
		h_window = new MusicVolume(ix, iy, flags);
		break;
	
	case SM_SELECT_MISSION_MENU:
		h_window = new SelectMission::SelectMissionMenu(ix, iy, flags);
		break;
	
	case RTS_BRIEFING:
		h_window = new RTS::Briefing();
		break;

	case RTS_OPTIONS:
		h_window = new RTS::Options();
		break;

	case RTS_MISSIONS:
		h_window = new RTS::Missions();
		break;
			
	case RTS_SQUAD_LIST:
		h_window = new RTS::SquadList(ix, iy, parem_one, parem_two, flags);
		break;

	case RTS_SQUAD_STATS:
		h_window = new RTS::SquadStatsInfo(ix, iy, parem_one, parem_two);
		break;

	case RTS_SQUAD_STATUS:
		h_window = new RTS::SquadStatusReport(ix, iy, parem_one, parem_two);
		break;

	case RTS_SQUAD_VARS:
		h_window = new RTS::SquadVarsInfo(ix, iy, parem_one, parem_two);
		break;

	case RTS_DEBUG_WIN:
		h_window = new RTS::DebugWindow();
		break;

	case RTS_TARGET_RECON:
		h_window = new RTS::TargetMove(false);
		break;

	case RTS_TARGET_PATROL:
		h_window = new RTS::TargetMove(true);
		break;

	case RTS_TARGET_ATTACK:
		h_window = new RTS::TargetAttack();
		break;

	case RTS_RESTART:
		h_window = new RTS::RestartQ();
		break;
		
	case RTS_GAME_SPEED:
		h_window = new RTS::GameSpeedSlider(ix, iy, flags);
		break;
		
	case RTS_SCROLL_SPEED:
		h_window = new RTS::ScrollSpeedSlider(ix, iy, flags);
		break;

	case RTS_PORTRAIT_TEXT:
		h_window = new RTS::PortraitText();
		break;

	default:
		throw runtime_error("GenWindow standard constructor didn't recognise window type");
		break;
	}
}

GenWindow::GenWindow(WindowChoice i_type, const std::wstring& i_the_string) {
	h_window = new RTS::MissionFailed(i_the_string);
}

GenWindow::GenWindow(int ix, int iy, WindowChoice i_type, int iw, int ih, const wstring& the_string, bool bold_title, int flags) {
	if (i_type == INT_INTRO_TEXT)
		h_window = new Intro::IntroText(ix, iy, iw, ih, the_string, bold_title, flags);
	else if (i_type == CREDITS_CREDITS_TEXT)
		h_window = new Credits::CreditsText(ix, iy, iw, ih, the_string, bold_title, flags);
	else
		h_window = new WrappedString(ix, iy, iw, ih, the_string, bold_title, flags);
}

GenWindow::GenWindow(int ix, int iy, WindowChoice i_type, int iSliderVar, int i_var_min, int i_var_max, const wstring& iVarName, const wstring& iVarUnits, int i_parent_id, int flags) {
	h_window = new SliderWithUnits(ix, iy, iSliderVar, i_var_min, i_var_max, iVarName, iVarUnits, i_parent_id, flags);
}

GenWindow::GenWindow(int ix, int iy, WindowChoice i_type, int* iVarPointer, int i_var_min, int i_var_max, const wstring& iVarName, const wstring& iVarUnits, int i_parent_id, int flags) {
	h_window = new SliderWithUnits(ix, iy, iVarPointer, i_var_min, i_var_max, iVarName, iVarUnits, i_parent_id, flags);
}

GenWindow::GenWindow(const wstring& i_the_string) {
	h_window = new InfoString(i_the_string);
}

GenWindow::GenWindow(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const wstring& i_desc, bool minus_width, bool i_active) {
	h_window = new RTS::ControlIcon(i_type, i_rect,i_pic, i_desc, minus_width, i_active);
}

GenWindow::GenWindow(WindowChoice i_type, SDL_Rect& i_rect, SDL_Surface* i_pic, const wstring& i_desc, int i_text_offset_x, bool i_active) {
	h_window = new RTS::ControlIcon(i_type, i_rect,i_pic, i_desc, i_text_offset_x, i_active);
}

////

bool GenWindow::TakeActivate() {
	bool ret = h_window->activate;
	h_window->activate = 0;
	return ret;
}

void GenWindow::win_message(WindowChoice the_msg, int parem_one, int parem_two, int target_id, int source_id) {
	if (source_id != h_window->my_id)
		h_window->win_message(the_msg, parem_one, parem_two, target_id, source_id);
}

