/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

/*
This file based on glSDL 0.7 (c) David Olofson, 2001-2004
*/

#ifndef	GUARD_GLSDL
#define	GUARD_GLSDL

#define HAVE_OPENGL

#include "SDL.h"
//also wraps IMG_Load
#include "SDL_image.h"
#include "SDL_opengl.h"

const int GL_EFFECT_SCALED = 1;
const int GL_EFFECT_HFLIPPED = 2;

class OpenGLColor {
public:
	OpenGLColor(): sdl_color(0xFFFFFFFF) {}
	OpenGLColor& operator=(const OpenGLColor& color_two) {r = color_two.r; g = color_two.g; b = color_two.b; sdl_color = color_two.sdl_color; return *this;}

	GLfloat r;
	GLfloat g;
	GLfloat b;

	Uint32 sdl_color;
};

/*----------------------------------------------------------
	SDL style API
----------------------------------------------------------*/

typedef enum glSDL_TileModes
{
	GLSDL_TM_SINGLE,
	GLSDL_TM_HORIZONTAL,
	GLSDL_TM_VERTICAL,
	GLSDL_TM_HUGE
} glSDL_TileModes;


typedef struct glSDL_TexInfo
{
	int		textures;
	int		*texture;
	int		texsize;	/* width/height of OpenGL texture */
	glSDL_TileModes	tilemode;
	int		tilew, tileh;	/* At least one must equal texsize! */
	int		tilespertex;

	/* Area of surface to download when/after unlocking */
	SDL_Rect	invalid_area;
} glSDL_TexInfo;

#define	GLSDL_FIX_SURFACE(s)	(s)->unused1 = 0;
#define	IS_GLSDL_SURFACE(s)	((s) && texinfotab && glSDL_GetTexInfo(s))

/*
 *
 * SDL can be closed as usual (using SDL_ calls), but you should call
 * glSDL_Quit() (kludge) to allow glSDL to clean up it's internal stuff.
 */
SDL_Surface *glSDL_SetVideoMode(int width, int height, int bpp, Uint32 flags);
void glSDL_Quit(void);

void glSDL_QuitSubSystem(Uint32 flags);

/* Replaces SDL_Quit() entirely, when using the override defines */
void glSDL_FullQuit(void);

SDL_Surface *glSDL_GetVideoSurface(void);

/*
 * Works like SDL_Flip(), but may also perform enqueued blits.
 * (That is, it's possible that the implementation renders
 * *nothing* until glSDL_Flip() is called.)
 */
int glSDL_Flip(SDL_Surface *screen);

void glSDL_FreeSurface(SDL_Surface *surface);

int glSDL_LockSurface(SDL_Surface *surface);
void glSDL_UnlockSurface(SDL_Surface *surface);

/*
 * Like the respective SDL functions, although they ignore
 * SDL_RLEACCEL, as it makes no sense in this context.
 */
int glSDL_SetColorKey(SDL_Surface *surface, Uint32 flag, Uint32 key);
int glSDL_SetAlpha(SDL_Surface *surface, Uint32 flag, Uint8 alpha);

/*
 * Sets up clipping for the screen, or a SDL_Surface.
 *
 * Note that this function takes both SDL_Surfaces and
 * glSDL_Surfaces.
 */
SDL_bool glSDL_SetClipRect(SDL_Surface *surface, SDL_Rect *rect);

int glSDL_BlitSurface(SDL_Surface *src, SDL_Rect *srcrect, SDL_Surface *dst, SDL_Rect *dstrect, int gl_effect = 0);

int glSDL_FillRect(SDL_Surface *dst, SDL_Rect *dstrect, const OpenGLColor& color, float alpha);

/*
 * Convert the given surface into a SDL_Surface (if it isn't
 * one already), and makes sure that the underlying SDL_Surface
 * is of a pixel format suitable for fast texture downloading.
 *
 * Note that you *only* have to use this function if you want
 * fast pixel access to surfaces (ie "procedural textures").
 * Any surfaces that aren't converted will be downloaded
 * automatically upon the first call to glSDL_BlitSurface(),
 * but if conversion is required, it will be required for
 * every glSDL_UnlockSurface() call.
 *
 * IMPORTANT:
 *	You *can* pass an SDL_Surface directly to this function,
 *	and it will try to deal with it nicely. However, this
 *	requires that a temporary SDL_Surface is created, and
 *	this surface is cached only until the texture memory is
 *	needed for new surfaces.
 */
SDL_Surface *glSDL_DisplayFormat(SDL_Surface *surface);
SDL_Surface *glSDL_DisplayFormatAlpha(SDL_Surface *surface);

SDL_Surface *glSDL_ConvertSurface
			(SDL_Surface *src, SDL_PixelFormat *fmt, Uint32 flags);
SDL_Surface *glSDL_CreateRGBSurface
			(Uint32 flags, int width, int height, int depth, 
			Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask);
SDL_Surface *glSDL_CreateRGBSurfaceFrom(void *pixels,
			int width, int height, int depth, int pitch,
			Uint32 Rmask, Uint32 Gmask, Uint32 Bmask, Uint32 Amask);
SDL_Surface *glSDL_LoadBMP(const char *file);
int glSDL_SaveBMP(SDL_Surface *surface, const char *file);



/*----------------------------------------------------------
	glSDL specific API extensions
----------------------------------------------------------*/

/*
 * Invalidate part of a texture.
 *
 * This function can be used either between calls to
 * glSDL_LockSurface() and glSDL_UnlockSurface(), or before
 * calling glSDL_DownloadSurface().
 *
 * In either case, it causes only the specified area to be
 * downloaded when unlocking the surface, or calling
 * glSDL_UnlockSurface(), respectively.
 *
 * Note that if this function is not used, glSDL assumes that
 * the entire surface has to be updated. (That is, it's safe
 * to ignore this function - it's "just a performance hack.")
 *
 * Passing a rectangle with zero height or width cancels the
 * downloading when/after unlocking the surface. Use if you
 * just want to read the texture, but feel like being nice and
 * obeying SDL_MUSTLOCK() - which is a good idea, as things
 * may change...
 *
 * Passing NULL for the 'area' argument results in the entire
 * surface being invalidated.
 *
 * NOTE: This function does NOT perform clipping! Weird or
 *       even Bad Things may happen if you specify areas
 *       that protrude outside the edges of the actual
 *       surface.
 */
void glSDL_Invalidate(SDL_Surface *surface, SDL_Rect *area);

/*
 * Make sure that the texture of the specified surface is up
 * to date in OpenGL texture memory.
 *
 * This can be used together with glSDL_UnloadSurface() to
 * implement custom texture caching schemes.
 *
 * Returns 0 on success, or a negative value if something
 * went wrong.
 */
int glSDL_UploadSurface(SDL_Surface *surface);

/*
 * Free the texture space used by the specified surface.
 *
 * Normally, glSDL should download textures when needed, and
 * unload the oldest (in terms of use) surfaces, if it runs out
 * of texture space.
 */
void glSDL_UnloadSurface(SDL_Surface *surface);

/* Some ugly "overriding"... */
#ifndef	_GLSDL_NO_REDEFINES_
/*
 * You *always* need to lock and unlock a glSDL surface in
 * order to get glSDL to update the OpenGL texture!
 */
#undef	SDL_MUSTLOCK
#define SDL_MUSTLOCK(surface)   \
  (surface->offset ||           \
  ((surface->flags & (SDL_HWSURFACE|SDL_ASYNCBLIT|SDL_RLEACCEL)) != 0) ||	\
  IS_GLSDL_SURFACE(surface))

#endif

SDL_Surface *glSDL_IMG_Load(const char *file);

#endif
