/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "RTSUnit_Base.h"
#include "FileUtils.h"
#include "Globals.h"
#include "Group.h"
#include "Inlines.h"
#include "Display.h"
#include "Projectile.h"
#include "Side.h"

using std::find;
using std::wstring;
using std::vector;

//static members:
vector<SDL_Surface*> BigUnit::big_explode_pics;

BigUnit::BigUnit(const wstring& i_name, int i_my_side, int i_my_group, bool face_left):
RTSUnit_Base(i_name, i_my_side, i_my_group) {
	load_data();
	missile_stage = W_READY;
	b_flip = face_left;
}

void BigUnit::type_dep_stats(FileReader& the_file) {
	if (small_type == WT_NONE)
		small_number = 0;
	else {
		switch (my_type) {
		case UT_FRIGATE:
			small_number = frig_small_lasers;
			if (big_type == WT_LARGE)
				have_missiles = true;
			break;

		case UT_CAPITAL:
		case UT_FREIGHTER:
			small_number = cap_small_lasers;
			have_missiles = false;
			break;

		case UT_DEFENCE_NODE:
			small_number = defence_node_small_lasers;
			have_missiles = true;
			break;

		case UT_PLANET:
			small_number = 0;
			have_missiles = false;
			break;
		}
	}

	the_file.go_in_brackets();

	CoordsInt tmp_coords;

	for (int i = 0; i != small_number; ++i) {
		tmp_coords.x = the_file.get_int();
		tmp_coords.y = the_file.get_int();
		small_positions.push_back(tmp_coords);
	}

	big_position.x = the_file.get_int_after_colon();
	big_position.y = the_file.get_int();

	switch (my_type) {
		case UT_FRIGATE:
		case UT_FREIGHTER:
		case UT_DEFENCE_NODE:
			capacity = 0;			
			break;

		case UT_CAPITAL:
			capacity = hangar_capacity;			
			break;

		case UT_PLANET:
			capacity = hangar_capacity;
			break;
	}
}

void BigUnit::fire() {
	fire_small();
	fire_big();
}

void BigUnit::fire_small() {
	//loop through this unit's weapons
	for (int i = 0; i != small_number; ++i) {
		//if aiming time is up...
		if (small_stage[i] == W_FIRING) {
			small_stage[i] = W_RELOADING;
			small_timer[i] = world.frame_counter;

			float startx;
			if (b_flip)
				startx = myx + width - small_positions[i].x;
			else
				startx = myx + small_positions[i].x;

			world.projectiles.push_back(Projectile(startx, myy + small_positions[i].y, small_targets[i], small_type, sides[my_side].laser_color));

			//once laser fire has commenced, start aiming missiles
			if (have_missiles && missile_stage == W_READY) {
				missile_stage = W_AIMING;
				missile_timer = world.frame_counter;
			}
		} else if (small_stage[i] == W_READY && small_targets[i].x != -1) {
			if (!sides[small_targets[i].x].groups[small_targets[i].y].get_in_hangar()
			&& sides[small_targets[i].x].groups[small_targets[i].y].get_alive()) {
				float startx;
				if (b_flip)
					startx = myx + width - small_positions[i].x;
				else
					startx = myx + small_positions[i].x;

				CoordsFloat weapon_coords(startx, myy + small_positions[i].y);
				CoordsFloat target = sides[small_targets[i].x].groups[small_targets[i].y].get_closest_point(weapon_coords);
				
				if (distance_within_range(weapon_coords, target, weapon_lookup[small_type].range))
					setup_small_for_firing(i);
				else
					small_targets[i].x = -1;
			} else
				small_targets[i].x = -1;
		}
	}

	if (missile_stage == W_FIRING) {
		missile_stage = W_RELOADING;
		missile_timer = world.frame_counter;

		//a missile is launched from each small weapon position simultaneously, as long as we have a valid, in range target
		//for that small weapon
		for (int i = 0; i != small_number; ++i) {
			if (small_targets[i].x == -1 || !sides[small_targets[i].x].groups[small_targets[i].y].get_alive() || sides[small_targets[i].x].groups[small_targets[i].y].get_in_hangar())
				continue;
			
			float startx;
			if (b_flip)
				startx = myx + width - small_positions[i].x;
			else
				startx = myx + small_positions[i].x;

			CoordsFloat weapon_coords(startx, myy + small_positions[i].y);
			CoordsFloat target = sides[small_targets[i].x].groups[small_targets[i].y].get_closest_point(weapon_coords);

			if (distance_within_range(weapon_coords, target, weapon_lookup[WT_MISSILE].range))
				world.projectiles.push_back(Projectile(weapon_coords.x, weapon_coords.y, small_targets[i], WT_MISSILE, standard_colors.missile_grey));
		}
	}
}

void BigUnit::fire_big() {
	if (big_type != WT_LARGE)
		return;

	//check for large laser finishing firing
	if (big_stage == W_FIRING) {
		if (world.frame_counter - big_timer > weapon_lookup[big_type].length) {
			big_stage = W_RELOADING;
			big_timer = world.frame_counter;
			if (we_hit_with_big)
				sides[big_target.x].groups[big_target.y].been_hit(big_target_unit, weapon_lookup[big_type].power);
		}
	} else if (big_stage == W_READY && big_target.x != -1) {
		if (!sides[big_target.x].groups[big_target.y].get_in_hangar()
		&& sides[big_target.x].groups[big_target.y].get_alive()) {
			float startx;
			if (b_flip)
				startx = myx + width - big_position.x;
			else
				startx = myx + big_position.x;

			CoordsFloat weapon_coords(startx, myy + big_position.y);
			CoordsFloat target = sides[big_target.x].groups[big_target.y].get_closest_point(weapon_coords);			

			if (distance_within_range(weapon_coords, target, weapon_lookup[big_type].range))
				setup_big_for_firing();
			else
				big_target.x = -1;
		} else
			big_target.x = -1;
	}
}

void BigUnit::find_in_range(vector<CoordsInt>& in_range, CoordsFloat weapon_coords, float range, bool including_small_targets) {
	for (int i = 0; i != sides.size(); ++i) {
		//only enemy sides
		if (sides[i].my_flag == sides[my_side].my_flag)
			continue;

		for (int j = 0; j != sides[i].groups.size(); ++j) {
			if (!sides[i].groups[j].get_alive()
			|| !sides[my_side].scanned_groups[i][j]
			|| sides[i].groups[j].get_in_hangar()
			|| sides[i].groups[j].get_type() == UT_PLANET)
				continue;

			if (!including_small_targets && sides[i].groups[j].is_small())
				continue;

			CoordsFloat target = sides[i].groups[j].get_closest_point(weapon_coords);

			if (distance_within_range(weapon_coords, target, range)) {
				CoordsInt temp_coords(i, j);
				in_range.push_back(temp_coords);
			}
		}
	}
}

CoordsInt BigUnit::get_weak_spot() const {
	CoordsInt ret;
    //relative to centre of unit
	ret.x = rand() % (width / 2)  - (width / 4);
	ret.y = rand() % (height / 2) - (height / 4);	
	return ret;
}

void BigUnit::random_small_explosions() {
	int dice_size = 4;
	if (width == large_explosion_width)
		dice_size = 2;
    if (rand() % dice_size)
		return;

	float x = static_cast<float>(rand() % width);
	float y = static_cast<float>(rand() % height);
	world.projectiles.push_back(Projectile(myx + x, myy + y));
}

void BigUnit::bombardment_explosions() {
	float x = static_cast<float>(rand() % (width * 3 / 4) + (width / 8)) + myx;
	float y = static_cast<float>(rand() % (height * 3 / 4) + (height / 8)) + myy;
	world.projectiles.push_back(Projectile(x, y));
	world.play_sound(x, y, SE_NEUTRON_BOMB_EXPLOSION);
}

void BigUnit::launch_drop_ships() {
	if (rand() % 10 != 9)
		return;
	CoordsInt target(1, 0);
	CoordsFloat coords = sides[my_side].groups[my_group].get_hangar_bay_coords(rand() % 2);
	world.projectiles.push_back(Projectile(coords.x, coords.y, target, WT_DROP_SHIP, standard_colors.black));
}

void BigUnit::draw_big_laser() {
	if (big_type == WT_LARGE && big_stage == W_FIRING) {
		CoordsFloat tmp = sides[big_target.x].groups[big_target.y].get_unit_center(big_target_unit);

		float startx;
		if (b_flip)
			startx = myx + width - big_position.x;
		else
			startx = myx + big_position.x;

		float finx = tmp.x + target_weak_spot.x;
		float finy = tmp.y + target_weak_spot.y;

		Projectile_Base::draw_big_laser(startx, myy + big_position.y, finx, finy, sides[my_side].laser_color);
	}
}

void BigUnit::draw_bound(bool b_draw_weapon_range) {
	if (b_draw_weapon_range) {
		for (int i = 0; i != small_number; ++i) {
			int xcoord;
			if (b_flip)
				xcoord = static_cast<int>(myx) + width - small_positions[i].x;
			else
				xcoord = static_cast<int>(myx) + small_positions[i].x;
			int ycoord = static_cast<int>(myy) + small_positions[i].y;
	
			display.draw_circle(xcoord - world.viewx, ycoord - world.viewy, static_cast<int>(weapon_lookup[small_type].range), standard_colors.white);
		}

		if (big_type != WT_NONE) {
			int xcoord;
			if (b_flip)
				xcoord = static_cast<int>(myx) + width - big_position.x;
			else
				xcoord = static_cast<int>(myx) + big_position.x;
			int ycoord = static_cast<int>(myy) + big_position.y;
	
			display.draw_circle(xcoord - world.viewx, ycoord - world.viewy, static_cast<int>(weapon_lookup[big_type].range), standard_colors.big_range_blue);
		}
	}
}

void BigUnit::upkeep() {
	RTSUnit_Base::upkeep();

	if (armour_current < 1) {
		alive = false;
		//don't start drawing explosion images until sound has been playing for a second for frigs or 6 secs for caps
		if (width == large_explosion_width) {
			anim_timer = world.frame_counter + 180;
			//FIXME a bit hacky need to check on_screen because screen_rect only updated if the group is on screen
			if (sides[my_side].groups[my_group].get_on_screen())
				world.play_sound(screen_rect, SE_LARGE_EXPLOSION);	
		} else {
			anim_timer = world.frame_counter + 30;
			//FIXME a bit hacky need to check on_screen because screen_rect only updated if the group is on screen
			if (sides[my_side].groups[my_group].get_on_screen())
				world.play_sound(screen_rect, SE_MEDIUM_EXPLOSION);
		}

		finished_anim = false;	

		for (int i = 0; i != sides[my_side].groups.size(); ++i) {
			if (sides[my_side].groups[i].get_parent_cap() == my_group && sides[my_side].groups[i].get_in_hangar())
				sides[my_side].groups[i].destroy();
		}
		return;
	}

	//check for small reloading
	for (int i = 0; i != small_number; ++i) {
		if (small_stage[i] == W_RELOADING && world.frame_counter - small_timer[i] > weapon_lookup[small_type].reload)
			small_stage[i] = W_READY;
	}

	//check for missile reloading
	if (missile_stage == W_RELOADING && world.frame_counter - missile_timer > weapon_lookup[WT_MISSILE].reload)
		missile_stage = W_READY;

	//check for small aiming (time decided when we choose
	//a target)
	for (int i = 0; i != small_number; ++i) {
		if (small_stage[i] == W_AIMING && world.frame_counter - small_timer[i] > small_aiming[i])
			small_stage[i] = W_FIRING;
	}

	//check for missile aiming (we start aiming as soon as we fire first laser), always take max_aiming time
	if (missile_stage == W_AIMING && world.frame_counter - missile_timer > max_aiming)
		missile_stage = W_FIRING;

	//planetary bombardment anim
	if (my_type == UT_PLANET && world.script_anim == ANIM_BOMBARDMENT)
		bombardment_explosions();
	else if (my_type == UT_CAPITAL && world.script_anim == ANIM_INVASION)
		launch_drop_ships();
}

void BigUnit::dead_upkeep() {
	if (!finished_anim && world.frame_counter - anim_timer < 0) {
		random_small_explosions();
	}
}

void BigUnit::select_small_targets(const AICommands& the_commands) {
	float range_small = weapon_lookup[small_type].range;

	//loop through this unit's weapons, and work out if we're in range for each one
	for (int i = 0; i != small_number; ++i) {
		if (small_stage[i] != W_READY || small_targets[i].x != -1)	 
			continue;

		float startx;
		if (b_flip)
			startx = myx + width - small_positions[i].x;
		else
			startx = myx + small_positions[i].x;
		CoordsFloat weapon_coords(startx, myy + small_positions[i].y);
			
		if (the_commands.ordered_to_fire) {
			CoordsFloat target = sides[the_commands.fire_target.x].groups[the_commands.fire_target.y].get_closest_point(weapon_coords);

			if (distance_within_range(weapon_coords, target, range_small)) {
				small_targets[i] = the_commands.fire_target;
				continue;
			}
		}

		//our manual target is not in range, so try to autoselect
		vector<CoordsInt> in_range_small;
		find_in_range(in_range_small, weapon_coords, range_small);

		if (in_range_small.size() > 0) {
			bool found = false;
			for (int j = 0; j != in_range_small.size(); ++j) {
				if (sides[in_range_small[j].x].groups[in_range_small[j].y].get_type() == UT_BOMBER) {
					small_targets[i] = in_range_small[j];
					found = true;
					break;
				}
			}
			for (int j = 0; j != in_range_small.size(); ++j) {
				if (sides[in_range_small[j].x].groups[in_range_small[j].y].get_type() == UT_FIGHTER) {
					small_targets[i] = in_range_small[j];
					found = true;
					break;
				}
			}
			//just choose the first person in the vector
			if (!found)
				small_targets[i] = in_range_small[0];
		}
	}
}

void BigUnit::select_big_targets(const AICommands& the_commands) {
	float range_big = weapon_lookup[big_type].range;
	bool use_manual_big_target = false;

	//select a target for big weapon
	if (big_type == WT_LARGE && big_stage == W_READY && big_target.x == -1) {
		float startx;
		if (b_flip)
			startx = myx + width - big_position.x;
		else
			startx = myx + big_position.x;
		CoordsFloat weapon_coords(startx, myy + big_position.y);

		if (the_commands.ordered_to_fire) {	
			CoordsFloat target = sides[the_commands.fire_target.x].groups[the_commands.fire_target.y].get_closest_point(weapon_coords);
			
			if (distance_within_range(weapon_coords, target, range_big)) {
				big_target = the_commands.fire_target;
				use_manual_big_target = true;
			}
		}
		
		if (!use_manual_big_target) {
			vector<CoordsInt> in_range_big;
			find_in_range(in_range_big, weapon_coords, weapon_lookup[big_type].range, false);

			if (in_range_big.size() > 0)
				//just choose the first person in the vector
				big_target = in_range_big[0];
		}
	}
}

void BigUnit::explode() {
	//when big ships explode they play sound for a while before starting explosion anim
	if (world.frame_counter - anim_timer < 0) {
		if (b_flip)
			display.blt(pic, screen_rect, GL_EFFECT_HFLIPPED | GL_EFFECT_SCALED);
		else
			display.blt(pic, screen_rect, GL_EFFECT_SCALED);
		return;
	}

	for (int i = 0; i != big_explode_pics.size(); ++i) {
		if (world.frame_counter - anim_timer < frames_per_anim_frame * (i + 1)) {
			if (width == large_explosion_width)
				display.blt(big_explode_pics[i], screen_rect);
			else
				display.blt(big_explode_pics[i], screen_rect, GL_EFFECT_SCALED);
			return;
		}
	}

	finished_anim = true;
}


