/*
   Copyright (C) 2006 by James Gregory
   Part of the Really Rather Good Battles In Space project
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License.
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY.
 
   See the COPYING file for more details.
*/

#include "AIInterpreter.h"
#include "Globals.h"
#include "Group.h"
#include "Inlines.h"
#include "Projectile.h"
#include "Side.h"

#include <cstdlib>

using std::isdigit;
using std::list;
using std::vector;

/*
Interpet functions should be called with the iter on the first char of the thing you wanted interpreted
Functions should return with iter on the first thing they didn't look at
*/

void AIInterpreter::init(Group_Base* i_this_group, AICommands* i_the_commands) {
	this_group = i_this_group;
	the_commands = i_the_commands;
	
	//AIInterpreter is only inited once side sizes are stable, so this is OK. This just means the code can be slightly tidier as it's annoying having to put "this_group->" everywhere
	my_side = this_group->my_side;
	my_group = this_group->my_group;
}

void AIInterpreter::get_commands() {
	//ensure bools and enums and speeds start off with 0s
	memset(the_commands, 0, sizeof(AICommands));

	//check if any save groups have gone out of scanning range - if so, forget them
	//the rest start off invalid
	for (int i = 0; i != n_ai_vars; ++i) {
		if (save_groups[i].x != -1 && !sides[my_side].scanned_groups[save_groups[i].x][save_groups[i].y]) {
			save_groups[i].x = -1;
			save_groups[i].y = -1;
		}
	}
	
	/*
	current_line to keep track of where we are
	ins_counter to prevent infinite loops
	*/

	the_text = &(world.ai_scripts[this_group->ai_filename].script);
	current_line = world.ai_scripts[this_group->ai_filename].first_line;
	ins_counter = 0;
	
	call_function();
}

/*
this is the function that calls all the other AI line interpretation functions, and uses the information returned
from them to decide which lines of the script to read and which to skip.
It gets called recursively by script function calls.
*/
void AIInterpreter::call_function() {
	vector<bool> remember_if_else_result;

	while (current_line != the_text->size()) {
		if (ins_counter >= max_ai_instructions) {
			char output[80];
			sprintf(output, "Processed maximum %d instructions", max_ai_instructions);
			this_group->report_on_script_error(output);
			return;
		}

		l_iter = (*the_text)[current_line].begin();
		l_end = (*the_text)[current_line].end();
		
		//don't want blank lines resetting the if results
		if (*l_iter == TT_COMMENT) {
			++current_line;
			continue;
		}
		
		int tab_level = 0;
		
		while (*l_iter == TT_TAB) {
			++tab_level;
			++l_iter;
		}
		
		//more tabs than bools?
		while (tab_level > static_cast<int>(remember_if_else_result.size()) - 1)
			remember_if_else_result.push_back(false);
		
		//more bools than tabs?
		while (static_cast<int>(remember_if_else_result.size()) - 1 > tab_level)
			remember_if_else_result.pop_back();

		switch (*l_iter) {
		case TT_IF: {
				++ins_counter;

				bool if_result = interpret_if();

				if (if_result == true)
					//don't skip anything
					remember_if_else_result[tab_level] = true;
				else {
					//was false, skip at least one command
					remember_if_else_result[tab_level] = false;
					skip_block(tab_level);
				}
			}
			break;


		case TT_ELSE:
			++ins_counter;

			if (remember_if_else_result[tab_level] == true)
				skip_block(tab_level);
			else
				//don't skip anything
				remember_if_else_result[tab_level] = true;
			break;
		
		case TT_ELIF:
			++ins_counter;
			
			if (remember_if_else_result[tab_level] == true)
				skip_block(tab_level);			
			else {
				bool if_result = interpret_if();

				if (if_result == true)
					//don't skip anything
					remember_if_else_result[tab_level] = true;
				else
					//if it was false, skip at least one command
					skip_block(tab_level);
			}
			break;

		case TT_RETURN:
			++ins_counter;
			return;
			break;

		case TT_JUMP: {
				++ins_counter;
				int saveLN = current_line;

				++l_iter;
				ustring function_name(l_iter, l_end);

				int calledLine = world.ai_scripts[this_group->ai_filename].function_lookup[function_name];
				//++ called line because we want to start from the line after the function label
				current_line = ++calledLine;

				call_function();
				current_line = saveLN;
			}
			break;

		case TT_START_TIMER:
			++ins_counter;
			interpret_start_timer();
			break;

		case TT_SCRIPT_VAR:
		case TT_GLOBAL_SCRIPT_VAR:
			++ins_counter;
			interpret_set_var();
			break;
		
		case TT_SaveGroup:
		case TT_GSaveGroup:
			++ins_counter;
			interpret_set_save_group();
			break;

		case TT_MOVE:
		case TT_MOVE_SLOW:
		case TT_MOVE_AWAY:
			++ins_counter;
			interpret_move();
			break;

		case TT_FIRE:
			++ins_counter;
			interpret_fire();
			break;

		case TT_PATROL:
			++ins_counter;
			interpret_patrol();
			break;

		case TT_DOCK:
			interpret_dock();

		default:
			//I think this can only be a function label.
			//So we return, as we don't want to fall-through into another function.
			return;
			break;
		}

		++current_line;
	}

	//this only happens if we reach the end of the file before
	//coming across a return command
	return;
}

void AIInterpreter::skip_block(int tab_level) {
	++current_line;

	while (current_line != the_text->size()) {
		l_iter = (*the_text)[current_line].begin();
		
		//don't want blank lines setting tab level
		if (*l_iter == TT_COMMENT) {
			++current_line;
			continue;
		}
		
		int countTabs = 0;
		while (*l_iter == TT_TAB) {
			++countTabs;
			++l_iter;
		}

		if (countTabs <= tab_level) {
			//remember ++current_line at end of main function loop
			--current_line;
			return;
		}
			
		++current_line;
	}
	
	//remember ++current_line at end of main function loop
	--current_line;
}

//this function just gets our iterator past the initial TT_IF
bool AIInterpreter::interpret_if() {
	++l_iter;
	return interpret_if_part_two();
}

bool AIInterpreter::interpret_if_part_two() {
	bool first_result;

	while (l_iter != l_end) {
		//each case must put the iter one past the end of that word/
		//operator/whatever
		switch (*l_iter) {
		/* given up on this for now
		case TT_OpenB:
			++l_iter;
			//recursively call function
			first_result = interpret_if_part_two();
			break;
		
		
		case TT_CloseB:
			++l_iter;
			//got to end, return first_result
			return first_result;
			break;
		*/
		
		case TT_Not:
			++l_iter;
			//recursively call function and invert result
			first_result = !interpret_if_part_two();
			break;

		case TT_AND:
			++l_iter;

			//as it's AND, if either side is false, we return false

			//actually, as I always compare just the first and
			//current result, not the previous and current result,
			//it doesn't work quite as you think it does
			//However, for some crazy logic reason I don't
			//actually think it's possivle to come up with
			//a situation where it makes any difference if you
			//are comparing the first or the previous result
			
			if (!first_result || !interpret_if_part_two()) {
				short_circuit_skip();
				return false;
			}
 						
			//else keep going round
			break;

		case TT_OR:
			++l_iter;
				
			if (first_result || interpret_if_part_two()) {
				short_circuit_skip();
				return true;
			} 	
			
			//else keep going round
			break;

		default:
			//assume must be the first specifier of a comparison, or FIXME or what?
			first_result = true_or_false();
			break;
		}
	}

	//as with the note above &&: it doesn't seem to make sense to
	//just return the first result, but I think for some crazy
	//reason of logic that I don't understand, this works
	return first_result;
}

//only called from "if"
bool AIInterpreter::true_or_false() {
	//scripts are allowed to specify multiple pieces of information from a single
	//group
	int rememberNSide = 0;
	int rememberNGroup = 0;
	int first_value;
	
	switch (*l_iter) {
	case TT_Our:
	case TT_NearestEnemy:
	case TT_NearestFriend:
	case TT_NearestAlly:
	case TT_SaveGroup:
	case TT_GSaveGroup: {
		unsigned char nextToken;
		//past number
		if (*l_iter == TT_SaveGroup || *l_iter == TT_GSaveGroup)
			nextToken = *(l_iter + 2);
		else
			nextToken = *(l_iter + 1);
			
		if (nextToken == TT_Equal || nextToken == TT_NotEqual)
			return are_groups_the_same();
		else
			return is_nearest_like_this();
		}
		break;
		
	case TT_AnyEnemy:
	case TT_AnyFriend:
	case TT_AnyAlly:
		return do_any_have();
		break;

	default:
		first_value = token_to_int(rememberNSide, rememberNGroup);
		break;
	}

	unsigned char operator_type = *l_iter;
	++l_iter;
	int second_value = token_to_int();

	return compare_values(first_value, operator_type, second_value);
}

bool AIInterpreter::compare_values(int first_value, const unsigned char operator_type, int second_value) {
	switch (operator_type) {
	case TT_Equal:
		if (first_value == second_value)
			return true;
		break;

	case TT_NotEqual:
		if (first_value != second_value)
			return true;
		break;

	case TT_GreaterThan:
		if (first_value > second_value)
			return true;
		break;

	case TT_LessThan:
		if (first_value < second_value)
			return true;
		break;

	case TT_GreaterThanEqual:
		if (first_value >= second_value)
			return true;
		break;

	case TT_LessThanEqual:
		if (first_value <= second_value)
			return true;
		break;
	}

	return false;
}

void AIInterpreter::interpret_start_timer() {
	++l_iter;
	int which = token_to_int();
	script_timers[which] = world.frame_counter;
}

void AIInterpreter::interpret_set_var() {
	unsigned char var_token = *l_iter;
	++l_iter;
	int which = iter_to_int(l_iter, l_end);
	
	//++
	if (*l_iter == TT_Increment) {
		switch (var_token) {
		case TT_SCRIPT_VAR:
			script_vars[which]++;
			break;
		case TT_GLOBAL_SCRIPT_VAR:
			sides[my_side].missions[this_group->my_mission].script_vars[which]++;
			break;
		}
		return;
	}

	//--
	else if (*l_iter == TT_Decrement) {
		switch (var_token) {
		case TT_SCRIPT_VAR:
			script_vars[which]--;
			break;
		case TT_GLOBAL_SCRIPT_VAR:
			sides[my_side].missions[this_group->my_mission].script_vars[which]--;
			break;
		}
		return;
	}
	
	//else skip TT_Equal
	++l_iter;

	switch (var_token) {
	case TT_SCRIPT_VAR: {
			int new_value = token_to_int();
			script_vars[which] = new_value;
		}
		break;

	case TT_GLOBAL_SCRIPT_VAR: {
			int new_value = token_to_int();
			sides[my_side].missions[this_group->my_mission].script_vars[which] = new_value;
		}
		break;
	}
}

void AIInterpreter::interpret_set_save_group() {
	const unsigned char var_token = *l_iter;
	++l_iter;
	int which = iter_to_int(l_iter, l_end);
	//already one past end, but skip equals
	++l_iter;
	
	//we don't need to remember the returned distance
	int temp;
	CoordsInt target_indices;
	
	switch (var_token)
	{
	case TT_SaveGroup:
		find_nearest_with(target_indices, temp);
		save_groups[which] = target_indices;
		break;

	case TT_GSaveGroup:
		find_nearest_with(target_indices, temp);
		sides[my_side].missions[this_group->my_mission].save_groups[which] = target_indices;
		break;
	}
}

void AIInterpreter::interpret_move() {
	//is it move/moveslow or moveaway?
	if (*l_iter == TT_MOVE_AWAY)
		the_commands->b_inverse = true;
	//we might get multiple move commands so we need to overwrite any old inverting
	else
		the_commands->b_inverse = false;

	//move or moveslow?
	if (*l_iter == TT_MOVE_SLOW)
		the_commands->speed = this_group->speed_cruise;
	//we might get multiple move commands so we need to overwrite any old inverting
	else
		the_commands->speed = this_group->speed_max;

	++l_iter;

	switch (*l_iter) {
	case TT_N:
	case TT_NorthEdge:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_N;
		break;

	case TT_NE:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_NE;
		break;

	case TT_E:
	case TT_EastEdge:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_E;
		break;

	case TT_SE:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_SE;
		break;

	case TT_S:
	case TT_SouthEdge:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_S;
		break;

	case TT_SW:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_SW;
		break;

	case TT_W:
	case TT_WestEdge:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_W;
		break;

	case TT_NW:
		the_commands->move_command = MC_MOVE_COMPASS;
		the_commands->compass_target = CD_NW;
		break;

	case TT_NEAREST_WORLD_EDGE: {
			int throwaway;
			the_commands->move_command = MC_MOVE_COMPASS;
			the_commands->compass_target = NearestEdge(throwaway);
		}
		break;

	case TT_WAYPOINT: {
		the_commands->move_command = MC_MOVE_POINT;
		++l_iter;
		int which = iter_to_int(l_iter, l_end);
		the_commands->point_target.x = this_group->waypoints[which].x;
		the_commands->point_target.y =  this_group->waypoints[which].y; 
		}
		break;

	//lets hope the specifier specifies a group
	default:
		//if -1 is returned it means there is no available target meeting
		//all the criteria
		CoordsInt target_indices;
		find_nearest_with(target_indices, the_commands->move_target_dist);
		if (target_indices.x != -1) {
			if (target_indices.x != max_players) {
				the_commands->move_command = MC_MOVE_GROUP;
				the_commands->move_target = target_indices;
			//for world edges stored in save groups
			} else {
				the_commands->move_command = MC_MOVE_COMPASS;
				the_commands->compass_target = static_cast<CompassDirection>(target_indices.y);
			}
		}
		break;
	}
}

void AIInterpreter::interpret_fire() {
	//have we already got a fire command?
	if (the_commands->ordered_to_fire) {
		this_group->report_on_script_error("Multiple fire commands received");
		return;
	}
	
	if (this_group->units[0].get_big_type() == WT_LARGE) {
		//then we have a group name
		++l_iter;

		CoordsInt target_indices;
		find_nearest_with(target_indices, the_commands->fire_target_dist);
		if (target_indices.x != -1) {
			the_commands->fire_target = target_indices;

			if (the_commands->fire_target.x != max_players)
				the_commands->ordered_to_fire = true;
			else
				this_group->report_on_script_error("Attempt to fire at world edge", current_line);
		}
	} else
		the_commands->ordered_to_fire = true;
}

void AIInterpreter::interpret_patrol() {
	++l_iter;
	
	the_commands->speed = this_group->speed_max;

	//first we have a number
	the_commands->patrol_dist = token_to_int();

	//if -1 is returned it means there is no available target meeting
	//all the criteria
	CoordsInt target_indices;
	find_nearest_with(target_indices, the_commands->move_target_dist);
	if (target_indices.x != -1) {
		the_commands->move_target = target_indices;

		if (the_commands->move_target.x != max_players)
			the_commands->move_command = MC_PATROL;
		else
			this_group->report_on_script_error("Attempt to patrol world edge", current_line);
	}
}

void AIInterpreter::interpret_dock() {
	the_commands->ordered_to_dock = true;
}


int AIInterpreter::token_to_int() {
	int give_side = -1;
	int give_group = -1;
	return token_to_int(give_side, give_group);
}

int AIInterpreter::token_to_int(int& give_side, int& give_group) {
	int cache = 0;

	switch (*l_iter) {
	case TT_Integer:
		++l_iter;
		cache = iter_to_int(l_iter, l_end);
		//we ++l_iter at the end of this function to put it one past end
		--l_iter;
		break;

	case TT_SCRIPT_VAR: {
		++l_iter;
		int which = iter_to_int(l_iter, l_end);
		cache = script_vars[which];
		//we ++l_iter at the end of this function to put it one past end
		--l_iter;
		break;
	}

	case TT_GLOBAL_SCRIPT_VAR: {
		++l_iter;
		int which = iter_to_int(l_iter, l_end);
		cache = sides[my_side].missions[this_group->my_mission].script_vars[which];
		//we ++l_iter at the end of this function to put it one past end
		--l_iter;
		break;
	}

	case TT_ScriptTimer: {
		++l_iter;
		int which = iter_to_int(l_iter, l_end);
		cache = world.frame_counter - script_timers[which];
		//we ++l_iter at the end of this function to put it one past end
		--l_iter;
		break;
	}

	case TT_RANDOM:
		cache = rand() % 100 + 1;
		break;
		
	case TT_FIGHTER:
		cache = UT_FIGHTER;
		break;

	case TT_BOMBER:
		cache = UT_BOMBER;
		break;

	case TT_FRIGATE:
		cache = UT_FRIGATE;
		break;

	case TT_CAPITAL:
		cache = UT_CAPITAL;
		break;

	case TT_FREIGHTER:
		cache = UT_FREIGHTER;
		break;

	case TT_DEFENCE_NODE:
		cache = UT_DEFENCE_NODE;
		break;

	case TT_PLANET:
		cache = TT_PLANET;
		break;

	case TT_Our:
		++l_iter;

		give_side = my_side;
		give_group = my_group;

		cache = stat_to_int(*l_iter, my_side, my_group);
		break;

	case TT_NearestEnemy: 
	case TT_NearestFriend:
	case TT_NearestAlly:
	case TT_SaveGroup:
	case TT_GSaveGroup:
		get_group_indices(give_side, give_group);
		cache = stat_to_int(*l_iter, give_side, give_group);
	break;

	case TT_NEAREST_WORLD_EDGE:
		//++ on to "distance", not past because we ++ at end
		++l_iter;

		give_side = max_players;
		give_group = NearestEdge(cache);
		break;

	case TT_NorthEdge:
	case TT_EastEdge:
	case TT_SouthEdge:
	case TT_WestEdge:
		give_side = max_players;
		give_group = TokenToEdge(cache);
		break;
		
	case TT_NumEnemy:
	case TT_NumFriend:
	case TT_NumAlly:
		cache = how_many_have();
		//we ++l_iter at the end of this function to put it one past end
		--l_iter;
		break;

	case TT_WAYPOINT: {
		++l_iter;
		int which = iter_to_int(l_iter, l_end);
		give_side = max_players;
		cache = this_group->find_point_distance_from_center(this_group->waypoints[which].x, this_group->waypoints[which].y);

		//we ++l_iter at the end of this function so don't ++ past distance
		break;
	}

	case TT_NUM_WAYPOINT:
		cache = count_waypoints();
		break;

	default:
		if (give_side != -1)
			//assume must be a group stat of a previously specified group
			cache = stat_to_int(*l_iter, give_side, give_group);
		else {
			l_iter = l_end;
			this_group->report_on_script_error("Didn't recognise token when asked to find token_to_int", current_line);
			return cache;
		}
		break;
	}
	
	//either moving on to arithmetic, or otherwise leaving one past end
	++l_iter;

	if (l_iter != l_end) {
		unsigned char the_operator = *l_iter;

		switch (the_operator) {
		case TT_ADD:
			++l_iter;
			return cache + token_to_int(give_side, give_group);
			break;
		case TT_MINUS:
			++l_iter;
			return cache - token_to_int(give_side, give_group);
			break;
		case TT_MULTIPLY:
			++l_iter;
			return cache * token_to_int(give_side, give_group);
			break;
		case TT_DIVIDE: {
			++l_iter;
			int temp = token_to_int(give_side, give_group);

			if (temp != 0)
				return cache / temp;
			else
				return max_ai_int;
			break;
			}			
		case TT_MODULO: {
			++l_iter;
			int temp = token_to_int(give_side, give_group);

			if (temp != 0)
				return cache % temp;
			else
				return max_ai_int;
			break;
			}
		}
	}

	return cache;
}

int AIInterpreter::stat_to_int(const unsigned char the_token, int n_side, int n_group) {
	if (n_side == max_players) {
		if (the_token == TT_DISTANCE) {
			return DistToXEdge(static_cast<CompassDirection>(n_group));
		} else {
			l_iter = l_end;
			this_group->report_on_script_error("Attempt to find stat other than distance of saved world edge", current_line);
			return 0;
		}
	}

	switch (the_token) {
	case TT_Number:
		return sides[n_side].groups[n_group].get_units_left();
		break;
	
	case TT_Health:
		return sides[n_side].groups[n_group].get_health();
		break;
		
	case TT_Shield:
		return sides[n_side].groups[n_group].get_shield();
		break;
	
	case TT_Armour:
		return sides[n_side].groups[n_group].get_armour();
		break;
		
	case TT_HealthMax:
		return sides[n_side].groups[n_group].get_health_max();
		break;
		
	case TT_ShieldMax:
		return sides[n_side].groups[n_group].get_shield_max();
		break;
		
	case TT_ArmourMax:
		return sides[n_side].groups[n_group].get_armour_max();
		break;
		
	case TT_UnitShieldMax:
		return sides[n_side].groups[n_group].get_unit_shield_max();
		break;
		
	case TT_UnitArmourMax:
		return sides[n_side].groups[n_group].get_unit_armour_max();
		break;

	case TT_SPEED:
		return sides[n_side].groups[n_group].get_speed_max();
		break;

	case TT_FUEL:
		return sides[n_side].groups[n_group].get_fuel();
		break;

	case TT_DISTANCE:
		return this_group->find_distance_to(n_side, n_group);
		break;

	//FIXME no error checking
	case TT_DISTANCE_FROM: {
		++l_iter;
		int side, group;
		//could use get_group_indices instead to include our, enemy etc also ,though we would then have to remember to
		//--iter again
		which_saved_group(side, group);
		return sides[side].groups[group].find_distance_to(n_side, n_group);
	}

	case TT_SmallRange:
		return static_cast<int>(weapon_lookup[sides[n_side].groups[n_group].get_small_type()].range);
		break;

	case TT_SmallPower:
		return weapon_lookup[sides[n_side].groups[n_group].get_small_type()].power;
		break;

	case TT_BigRange:
		return static_cast<int>(weapon_lookup[sides[n_side].groups[n_group].get_big_type()].range);
		break;

	case TT_BigPower:
		return weapon_lookup[sides[n_side].groups[n_group].get_big_type()].power;
		break;

	case TT_BIG_AMMO:
		return sides[n_side].groups[n_group].get_big_ammo();
		break;

	case TT_Left:
		return sides[n_side].groups[n_group].get_units_left();
		break;
		
	case TT_MissTarget: {
			int total = 0;
			CoordsInt the_group(n_side, n_group);

			for (list<Projectile>::iterator iter = world.projectiles.begin(); iter != world.projectiles.end(); ++iter) {
				if (iter->get_type() == WT_MISSILE && iter->get_target() == the_group)
					++total;
			}
			return total;
		}
		break;

	case TT_TorpTarget: {
			int total = 0;
			CoordsInt the_group(n_side, n_group);

			for (list<Projectile>::iterator iter = world.projectiles.begin(); iter != world.projectiles.end(); ++iter) {
				if (iter->get_type() == WT_TORPEDO && iter->get_target() == the_group)
					++total;
			}
			return total;
		}
		break;

	case TT_GroupType:
		return sides[n_side].groups[n_group].get_type();
		break;

	case TT_IN_SMALL_RANGE: {
		int range = static_cast<int>(weapon_lookup[sides[my_side].groups[my_group].get_small_type()].range);
		int dist = this_group->find_distance_to(n_side, n_group);
		
		if (dist <= range)
			return 1;
		else
			return 0;
		}
		break;
	
	case TT_InBigRange: {
		int range = static_cast<int>(weapon_lookup[sides[my_side].groups[my_group].get_big_type()].range);
		int dist = this_group->find_distance_to(n_side, n_group);
		
		if (dist <= range)
			return 1;
		else
			return 0;
		}
		break;

	case TT_OurInSmallRange: {
		int range = static_cast<int>(weapon_lookup[sides[n_side].groups[n_group].get_small_type()].range);
		int dist = this_group->find_distance_to(n_side, n_group);
		
		if (dist <= range)
			return 1;
		else
			return 0;
		}
		break;
	
	case TT_OurInBigRange: {
		int range = static_cast<int>(weapon_lookup[sides[n_side].groups[n_group].get_big_type()].range);
		int dist = this_group->find_distance_to(n_side, n_group);
		
		if (dist <= range)
			return 1;
		else
			return 0;
		}
		break;

	case TT_IS_ALPHA:
		return static_cast<int>(sides[n_side].groups[n_group].is_alpha_wing());
		break;
	}
	
	l_iter = l_end;
	this_group->report_on_script_error("Failed to convert group stat to integer", current_line);
	return 0;
}

///

bool AIInterpreter::are_groups_the_same() {
	int side1 = 0;
	int side2 = 0;
	int group1 = 0;
	int group2 = 0;

	get_group_indices(side1, group1, true);

	unsigned char operator_type = *l_iter;
	++l_iter;

	get_group_indices(side2, group2, true);

	int lhs, rhs;

	if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
		lhs = (side1 >> 16) | group1;
		rhs = (side2 >> 16) | group2;
	} else {
		lhs = (side1 << 16) | group1;
		rhs = (side2 << 16) | group2;
	}

	return compare_values(lhs, operator_type, rhs);
}

void AIInterpreter::get_group_indices(int& side, int& group, bool ignore_invalid_save_groups) {
	switch(*l_iter) {
	case TT_Our:
		side = my_side;
		group = my_group;
		break;

	case TT_NearestEnemy:
		NearestEnemy(side, group);
		break;

	case TT_NearestFriend:
		side = my_side;
		NearestFriend(group);
		break;

	case TT_NearestAlly:
		NearestAlly(side, group);
		break;

	case TT_SaveGroup:
	case TT_GSaveGroup:
		which_saved_group(side, group, ignore_invalid_save_groups);
		break;

	case TT_NONE:
		side = -1;
		group = -1;
		break;
	}

	++l_iter;
}

//if x is -1 it means no group found
void AIInterpreter::find_nearest_with(CoordsInt& give_indices, int& give_dist) {
	//in part two all the groups start off as valid and are then eliminated,
	//here all sides start off as invalid and are then added
	vector<bool> possible_sides(sides.size(), false);

	switch (*l_iter) {
	case TT_Our:
		//leave one past end
		++l_iter;
		give_indices.x = my_side;
		give_indices.y = my_group;
		break;
		
	case TT_NearestEnemy:
		++l_iter;
		for (int i = 0; i != sides.size(); ++i) {
			if (sides[i].my_flag != sides[my_side].my_flag)
				possible_sides[i] = true;
		}
		find_nearest_with_part_two(give_indices, possible_sides, give_dist);
		break;

	case TT_NearestFriend:
		++l_iter;
		possible_sides[my_side] = true;
		find_nearest_with_part_two(give_indices, possible_sides, give_dist);
		break;

	case TT_NearestAlly:
		++l_iter;
		for (int i = 0; i != sides.size(); ++i) {
			if (sides[i].my_flag == sides[my_side].my_flag && i != my_side)
				possible_sides[i] = true;
		}
		find_nearest_with_part_two(give_indices, possible_sides, give_dist);
		break;

	case TT_SaveGroup: {
			++l_iter;
			int which = iter_to_int(l_iter, l_end);

			if (save_groups[which].x == -1) {
				l_iter = l_end;
				this_group->report_on_script_error("Use of invalidated save group", current_line);
				give_indices.x = -1;
				give_indices.y = -1;
				break;
			}
			
			if (save_groups[which].x == max_players)
				give_dist = DistToXEdge(static_cast<CompassDirection>(save_groups[which].y));
			else
				give_dist = this_group->find_distance_to(save_groups[which].x, save_groups[which].y);

			give_indices.x = save_groups[which].x;
			give_indices.y = save_groups[which].y;
		}
		break;

	case TT_GSaveGroup: {
			++l_iter;
			int which = iter_to_int(l_iter, l_end);

			if (sides[my_side].missions[this_group->my_mission].save_groups[which].x == -1) {
				l_iter = l_end;
				this_group->report_on_script_error("Use of invalidated save group", current_line);
				give_indices.x = -1;
				give_indices.y = -1;
				break;
			}
			
			if (sides[my_side].missions[this_group->my_mission].save_groups[which].x == max_players)
				give_dist = DistToXEdge(static_cast<CompassDirection>(sides[my_side].missions[this_group->my_mission].save_groups[which].y));
			else
				give_dist = this_group->find_distance_to(sides[my_side].missions[this_group->my_mission].save_groups[which].x, sides[my_side].missions[this_group->my_mission].save_groups[which].y);

			give_indices.x = sides[my_side].missions[this_group->my_mission].save_groups[which].x;
			give_indices.y = sides[my_side].missions[this_group->my_mission].save_groups[which].y;
		}
		break;

	case TT_NEAREST_WORLD_EDGE:
			//leave one past end
			++l_iter;

			give_indices.x = max_players;
			give_indices.y = NearestEdge(give_dist);
		break;

	case TT_NorthEdge:
	case TT_EastEdge:
	case TT_SouthEdge:
	case TT_WestEdge:
		give_indices.x = max_players;
		give_indices.y = TokenToEdge(give_dist);
		break;
	
	default:
		l_iter = l_end;
		this_group->report_on_script_error("Didn't recognise group type when reading AI script", current_line);
		give_indices.x = -1;
		give_indices.y = -1;
		break;
	}
}

//if x is -1 it means no group found
void AIInterpreter::find_nearest_with_part_two(CoordsInt& give_indices, vector<bool>& possible_sides, int& give_dist) {
	give_indices.x = -1;
	give_indices.y = -1;
	give_dist = max_ai_int;

	vector<vector <bool> > possible_groups(possible_sides.size());

	search_through_groups(possible_sides, possible_groups);

	for (int i = 0; i != possible_sides.size(); ++i) {
		for (int j = 0; j != possible_groups[i].size(); ++j) {
			if (possible_groups[i][j] == true) {
				int tmp_dist = this_group->find_distance_to(i, j);

				if (tmp_dist < give_dist) {
					give_dist = tmp_dist;
					give_indices.x = i;
					give_indices.y = j;
				}
			}
		}
	}
}

void AIInterpreter::search_through_groups(vector<bool>& possible_sides, vector<vector <bool> >& possible_groups) {
	for (int i = 0; i != possible_sides.size(); ++i) {
		if (possible_sides[i] == false)
			continue;

		//all start off as being possible, the opposite to the sides vector
		possible_groups[i].resize(sides[i].groups.size(), true);
	}
	
	ustring::const_iterator saveIter = l_iter;
	
	for (int i = 0; i != possible_sides.size(); ++i) {
		if (possible_sides[i] == false)
			continue;
		
		for (int j = 0; j != possible_groups[i].size(); ++j) {
			l_iter = saveIter;
			possible_groups[i][j] = is_group_like_this(i, j, true, true);
		}
	}
}

///

bool AIInterpreter::do_any_have() {
	//in shared search all the groups start off as valid and are then eliminated,
	//here all sides start off as invalid and are then added
	vector<bool> possible_sides(sides.size(), false);

	which_sides_valid(possible_sides);
	
	++l_iter;
	
	ustring::const_iterator saveIter = l_iter;
	
	for (int i = 0; i != possible_sides.size(); ++i) {
		if (possible_sides[i] == false)
			continue;
		
		for (int j = 0; j != sides[i].groups.size(); ++j) {
			l_iter = saveIter;
			if (is_group_like_this(i, j, true, true))
				return true;
		}
	}
	
	//got to end, must be false
	return false;
}

int AIInterpreter::how_many_have() {
	//in shared search all the groups start off as valid and are then eliminated,
	//here all sides start off as invalid and are then added
	vector<bool> possible_sides(sides.size(), false);

	which_sides_valid(possible_sides);
	
	++l_iter;
	
	ustring::const_iterator saveIter = l_iter;
	
	int total = 0;
	
	for (int i = 0; i != possible_sides.size(); ++i) {
		if (possible_sides[i] == false)
			continue;
		
		for (int j = 0; j != sides[i].groups.size(); ++j) {
			l_iter = saveIter;
			if (is_group_like_this(i, j, true, true))
				++total;
		}
	}
	
	return total;
}

void AIInterpreter::which_sides_valid(vector<bool>& possible_sides) {
	switch (*l_iter) {
	case TT_AnyEnemy:
	case TT_NumEnemy:
		for (int i = 0; i != sides.size(); ++i) {
			if (sides[i].my_flag != sides[my_side].my_flag)
				possible_sides[i] = true;
		}
		break;

	case TT_AnyFriend:
	case TT_NumFriend:
		possible_sides[my_side] = true;
		break;

	case TT_AnyAlly:
	case TT_NumAlly:
		for (int i = 0; i != sides.size(); ++i) {
			if (sides[i].my_flag == sides[my_side].my_flag && i != my_side)
				possible_sides[i] = true;
		}
		break;
	}
}

bool AIInterpreter::is_nearest_like_this() {
	int n_side, n_group;
	bool discount_dead = true;
	bool discount_our = true;
	
	switch (*l_iter) {
	case TT_Our:
		n_side = my_side;
		n_group = my_group;
		discount_our = false;
		break;

	case TT_NearestEnemy:
		NearestEnemy(n_side, n_group);
		break;

	case TT_NearestFriend:
		n_side = my_side;
		NearestFriend(n_group);
		break;

	case TT_NearestAlly:
		NearestAlly(n_side, n_group);
		break;
		
	case TT_SaveGroup:
	case TT_GSaveGroup:
		which_saved_group(n_side, n_group);
		discount_dead = false;
		discount_our = false;

		if (n_side == max_players) {
			++l_iter;
			return is_edge_like_this(n_side, n_group);
		}
		break;

	default: {
			l_iter = l_end;
			this_group->report_on_script_error("is_group_like_this failed", current_line);
			return false;
		}
		break;
	}
	
	++l_iter;
	return is_group_like_this(n_side, n_group, discount_dead, discount_our);
}

bool AIInterpreter::is_group_like_this(int n_side, int n_group, bool discount_dead, bool discount_our) {
	//discount dead people and/or ourselves?
	if ((!sides[n_side].groups[n_group].get_alive() && discount_dead)
	|| !sides[my_side].scanned_groups[n_side][n_group]
	|| sides[n_side].groups[n_group].get_in_hangar()
	|| (n_side == my_side && n_group == my_group && discount_our)) {
		short_circuit_skip();
		return false;
	}

	while (l_iter != l_end) {
		//possibly we have reached the end of this if, if so must be true
		if (*l_iter == TT_AND || *l_iter == TT_OR)
			return true;
		
		bool boolean_result = false;
		if (*l_iter == TT_IN_SMALL_RANGE || *l_iter ==TT_InBigRange || *l_iter == TT_OurInSmallRange || *l_iter == TT_OurInBigRange
		|| *l_iter == TT_IS_ALPHA)
			boolean_result = true;
			
		int stat_value = token_to_int(n_side, n_group);
		
		if (boolean_result) {
			if (!stat_value) {
				short_circuit_skip();
				return false;
			}
		} else {
			const unsigned char operator_type = *l_iter;
			++l_iter;
			int second_value = token_to_int();

			if (!compare_values(stat_value, operator_type, second_value)) {
				short_circuit_skip();
				return false;
			}
		}
	}
	
	//if we reach the end it is true
	return true;
}

bool AIInterpreter::is_edge_like_this(int n_side, int n_group) {
	int stat_value = token_to_int(n_side, n_group);
	
	const unsigned char operator_type = *l_iter;
	++l_iter;
	int second_value = token_to_int();
	
	if (!compare_values(stat_value, operator_type, second_value))
		return false;
	else
		return true;
}

void AIInterpreter::short_circuit_skip() {
	while (l_iter != l_end && *l_iter != TT_AND && *l_iter != TT_OR)
		++l_iter;
}

void AIInterpreter::NearestEnemy(int& give_side, int& give_group) {
	//if no enemies left (unlikely) we'll use 0 just to prevent a crash
	give_side = 0;
	give_group = 0;
	
	int current_dist = max_ai_int;
	
	for (int i = 0; i != sides.size(); ++i) {
		if (sides[i].my_flag != sides[my_side].my_flag) {
			for (int j = 0; j != sides[i].groups.size(); ++j) {
				if (!sides[i].groups[j].get_alive()
					|| !sides[my_side].scanned_groups[i][j]
					|| sides[i].groups[j].get_in_hangar())
					continue;

				int tmp_dist = this_group->find_distance_to(i, j);

				if (tmp_dist < current_dist) {
					current_dist = tmp_dist;
					give_side = i;
					give_group = j;
				}
			}
		}
	}
}

void AIInterpreter::NearestFriend(int& give_group) {
	//if no friends left (possible) we'll use 0 just to prevent a crash
	give_group = 0;
	
	int current_dist = max_ai_int;
	
	//who is nearest?
	for (int j = 0; j != sides[my_side].groups.size(); ++j) {
		if (!sides[my_side].groups[j].get_alive()
			|| j == my_group
			|| sides[my_side].groups[j].get_in_hangar())
			continue;

		int tmp_dist = this_group->find_distance_to(my_side, j);

		if (tmp_dist < current_dist) {
			current_dist = tmp_dist;
			give_group = j;
		}
	}
}

void AIInterpreter::NearestAlly(int& give_side, int& give_group) {
	//if no allies left (possible) we'll use 0 just to prevent a crash
	give_side = 0;
	give_group = 0;
	
	int current_dist = max_ai_int;
	
	//who is nearest?
	for (int i = 0; i != sides.size(); ++i) {
		if (sides[i].my_flag == sides[my_side].my_flag && i != my_side) {
			for (int j = 0; j != sides[i].groups.size(); ++j) {
				if (!sides[i].groups[j].get_alive()
				|| sides[i].groups[j].get_in_hangar())
					continue;

				int tmp_dist = this_group->find_distance_to(i, j);

				if (tmp_dist < current_dist) {
					current_dist = tmp_dist;
					give_side = i;
					give_group = j;
				}
			}
		}
	}
}

//this doesn't leave token one past end to stay in line with nearest-type functions
void AIInterpreter::which_saved_group(int& give_side, int& give_group, bool ignore_invalid_save_groups) {
	const unsigned char groupType = *l_iter;
	++l_iter;
	int which = iter_to_int(l_iter, l_end);
	//this doesn't leave token one past end to stay in line with nearest-type functions
	--l_iter;
	
	switch(groupType) {
	case TT_SaveGroup:
		give_side = save_groups[which].x;
		give_group = save_groups[which].y;
		break;

	case TT_GSaveGroup:
		give_side = sides[my_side].missions[this_group->my_mission].save_groups[which].x;
		give_group = sides[my_side].missions[this_group->my_mission].save_groups[which].y;
		break;
	}

	if (!ignore_invalid_save_groups && give_side == -1) {
		this_group->report_on_script_error("Use of invalidated save group", current_line);
		//don't crash
		give_side = 0;
		give_group = 0;
	}
}

///

CompassDirection AIInterpreter::NearestEdge(int& distance) {
		CompassDirection ret = CD_N;
		distance = DistToTopEdge();
		int edist = DistToRightEdge();
		int sdist = DistToBottomEdge();
		int wdist = DistToLeftEdge();
		
		if (edist < distance) {
			distance = edist;
			ret = CD_E;
		}
		if (sdist < distance) {
			distance = sdist;
			ret = CD_S;
		}
		if (wdist < distance) {
			distance = wdist;
			ret = CD_W;
		}

		return ret;
}

int AIInterpreter::DistToXEdge(CompassDirection which) {
	switch(which) {
	case CD_N:
		return DistToTopEdge();
	case CD_E:
		return DistToRightEdge();
	case CD_S:
		return DistToBottomEdge();
	case CD_W:
		return DistToLeftEdge();
	//corners/8 compass points not supported as edges
	default:
		return max_ai_int;
		break;
	}
}

CompassDirection AIInterpreter::TokenToEdge(int& give_dist) {
	CompassDirection ret;

	switch(*l_iter) {
	case TT_NorthEdge:
		ret = CD_N;
		break;

	case TT_EastEdge:
		ret = CD_E;
		break;

	case TT_SouthEdge:
		ret = CD_S;
		break;

	case TT_WestEdge:
		ret = CD_W;
		break;
	}

	give_dist = DistToXEdge(ret);

	//one past end
	++l_iter;
	return ret;
}

int AIInterpreter::DistToLeftEdge() const {
	return static_cast<int>(this_group->myx);
}

int AIInterpreter::DistToRightEdge() const {
	return static_cast<int>(world.width - (this_group->myx + this_group->width));
}

int AIInterpreter::DistToTopEdge() const {
	return static_cast<int>(this_group->myy);
}

int AIInterpreter::DistToBottomEdge() const {
	return static_cast<int>(world.height - (this_group->myy + this_group->height));
}

int AIInterpreter::count_waypoints() const {
	int total = 0;
	for (int i = 0; i != n_ai_vars; ++i) {
		if (this_group->waypoints[i].x != -1)
			++total;
		else
			break;
	}

	return total;
}

