/*-
 * $Id: rr-listener.c,v 1.7 2003/01/03 16:01:05 jonas Exp $
 *
 * See the file LICENSE for redistribution information. 
 * If you have not received a copy of the license, please contact CodeFactory
 * by email at info@codefactory.se, or on the web at http://www.codefactory.se/
 * You may also write to: CodeFactory AB, SE-903 47, Ume, Sweden.
 *
 * Copyright (c) 2002 Jonas Borgstrm <jonas@codefactory.se>
 * Copyright (c) 2002 Daniel Lundin   <daniel@codefactory.se>
 * Copyright (c) 2002 CodeFactory AB.  All rights reserved.
 */

#include <librr/rr.h>

#define DEFAULT_MAX_CONNECTIONS 100

static GObjectClass *parent_class = NULL;

static gboolean shutdown (RRListener *listener, GError **error);
static void finalize (GObject *object);

static void
rr_listener_init (GObject *object)
{
	RRListener *listener = (RRListener *)object;

	g_static_rw_lock_init (&listener->connections_lock);
	g_static_rw_lock_init (&listener->dead_connections_lock);
	listener->connections = NULL;
	listener->dead_connections = NULL;

	listener->num_connections = 0;
	listener->max_connections = DEFAULT_MAX_CONNECTIONS;
}

static void
rr_listener_class_init (GObjectClass *klass)
{
	RRListenerClass *listener_class = (RRListenerClass *)klass;

	listener_class->shutdown = shutdown;

	klass->finalize = finalize;

	parent_class = g_type_class_peek_parent (klass);
}

GType 
rr_listener_get_type (void)
{
	static GType rr_type = 0;

	if (!rr_type) {
		static GTypeInfo type_info = {
			sizeof (RRListenerClass),
			NULL,
			NULL,
			(GClassInitFunc) rr_listener_class_init,
			NULL,
			NULL,
			sizeof (RRListener),
			16,
			(GInstanceInitFunc) rr_listener_init
		};
		rr_type = g_type_register_static (G_TYPE_OBJECT, "RRListener", 
						  &type_info, 
						  G_TYPE_FLAG_ABSTRACT);
	}
	return rr_type;
}

static void
flush_dead_connections (RRListener *listener)
{
	g_static_rw_lock_writer_lock (&listener->dead_connections_lock);
	g_slist_foreach (listener->dead_connections, (GFunc)g_object_unref, NULL);
	g_slist_free (listener->dead_connections);
	listener->dead_connections = NULL;
	g_static_rw_lock_writer_unlock (&listener->dead_connections_lock);
}

static void
finalize (GObject *object)
{
	RRListener *listener = (RRListener *)object;

	if (listener->profreg)
		g_object_unref (G_OBJECT (listener->profreg));

	g_static_rw_lock_writer_lock (&listener->connections_lock);
	g_static_rw_lock_free (&listener->connections_lock);
	flush_dead_connections (listener);
	g_static_rw_lock_free (&listener->dead_connections_lock);

	parent_class->finalize (object);
}

void
rr_listener_set_profile_registry (RRListener *listener, 
				  RRProfileRegistry *profreg)
{
	g_return_if_fail (RR_IS_LISTENER (listener));
	g_return_if_fail (RR_IS_PROFILE_REGISTRY (profreg));
	
	if (listener->profreg)
		g_object_unref (G_OBJECT (profreg));

	listener->profreg = g_object_ref (G_OBJECT (profreg));
}

void
rr_listener_add_connection (RRListener *listener,
			    RRConnection *connection)
{
	g_return_if_fail (RR_IS_LISTENER (listener));
	g_return_if_fail (RR_IS_CONNECTION (connection));

	flush_dead_connections (listener);

	rr_debug3 ("listener::adding connection %p", connection);

	connection->listener = listener;
	rr_connection_set_profile_registry (connection, listener->profreg);

	g_static_rw_lock_writer_lock (&listener->connections_lock);
	listener->num_connections++;
	listener->connections = g_slist_append (listener->connections, 
						g_object_ref (G_OBJECT (connection)));

	g_static_rw_lock_writer_unlock (&listener->connections_lock);
}

void
rr_listener_remove_connection (RRListener *listener,
			    RRConnection *connection)
{
	g_return_if_fail (RR_IS_LISTENER (listener));
	g_return_if_fail (RR_IS_CONNECTION (connection));

	rr_debug3 ("listener::removing connection %p\n", connection);

	g_static_rw_lock_writer_lock (&listener->connections_lock);

	listener->connections = g_slist_remove (listener->connections, 
						connection);
	g_static_rw_lock_writer_unlock (&listener->connections_lock);

	/* Add the connection to the soon to die list :) */
	g_static_rw_lock_writer_lock (&listener->dead_connections_lock);
	listener->num_connections--;
	listener->dead_connections = g_slist_append (listener->dead_connections, 
						     connection);
	g_static_rw_lock_writer_unlock (&listener->dead_connections_lock);
}

gboolean
rr_listener_shutdown (RRListener *listener, GError **error)
{
	return RR_LISTENER_GET_CLASS (listener)->shutdown (listener, error);
}

gint
rr_listener_get_num_connections (RRListener *listener)
{
	g_return_val_if_fail (RR_IS_LISTENER (listener), -1);

	return listener->num_connections;
}

/**
 * rr_listener_set_max_connections:
 * @listener: A #RRListener
 * @max: maximum number of connections.
 * 
 * use -1 to allow unlimited number of connections.
 **/
void
rr_listener_set_max_connections (RRListener *listener, int max)
{
	g_return_if_fail (RR_IS_LISTENER (listener));

	listener->max_connections = max;
}

gint
rr_listener_get_max_connections (RRListener *listener)
{
	g_return_val_if_fail (RR_IS_LISTENER (listener), -1);

	return listener->max_connections;
}

static gboolean
shutdown (RRListener *listener, GError **error)
{
	if (!rr_listener_disconnect_all (listener, error))
		return FALSE;
	g_object_unref (G_OBJECT (listener));
	return TRUE;
}

/**
 * rr_listener_disconnect_all:
 * @listener: A #RRListener
 * @error: location to store an error message
 * 
 * Tries to disconnect all open connections on this listener
 * 
 * Return value: %TRUE on success, %FALSE on failure.
 **/
gboolean
rr_listener_disconnect_all (RRListener *listener, GError **error)
{
	GSList *iter;
	g_return_val_if_fail (RR_IS_LISTENER (listener), FALSE);

	g_static_rw_lock_writer_lock (&listener->connections_lock);

	iter = listener->connections;
	while (iter) {

		RRConnection *conn = iter->data;
		conn->listener = NULL;
		if (!rr_connection_disconnect (conn, error)) {

			g_static_rw_lock_writer_unlock (&listener->connections_lock);
			return FALSE;
		}
		listener->connections = iter = g_slist_delete_link (iter, iter);
	}
	g_static_rw_lock_writer_unlock (&listener->connections_lock);

	return TRUE;
}
