/*-
 * $Id: rr-message.c,v 1.15 2002/12/12 14:59:53 jonas Exp $
 *
 * See the file LICENSE for redistribution information. 
 * If you have not received a copy of the license, please contact CodeFactory
 * by email at info@codefactory.se, or on the web at http://www.codefactory.se/
 * You may also write to: CodeFactory AB, SE-903 47, Ume, Sweden.
 *
 * Copyright (c) 2002 Jonas Borgstrm <jonas@codefactory.se>
 * Copyright (c) 2002 Daniel Lundin   <daniel@codefactory.se>
 * Copyright (c) 2002 CodeFactory AB.  All rights reserved.
 */

#include <librr/rr.h>
#include <string.h>

static GObjectClass *parent_class = NULL;

static void
finalize (GObject *object)
{
	RRMessage *message = (RRMessage *)object;

	rr_frame_aggregator_free (&message->frame_list);
	parent_class->finalize (object);
}


static void
rr_message_init (GObject *object)
{
	RRMessage *msg = (RRMessage *)object;
	msg->msgno = -1;
	msg->aggregate_frames = TRUE;
}


static void
rr_message_class_init (GObjectClass *klass)
{
	klass->finalize = finalize;

	parent_class = g_type_class_peek_parent (klass);
}


GType 
rr_message_get_type (void)
{
	static GType rr_type = 0;

	if (!rr_type) {
		static GTypeInfo type_info = {
			sizeof (RRMessageClass),
			NULL,
			NULL,
			(GClassInitFunc) rr_message_class_init,
			NULL,
			NULL,
			sizeof (RRMessage),
			16,
			(GInstanceInitFunc) rr_message_init
		};
		rr_type = g_type_register_static (G_TYPE_OBJECT, "RRMessage", 
						  &type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return rr_type;
}

RRFrame *
rr_message_get_frame (RRMessage *message, gsize max_size)
{
	g_return_val_if_fail (RR_MESSAGE_GET_CLASS (message)->get_frame, NULL);

	return RR_MESSAGE_GET_CLASS (message)->get_frame (message, max_size);
}


gboolean
rr_message_process_frame (RRMessage *message, RRFrame *frame, 
			  GError **error)
{
	g_return_val_if_fail (RR_MESSAGE_GET_CLASS (message)->process_frame,
			      FALSE);

	if (message->aggregate_frames) {
		RRFrame *new_frame;

		new_frame = rr_frame_aggregate (&message->frame_list, frame);

		if (new_frame) {
			gint ret;
			
			ret = RR_MESSAGE_GET_CLASS (message)->
				process_frame (message, frame, error);

			g_object_unref (G_OBJECT (new_frame));
			return ret;
		}
		else
			return TRUE;
	}
	return RR_MESSAGE_GET_CLASS (message)->process_frame (message, frame, 
							      error);
}

void
rr_message_set_channel (RRMessage *message, RRChannel *channel)
{
	g_return_if_fail (RR_IS_MESSAGE (message));
	g_return_if_fail (RR_IS_CHANNEL (channel));

	message->channel = channel;
}

/**
 * rr_message_set_aggregate:
 * @message: a #RRMessage
 * @aggregate: %TRUE or %FALSE.
 * 
 * Selects if the frames should be aggregated or not.
 **/
void
rr_message_set_aggregate (RRMessage *message, gboolean aggregate)
{
	g_return_if_fail (RR_IS_MESSAGE (message));

	message->aggregate_frames = aggregate;
}

/**
 * rr_message_get_aggregate:
 * @message: a #RRMessage
 * 
 * Returns %TRUE if the frames are aggregated or %FALSE if they are
 * passed to #rr_message_process_frame directly.
 * 
 * Return value: %TRUE or %FALSE. 
 **/
gboolean
rr_message_get_aggregate (RRMessage *message)
{
	g_return_val_if_fail (RR_IS_MESSAGE (message), FALSE);

	return message->aggregate_frames;
}
