/*
 *   retail - a logfile monitoring utility
 *
 *   Copyright (C) Y2K (2000) A.L.Lambert
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* Yee ole includes (I put this all in one file for my sanity) */
#include "include.h"

/* our 'print usage to user' function */
void proc_args_usage(char *myname) {
	fprintf(stderr, "retail v %s\n", VERSION);
	fprintf(stderr, "usage: %s /path/file1 /path/file2  [-s suffix] [-p prefix] [-h][-d]\n", myname);
}

/* a little more detailed help for users who use -h */
void proc_args_help(char *myname) {
	/* we want to print usage to user first */
	proc_args_usage(myname);
	/* describe in better detail on how to use this tool */
	fprintf(stderr, "\n");
	fprintf(stderr, "where /path/file(s) are files you want to tail for new data\n");
	fprintf(stderr, "-V = version information.\n");
	fprintf(stderr, "-h = this help screen.\n");
	fprintf(stderr, "-d = debug mode (constant flux; developer usage only)\n");
	fprintf(stderr, "\nfile.suffix is whrere retail stores info on input files.\n");
	fprintf(stderr, "prefix allows for placing stats files in alternate directories.\n");
	fprintf(stderr, "IE: if you want to store all your stats files in /tmp/retail/\n");
	fprintf(stderr, "you would use -p /tmp/retail\n");
}

/* function to error check val and bomb if it's NULL */
void proc_args_sanity_ecix(char *val) {
	if(val == NULL) {
		retval = RV_FATAL;
		perror("malloc");
		/* bomb out here */
		exit (retval);
	}
}

/* sanitize our args before we process input from user */
short int proc_args_sanity() {
	/* set the relevant values to something sane */
	suffix = NULL;
	prefix = NULL;
	debug = 0;
	
	/* malloc some space for suffix */
	suffix = malloc(strlen(DEF_SUFFIX) + 1);
	/* make sure we got what we expected */
	proc_args_sanity_ecix(suffix);
	/* and put the default value into the variable */
	strcpy(suffix, DEF_SUFFIX);
	/* we don't necessiarly have a default prefix, so we'll check to see
	 * if we do before we go through the malloc song and dance */
	if(DEF_PREFIX != NULL) {
		prefix = malloc(strlen(DEF_PREFIX) + 1);
		proc_args_sanity_ecix(prefix);
		strcpy(prefix, DEF_PREFIX);
	}
	/* ok, return with the retval whatever it may be */
	return (retval);
}

/* a wrapper to getopt just to keep it sane here */
int proc_args_getopt(int argc, char **argv) {
	int opt;
	long int i;
	
	/* set the args to something sane before we continue */
	proc_args_sanity();
	
	/* do the getopt() thing */
	while((opt = getopt(argc, argv, "s:p:dhV")) != -1) {
		switch(opt) {
			case 's':
				/* clear suffix from default and remalloc it */
				free(suffix);
				suffix = malloc(strlen(optarg) + 1);
				strcpy(suffix, optarg);
				break;
			case 'p':
				/* if prefix does exist, clear it and remalloc it */
				if(prefix != NULL) free(prefix);
				prefix = malloc(strlen(optarg) + 1);
				strcpy(prefix, optarg);
				break;
			case 'd':
				/* set us into debug mode */
				debug = 1;
				break;
			case 'h':
				/* print help to the screen */
				proc_args_help(argv[0]);
				exit (0);	
				break;
			case 'V':
				fprintf(stderr, "retail v %s\n", VERSION);
				exit (0);
				break;
			default:
				break;
		}
	}
	if(debug) fprintf(stderr, "suffix = %s\n", suffix);
	if(debug && prefix != NULL) fprintf(stderr, "prefix = %s\n", prefix);
	/* load up our files array */
	files = calloc(argc, sizeof(char *));
	if(files == NULL) { /* oopsie; calloc failed! */
		perror("calloc");
		exit(-1);
	}
	for(i = 0 ; argv[i + optind] != NULL ; ++i) {
		files[i] = malloc(strlen(argv[i + optind]) + 1);
		if(files[i] == NULL) { /* oopsie - malloc failed! */
			perror("malloc");
			exit (-1);
		}
		strcpy(files[i], argv[i + optind]);
		if(debug) fprintf(stderr, "files[%li]: %s\n", i, files[i]); 
	}
	/* and the last one == NULL to terminate the array */
	files[i] = NULL;
	
	/* return the current retval */
	return retval;
}

/* function that processes our input args */
int proc_args(int argc, char **argv) {
	/* if we got less than 2 args, we didn't get called right */
	if(argc < 2) {
		proc_args_usage(argv[0]);
		retval = RV_ARGS;
		return retval;
	}
	
	/* process our args and see what we got */	
	retval = proc_args_getopt(argc, argv);
	
	/* if we didn't get at least one file, we weren't called right */
	if(files[0] == NULL) {
		proc_args_usage(argv[0]);
		retval = RV_ARGS;
	}
	return retval;
}
