# Rekall Memory Forensics
# Copyright (c) 2010, 2011, 2012 Michael Ligh <michael.ligh@mnin.org>
# Copyright 2013 Google Inc. All Rights Reserved.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

# pylint: disable=protected-access


from rekall.plugins.windows import common


class WindowsPsxView(common.WinProcessFilter):
    "Find hidden processes with various process listings"

    __name = "psxview"

    def check_psscan(self, seen=None):
        """Enumerate processes with pool tag scanning"""
        _ = seen
        psscan = self.session.plugins.psscan()
        for _, physical_eprocess in psscan.scan_processes():
            yield self.virtual_process_from_physical_offset(physical_eprocess)

    def check_thrdproc(self, seen=None):
        """Enumerate processes indirectly by ETHREAD scanning"""
        _ = seen
        thrdscan_plugin = self.session.plugins.thrdscan()
        for ethread in thrdscan_plugin.generate_hits():
            if ethread.ExitTime != 0:
                continue

            # Bounce back to the threads owner
            process = ethread.Tcb.m('Process').dereference_as(
                '_EPROCESS', vm=self.kernel_address_space)

            if not process:
                process = ethread.m('ThreadsProcess').dereference(
                    vm=self.kernel_address_space)

            # Make sure the bounce succeeded
            if (process and process.ExitTime == 0 and
                    process.UniqueProcessId > 0 and
                    process.UniqueProcessId < 0xFFFF):

                yield process

    METHODS = common.WinProcessFilter.METHODS[:]
    METHODS.append(("PSScan", check_psscan))
    METHODS.append(("Thrdproc", check_thrdproc))

    def render(self, renderer):
        headers = [
            dict(type="_EPROCESS"),
            ]

        for method in self.methods:
            headers.append((method, method, "%s" % len(method)))

        renderer.table_header(headers)

        for eprocess in self.filter_processes():
            row = [eprocess]

            for method in self.methods:
                row.append(eprocess.obj_offset in self.cache[method])

            renderer.table_row(*row)
