/********************************************************************************
*                                                                               *
*                  Visualisation Toolkit adapter class                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#ifdef HAVE_VTK
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "exincs.h"
#include "FXVTKWindow.h"
using namespace FXEX;
namespace FXEX {

/*
 * Notes:
 * 1. we can currently save the vtk window to bmp/ppm which is probably a bad way to
 *    do it since fox provides good image saving functionality.  Need to change it so
 *    that the routine pulls the VTK image into local memory, then call the FOX image
 *    save routines.
 * 2. We may need to handle SEL_UPDATE,0 to handle and X/GDI update events...
 * 3. Why are the getDefaultWidth/Height accounting for 300 pixels?
 */

// Maps
FXDEFMAP(FXVTKWindow) FXVTKWindowMap[]={
  FXMAPFUNC(SEL_PAINT,0,FXVTKWindow::onPaint),
 // FXMAPFUNC(SEL_UPDATE,0,FXVTKWindow::onPaint),
  FXMAPFUNC(SEL_LEAVE,0,FXVTKWindow::onLeave),
  FXMAPFUNC(SEL_LEFTBUTTONPRESS,0,FXVTKWindow::onLeftBtnPress),
  FXMAPFUNC(SEL_LEFTBUTTONRELEASE,0,FXVTKWindow::onLeftBtnRelease),
  FXMAPFUNC(SEL_MIDDLEBUTTONPRESS,0,FXVTKWindow::onMiddleBtnPress),
  FXMAPFUNC(SEL_MIDDLEBUTTONRELEASE,0,FXVTKWindow::onMiddleBtnRelease),
  FXMAPFUNC(SEL_RIGHTBUTTONPRESS,0,FXVTKWindow::onRightBtnPress),
  FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,0,FXVTKWindow::onRightBtnRelease),
  FXMAPFUNC(SEL_TIMEOUT,FXVTKWindow::ID_ANIMATE,FXVTKWindow::onAnimate),
  };
FXIMPLEMENT(FXVTKWindow,FXFrame,FXVTKWindowMap,ARRAYNUMBER(FXVTKWindowMap))    

// ctor
FXVTKWindow::FXVTKWindow(FXComposite* p,FXuint opts,FXint x,FXint y,FXint w,FXint h,FXint pl,FXint pr,FXint pt,FXint pb): FXFrame(p,opts,x,y,w,h,pl,pr,pt,pb) {
  state=IS_IDLE;
  updateRate=0;
  renderer=vtkRenderer::New();
  renderWindow = vtkRenderWindow::New();
  renderWindow->AddRenderer(renderer);
  }

// dtor
FXVTKWindow::~FXVTKWindow() {
  if (state!=IS_IDLE) stopAnimate();
  if (renderer) {
    if (renderWindow) renderWindow->RemoveRenderer(render);
    renderer->Delete();
    }
  renderer=(vtkRenderer*)-1;
  if (renderWindow) renderWindow->Delete();
  renderWindow=(vtkRenderWindow*)-1;
  }

// Create window
void FXVTKWindow::create() { 
  FXFrame::create();
  renderWindow->SetDisplayId((void*)getApp()->getDisplay());
  renderWindow->SetParentId((void*)getParent()->id());
  renderWindow->SetWindowId((void*)id());
  renderWindow->Start();
  }

// save to stream
void FXVTKWindow::save(FXStream& store) const {
  FXFrame::save(store);
  store << state;
  store << updateRate;
  fxerror("VTK window is not saved to stream\n");
  }

// load from stream
void FXVTKWindow::load(FXStream& store) {
  FXFrame::load(store);
  store >> state;
  store >> updateRate;
  fxerror("VTK window is not restored from stream\n");
  }

// Get minimum width - FIXME why 300?
FXint FXVTKWindow::getDefaultWidth() {
  return 300+padleft+padright+border;  
  }

// Get minimum height - FIXME why 300?
FXint FXVTKWindow::getDefaultHeight() {
  return 300+padtop+padbottom+border;  
  }

// get the center of the renderer
FXdouble FXVTKWindow::getCenter(FXint n){
  return (FXdouble) renderer->GetCenter()[n];
  }

// tell the VTK render engine to render the window
void FXVTKWindow::render(){
  renderWindow->Render();
  }

// get the renderer size
FXint FXVTKWindow::getSize(FXint n){
  return renderer->GetSize()[n];
  }

// start animation
void FXVTKWindow::animate(){
  getApp()->removeTimeout(this,ID_ANIMATE);
  getApp()->removeChore(this,ID_ANIMATE);
  if (updateRate <= 0) getApp()->addChore(this,ID_ANIMATE);
  else getApp()->addTimeout(this,ID_ANIMATE,updateRate);
  }

// stop animation
void FXVTKWindow::stopAnimate() {
  getApp()->removeTimeout(this,ID_ANIMATE);
  getApp()->removeChore(this,ID_ANIMATE);
  state=IS_IDLE;
  //FIXME render();
  }

// helper to ensure that the image light source, follows the camera
void FXVTKWindow::CheckLightFollowCamera() {
  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  vtkLightCollection* lc = renderer->GetLights();
  lc->InitTraversal();
  vtkLight* CurrentLight = lc->GetNextItem();
  CurrentLight->SetPosition(CurrentCamera->GetPosition());
  CurrentLight->SetFocalPoint(CurrentCamera->GetFocalPoint());
  }

// implement dolly functionality - to dolly in y only
void FXVTKWindow::Dolly(FXint x,FXint y) {
  FXdouble y1 = (FXdouble) (getSize(1) - y);
  FXdouble c1 = getCenter(1);
  FXdouble dyf = 0.5 * (y1 - c1) / c1;
  FXdouble zoomFactor = pow((FXdouble)1.1, dyf);
  if (zoomFactor < 0.5) zoomFactor=0.5;
  if (zoomFactor > 1.5) zoomFactor=1.5;
  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  if (CurrentCamera->GetParallelProjection()) {
    CurrentCamera->SetParallelScale(CurrentCamera->GetParallelScale()/zoomFactor);
    }
  else {
    CurrentCamera->Dolly(zoomFactor);
    renderer->ResetCameraClippingRange();
    }
  CheckLightFollowCamera();
  }

// implement Pan functionality - to pan to x,y
void FXVTKWindow::Pan(FXint x,FXint y){
  float x1 = (float) x;
  float y1 = (float) (getSize(1) - y);
  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  double ViewFocus[4];
  CurrentCamera->GetFocalPoint(ViewFocus);
  renderer->SetWorldPoint(ViewFocus[0],ViewFocus[1],ViewFocus[2],1.0);
  renderer->WorldToDisplay();
  renderer->GetDisplayPoint(ViewFocus);
  double focalDepth = ViewFocus[2];
  renderer->SetDisplayPoint(x1,y1,focalDepth);
  renderer->DisplayToWorld();

  double NewPickPoint[4];
  renderer->GetWorldPoint(NewPickPoint);
  if (NewPickPoint[3]) {
    NewPickPoint[0] /= NewPickPoint[3];
    NewPickPoint[1] /= NewPickPoint[3];
    NewPickPoint[2] /= NewPickPoint[3];
    NewPickPoint[3] = 1.0;
    }
  CurrentCamera->GetFocalPoint(ViewFocus);
  double MotionVector[3];
  MotionVector[0] =  0.1 * (ViewFocus[0] - NewPickPoint[0]) + ViewFocus[0];
  MotionVector[1] =  0.1 * (ViewFocus[1] - NewPickPoint[1]) + ViewFocus[1];
  MotionVector[2] =  0.1 * (ViewFocus[2] - NewPickPoint[2]) + ViewFocus[2];
  CurrentCamera->SetFocalPoint(MotionVector[0],MotionVector[1],MotionVector[2]);
  CurrentCamera->SetPosition(MotionVector[0],MotionVector[1],MotionVector[2]);
  CheckLightFollowCamera();
  }

// implement rotate functionality - to rotate to x,y
void FXVTKWindow::Rotate(FXint x,FXint y) {
  int* size = renderer->GetSize();
  float* vp = renderer->GetViewport();
  double DeltaElevation = (double) (-20.0/((vp[3] - vp[1])*size[1]));
  double DeltaAzimuth   = (double) (-20.0/((vp[2] - vp[0])*size[0]));
  double x1 = (double) x;
  double y1 = (double) (size[1] - y);
  double c0 = (double) renderer->GetCenter()[0];
  double c1 = (double) renderer->GetCenter()[1];
  double rxf = (x1 - c0) * DeltaAzimuth;
  double ryf = (y1 - c1) * DeltaElevation;

  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  CurrentCamera->Azimuth(rxf);
  CurrentCamera->Elevation(ryf);
  CurrentCamera->OrthogonalizeViewUp();
  renderer->ResetCameraClippingRange();
  CheckLightFollowCamera();
  }

// implement rotate functionality for elevation only
void FXVTKWindow::Rotate(double ryf) {
  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  CurrentCamera->Elevation(ryf);
  CurrentCamera->OrthogonalizeViewUp();
  renderer->ResetCameraClippingRange();
  CheckLightFollowCamera();
  }

// implement spin in y only
void FXVTKWindow::Spin(int, int y) {
  double y1 = (double) (getSize(1) - y);
  double c1 = getCenter(1);
  double yf = (y1 - c1)/ (c1);
  if (yf > 1)  yf=1;
  else if (yf < -1) yf=-1;
  double newAngle = asin(yf) * 180.0 / PI;
  vtkCamera* CurrentCamera = renderer->GetActiveCamera();
  CurrentCamera->Roll(newAngle);
  CurrentCamera->OrthogonalizeViewUp();
  }

// just call our render whenever we get a paint event
// FIXME we should really pass on the rectanle that actually needs repainting...
long FXVTKWindow::onPaint(FXObject*,FXSelector,void*){
  render();
  return 1;
  }

// handle update event - for animation
long FXVTKWindow::onAnimate(FXObject*,FXSelector,void *ptr) {  
  FXEvent *ev = (FXEvent*)ptr;
  if (state & IS_ROTATE) Rotate(ev->win_x,ev->win_y); 
  if (state & IS_PAN) Pan(ev->win_x,ev->win_y); 
  if (state & IS_DOLLY) Dolly(ev->win_x,ev->win_y); 
  if (state & IS_SPIN) Spin(ev->win_x,ev->win_y); 
  if (!state & IS_IDLE) {
    animate();
    render();
    }
  return 1;
  }

// stops pan/rotate animation
long FXVTKWindow::onLeftBtnRelease(FXObject*,FXSelector,void*) {  
  stopAnimate();
  return 1;
  }

// left button press, either pan or rotate, depending of shifht key state
long FXVTKWindow::onLeftBtnPress(FXObject*,FXSelector,void *ptr) {  
  FXEvent *ev = (FXEvent*)ptr;
  if(ev->state&SHIFTMASK) state=IS_PAN;
  else state=IS_ROTATE;  
  animate();
  return 1;
  }

// stop pan animation
long FXVTKWindow::onMiddleBtnRelease(FXObject*,FXSelector,void*) {  
  stopAnimate();
  return 1;
  }

// pan on middle button press
long FXVTKWindow::onMiddleBtnPress(FXObject*,FXSelector,void*) {  
  state=IS_PAN;  
  animate();
  return 1;
  }

// stop dolly animation
long FXVTKWindow::onRightBtnRelease(FXObject*,FXSelector,void*) {  
  stopAnimate();
  return 1;
  }

// dolly on right button press
long FXVTKWindow::onRightBtnPress(FXObject*,FXSelector,void*) {  
  state=IS_DOLLY;  
  animate();
  return 1;
  }

// mouse move out of window - stop any animation
long FXVTKWindow::onLeave(FXObject*,FXSelector,void*) {
  stopAnimate();
  return 1;
  }

// set VTK window to be double buffered
void FXVTKWindow::doubleBuffer(FXbool b){
  renderWindow->SetDoubleBuffer(b);
  }

// returns a pointer to the VTK renderer
vtkRenderer* FXVTKWindow::getRenderer(){
  return renderer;
  }

// save VTK image as a PPM image file
void FXVTKWindow::SaveImageAsPPM(const FXString& file){
  renderWindow->SetFileName(file.text());
  renderWindow->SaveImageAsPPM();
  }

// save VTK image as BMP image file
void FXVTKWindow::SaveImageAsBMP(const FXString& file){
  vtkWindowToImageFilter* WinImage = vtkWindowToImageFilter::New();
  WinImage->SetInput(renderWindow); 
  vtkBMPWriter* BitmapWriter = vtkBMPWriter::New();
  BitmapWriter->SetInput(WinImage->GetOutput());
  BitmapWriter->SetFileName(file.text());
  BitmapWriter->Write();
  BitmapWriter->Delete();
  WinImage->Delete();
  }

}
#endif
