/********************************************************************************
*                                                                               *
*                  Utility functions                                            *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
********************************************************************************/
#ifndef FXEXUTILS_H
#define FXEXUTILS_H
namespace FXEX {

/**
 * The FXUtils namespace encapsulates some useful functions, into a single collection.
 */
namespace FXUtils {

/**
 * Save the machine dependant endianness value to the stream
 */
extern FXAPI void saveEndianness(FXStream& store);

/**
 * Load the machine dependant endianness value from the stream,
 * and setup stream to swap the bytes if the stream byte order
 * is non-native to the CPU architecture
 */
extern FXAPI void loadEndianness(FXStream& store);

/**
 * Create a random seed for use with fxrandom.
 * It is based on the current time, so its a _reasonably_ random seed
 */
extern FXAPI FXuint fxrandomseed();

/**
 * A useful function calls for generating random numbers
 * generate an integer/float number between [lowerBound,upperBound).
 */
extern FXAPI FXint fxrand(FXint upperBound,FXint lowerBound=0);
/// Returns a float rather than an int.
extern FXAPI FXfloat fxrand(FXfloat upperBound,FXfloat lowerBound=0.0);

/**
 * Set the font of all children - this is recursive
 */
extern FXAPI void setFontOfChildren(FXComposite* w,FXFont* f);

/**
 * Enable every widget and its children
 */
extern FXAPI void enableWindow(FXWindow* w);

/**
 * Disable every widget and its children
 */
extern FXAPI void disableWindow(FXWindow* w);

/// set the flags appropriately for the platform
extern FXAPI FXuint fxfilemode(FXFilePermission permissions);

/// load an image from a file
extern FXAPI FXbool loadFileImage(const FXString& file,FXImage *img);

/// load an icon from a file
extern FXAPI FXbool loadFileIcon(const FXString& file,FXIcon *ico);

/**
 * Load an image from a file; return an FXImage.
 * ... dont forget to delete the image, when finished with...
 */
extern FXAPI FXImage* loadImage(const FXString& file);

/**
 * Load an icon from a file; return an FXIcon
 * ... dont forget to delete the icon, when finished with...
 */
extern FXAPI FXIcon* loadIcon(const FXString& file);

/**
 * For anyone that uses images/icons with transparency here is a pretty
 * good function to have.  A function I wrote called blendIcon is below.
 * Basically it takes and icon and applies the transparency blend for you
 * and sets the color back to the icon.  When using icons/images with
 * gradient transparencies (such as those used for drop shadows) on menus
 * and buttons they will look perfect.  It should be somewhat optimized so
 * that no floating point math is needed.  Here is a simple example of how
 * to use it:
 *
 * FXPNGIcon *icon = new FXPNGIcon(getApp(), my_icon);
 * blendIcon(icon, getApp()->getBaseColor());
 * FXButton *button = new FXButton(this, "Click", icon);
 */
extern FXAPI FXIcon* blendIcon(FXIcon *icon, FXColor baseColor);


}; // namespace FXUtils
}; // namespace FXEX

#endif // FXEXUTILS_H
