/********************************************************************************
*                                                                               *
*                  Binary log file reader                                       *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.   All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#ifndef FXBINARYLOGREADER_H
#define FXBINARYLOGREADER_H

namespace FXEX {

/// Container data structure for a log entry
struct FXBinaryLogData {
  FXbool  subsecond;      // does the log contain valid subsecond resolution
  FXTime  seconds;        // current time, seconds resolution
  FXint   microseconds;   // current time, microseconds resolution
  FXshort code;           // status code of log entry
  FXshort value;          // status value of log entry
  };

/// Binary Log Reader options
enum {
  BINARYLOGREADER_TRUNCATE=0x02   // Truncate the file on opening ? FIXME...
  };

/**
 * A binary log file allows applications to log entries to a file, faster and more compact
 * than a normal log file.
 *
 * How this differs from a normal log file is that, up to 1000% of the speed of a log file
 * is consumed in the printf style parsing of the input string.  By using numbers only,
 * no parsing is necessary.  As an added bonus, you dont log many bytes per log entry so
 * the disk I/O time is reduced.
 *
 * The upshot is that a binary logger can run many times quicker than a text file logger,
 * and that the storage of that information is more compact.  The downside is that you need
 * a custom program to read the file (see FXBinaryLogReader).
 *
 * File format:
 * - file header (2 bytes)
 *   - 1 byte version code
 *   - 1 byte options value
 * - written per log entry (8-12 bytes)
 *   - 32bit date stamp
 *     - 1 second resolution
 *   - 32bit subsecond date stamp
 *     - microsecond resolution
 *     - optional
 *   - 32bit log entry
 *     - upper 16bits are used for indicating error severity
 *     - lower 16bits are used for indicating enumerated value
 *
 * Notes:
 * 1. The log file stores the endian-ness of the CPU architecture so that the log file
 *    can later be read back on any type of CPU.
 * 2. Log entries are automatically written on file open/close (as debug entries)
 *
 * It is envisaged that you could use this logger in an embedded application.  You would
 * log the values you need, when you identify a condition that needs to be logged.  You 
 * should create a specific enumerated value, for all possible log cases.
 */
class FXAPI FXBinaryLogReader {
private:
  FXFileIO              *fileio;    // file IO object
  FXBinaryLogData        logentry;  // holding variable for a log entry
  FXuchar                options;

protected:
  /// opens the log file for writing
  FXbool open();

  /// closes the log file
  void close();

public:
  /// Give me a log file
  FXBinaryLogReader(const FXString& file="",FXuint opts=0);

  /// Change the location of the log file - change is immediate
  FXbool name(const FXString& file);

  /// get the current filename
  FXString name();

  /// indicates whether the log file can/will be read from
  FXbool opened();

  /// read log entries from the file
  FXBinaryLogData* read();

  /// Save to stream
  friend FXAPI FXStream& operator<<(FXStream& store,const FXBinaryLogReader& b);

  /// load from stream
  friend FXAPI FXStream& operator>>(FXStream& store,FXBinaryLogReader& b);

  /// done
  virtual ~FXBinaryLogReader();
  };

} // namespace FXEX
#endif // FXBINARYLOGREADER_H
